#include "map.h"
#include <malloc.h>
#include <stdio.h>

const int TILESIZE = 30;

Map* map_create( long length )
{
  Map* map;
  int i;

  map = (Map*)malloc(sizeof(Map));
  map->offset = 0;
  map->fine_offset = 0;
  map->length = length;
  map->landscape = (unsigned char*)malloc(map->length*1);

  for (i=0; i<map->length; ++i)
  {
    map->landscape[i] = 0;
  }
  for (i=0; i<MAX_POWERUPS; ++i)
  {
    map->powerups[i].type = -1;
    map->powerups[i].x = -1;
    map->powerups[i].y = -1;
    map->powerups[i].gone = -1;
  }
  return map;
}

void map_destroy( Map** map )
{
  free((*map)->landscape);
  free(*map);
  (*map) = NULL;
}

Map* map_load( const char* filename )
{
  Map* map;
  map = (Map*)malloc(sizeof(Map));
  map->offset = 0;
  map->fine_offset = 0;

  FILE* f = fopen(filename,"rb");
  if (!f)
  {
    fprintf(stderr,"ERROR: Couldn't open '%s'\n",filename);
    free(map);
    return NULL;
  }

  fread(&map->length,4,1,f);
  map->landscape = (unsigned char*)malloc(map->length*1);
  fread(map->landscape,1,map->length,f);
  fread(map->powerups,sizeof(Powerup),MAX_POWERUPS,f);

  fclose(f);

  return map;
}
void map_save( Map* m, const char* filename )
{
  FILE* f;
  f = fopen(filename,"wb");
  if (!f)
  {
    fprintf(stderr,"ERROR: Couldn't write to '%s'\n",filename);
    return;
  }

  fwrite(&m->length,4,1,f);
  fwrite(m->landscape,1,m->length,f);
  fwrite(m->powerups,sizeof(Powerup),MAX_POWERUPS,f);

  fclose(f);
}

int _map_tile_solid( Map* m, int x, int y )
{
  unsigned char column = m->landscape[x];
  return (column & (1<<y));
}

int map_tile_solid( Map* m, int x, int y )
{
  if (y<0 || x<0) return 1;
  if (x>=m->length || y>7) return 0;
  return _map_tile_solid(m,x,y);
}

int map_powerup( Map* m, int x, int y )
{
  int i;
  for (i=0; i<MAX_POWERUPS; ++i)
  {
    if (m->powerups[i].x == x && m->powerups[i].y == y && !m->powerups[i].gone)
    {
      return i;
    }
  }
  return -1;
}

void map_eat_powerup( Map* m, int index )
{
  int i;
  if (index<0) return;
  m->powerups[MAX_POWERUPS-1].gone = -1;
  for (i=index; i<MAX_POWERUPS-1; ++i)
  {
    m->powerups[i] = m->powerups[i+1];
  }
  play_sample(data[SND_POWERUP].dat,100,127,1000,0);
}

int map_ground_number( Map* m, int x )
{
  int i;
  int ready;

  ready = 0;
  for (i=7; i>=0; --i)
  {
    if (!map_tile_solid(m,x,i))
    {
      if (ready)
      {
	return i;
      }
    }
    else
    {
      ready = 1;
    }
  }
  return -1;
}

// Tile coordinates relative to screen (0,0)
void _map_draw_tile( short number, int x, int y, int fine_offset )
{
  if (number >= 0)
  {
    masked_blit(data[SNOWTILES].dat,buffer,number*TILESIZE,0,x*TILESIZE-fine_offset,y*TILESIZE,TILESIZE,TILESIZE);
  }
}

void _map_draw_powerup( Powerup* p, long offset, int fine_offset )
{
  if (!p->gone)
  {
    masked_blit(data[POWERUPS].dat,buffer,p->type*TILESIZE,0,(p->x-offset)*TILESIZE-fine_offset,p->y*TILESIZE,TILESIZE,TILESIZE);
  }
}

void map_display( Map* m )
{
  // none,
  // left,right,up,down
  // lu,ru,ld,rd,lr,ud
  // lud,rud,lur,ldr
  // all

  int x, y;
  int number;
  int left, right, up, down; // temp vars indicating surrounding collision
  for (x = m->offset-1; x<=(m->offset + 320/TILESIZE + 1); ++x)
  {
    for (y=0; y<8; ++y)
    {
      if (!_map_tile_solid(m,x,y))
      {
	number = -1; // Don't draw
      }
      else
      {
	if (x<=0)
	{
	  left = 1;
	}
	else
	{
	  left = _map_tile_solid(m,x-1,y);
	}
	if (x>=m->length)
	{
	  right = 0;
	}
	else
	{
	  right = _map_tile_solid(m,x+1,y);
	}
	if (y<=0)
	{
	  up = 1;
	}
	else
	{
	  up = _map_tile_solid(m,x,y-1);
	}
	if (y>=7)
	{
	  down = 1;
	}
	else
	{
	  down = _map_tile_solid(m,x,y+1);
	}

	if (!left)
	{
	  if (!right)
	  {
	    if (!up)
	    {
	      if (!down)
	      { // all
		number = 15;
	      }
	      else
	      { // lur
		number = 13;
	      }
	    }
	    else
	    {
	      if (!down)
	      { // ldr
		number = 14;
	      }
	      else
	      { // lr
		number = 9;
	      }
	    }
	  }
	  else
	  {
	    if (!up)
	    {
	      if (!down)
	      { //lud
		number = 11;
	      }
	      else
	      { //lu
		number = 5;
	      }
	    }
	    else
	    {
	      if (!down)
	      { //ld
		number = 7;
	      }
	      else
	      { //l
		number = 1;
	      }
	    }
	  }
	}
	else
	{
	  if (!right)
	  {
	    if (!up)
	    {
	      if (!down)
	      { //rud
		number = 12;
	      }
	      else
	      { //ru
		number = 6;
	      }
	    }
	    else
	    {
	      if (!down)
	      { //rd
		number = 8;
	      }
	      else
	      { //r
		number = 2;
	      }
	    }
	  }
	  else
	  {
	    if (!up)
	    {
	      if (!down)
	      { //ud
		number = 10;
	      }
	      else
	      { //u
		number = 3;
	      }
	    }
	    else
	    {
	      if (!down)
	      { //d
		number = 4;
	      }
	      else
	      { //nothing
		number = 0;
	      }
	    }
	  }
	}
      }
      _map_draw_tile(number,x-m->offset,y,m->fine_offset);
    }
  }
  for (x=0; x<MAX_POWERUPS; ++x)
  {
    _map_draw_powerup(&m->powerups[x],m->offset,m->fine_offset);
  }
}
