/*  TA3D, a remake of Total Annihilation
    Copyright (C) 2005  Roland BROCHARD

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA*/

/*----------------------------------------------------------\
|                      EngineClass.h                        |
|    Contient toutes les classes ncessaires au moteur 3D   |
| et au moteur physique.                                    |
|                                                           |
\----------------------------------------------------------*/

#include "particles.h"			// Pour le moteur  particules
#include "tdf.h"				// Pour le gestionnaire de sprites

#ifndef ENGINE_CLASS		// Ne dclare pas 2 fois les classes du moteur
#define ENGINE_CLASS		// Dfinit les classes comme dclares

#define MAX_UNIT_PER_PLAYER		250		// 250 Units par joueur maximum

#define PARTICLE_LIMIT		100000		// pas plus de 100000 particules
#define HMAP_RESIZE			0.04f

#define H_DIV		0.5f
float const transform=1.0f/tan(63.44f*DEG2RAD)/H_DIV;
float const transform_H_DIV=1.0f/tan(63.44f*DEG2RAD);

extern float	player_color[30];

class BLOC				// Blocs composant la carte
{
public:
	int			nbindex;	// Nombre d'indices
	int			nbpoint;	// Nombre de points
	POINTF		*point;		// Points du bloc
//	GLuint		*index;		// Tableau d'indices de ces points
	float		*texcoord;	// Coordonnes de texture
	GLuint		tex;		// Indice de texture OpenGl
	bool		lava;		// Indique si le bloc est de type lave

	void init()
	{
		nbindex=nbpoint=0;
		point=NULL;
//		index=NULL;
		texcoord=NULL;
		tex=0;
		lava=false;
	}

	BLOC()
	{
		init();
	}

	void destroy()
	{
		if(point) delete(point);
//		if(index) delete(index);
		if(texcoord) delete(texcoord);
		init();
	}
};

class SECTOR			// Structure pour regrouper les informations sur le terrain (variations d'altitude, submerg, teneur en metal, ...)
{
public:
	float	dh;					// drive maximale en valeur absolue de l'altitude
	bool	underwater;			// indique si le bloc est sous l'eau
	byte	metal;				// Teneur en metal du sol
	byte	view;				// Indique quels sont les joueurs voyant ce morceau de terrain
	int		stuff;				// Indique l'lment graphique prsent sur ce secteur
	short	unit_idx;			// Indice de l'unit qui se trouve sur ce secteur
	byte	timer;				// Used to reset the fog of war

	inline void init()
	{
		dh=0.0f;
		underwater=false;
		metal=0;
		view=0;				// Personne ne peut voir ce secteur au dbut
		stuff=-1;
		unit_idx=-1;
		timer=0;
	}
};

#include "pathfinding.h"		// Algorithme de pathfinding

class MAP				// Donnes concernant la carte
{
public:
	int			ntex;		// Indique si la texture est charge et doit tre dtruite
	GLuint		*tex;		// Texture de surface
	int			nbbloc;		// Nombre de blocs
	BLOC		*bloc;		// Blocs composant le terrain
	unsigned short	**bmap;		// Tableau d'indice des blocs
	float		**h_map;	// Tableau de l'lvation du terrain
	float		**ph_map;	// Tableau du relief projet pour le calcul inverse(projection) lors de l'affichage
	int			**ph_map_2;	// Tableau du relief projet (multipli par un facteur flottant) pour le calcul inverse(projection) lors de l'affichage
	SECTOR		**map_data;	// Tableau d'informations sur le terrain
	byte		**view;		// Indique quels sont les parcelles de terrain visibles  l'cran
	byte		**path;		// Tableau pour le pathfinding

	int			map_w;		// Largeur de la carte en 16me de bloc
	int			map_h;		// Hauteur de la carte en 16me de bloc
	int			bloc_w;		// Largeur en blocs
	int			bloc_h;		// Hauteur en blocs
	BITMAP		*mini;		// Minimap
	GLuint		glmini;		// Texture OpenGl pour la minimap
	int			mini_w;
	int			mini_h;
	float		sealvl;		// Niveau de la mer
	POINTF		**lvl;		// Bloc de flottants pour le relief de la carte
	bool		water;		// Indique qu'il faut dessiner la mer
	bool		tnt;		// Indique si la carte est format tnt(format de total annihilation)
	float		sea_dec;	// Dcalage de la mer
	int			ox1,ox2;	// Coordonnes de la dernire fentre de terrain dessine
	int			oy1,oy2;
	GLuint		buf_i[1300];	// Pour acclrer l'affichage
	GLuint		lava_map;	// texture des zones de lave

/*---------------Experimental: pathfinding-----------------------------------------------*/

	byte		**global_path;	// Array to specify how is the map but at low level of detail to speed things up
	float		**global_hmap;	// Array to specify how is the map but at low level of detail to speed things up

/*---------------------------------------------------------------------------------------*/

	void init()
	{
		global_path=NULL;
		global_hmap=NULL;
		lava_map=0;
		path=NULL;
		mini_w=mini_h=252;
		ntex=0;
		tex=NULL;
		nbbloc=0;
		bloc=NULL;
		mini=NULL;
		bmap=NULL;
		h_map=NULL;
		ph_map=NULL;
		ph_map_2=NULL;
		map_data=NULL;
		sealvl=0.0f;
		glmini=0;
		lvl=NULL;
		water=true;
		tnt=false;			// Laisse la possibilit de crer un autre format de cartes
		sea_dec=0.0f;
		view=NULL;
		ox1=ox2=oy1=oy2=0;
		int buf_size=0;
		for(int i=0;i<1300;i++) {
			buf_i[i++]=0+buf_size;
			buf_i[i++]=1+buf_size;
			buf_i[i++]=3+buf_size;
			buf_i[i++]=4+buf_size;
			buf_i[i++]=6+buf_size;
			buf_i[i++]=7+buf_size;
			buf_i[i++]=7+buf_size;
			buf_i[i++]=8+buf_size;
			buf_i[i++]=4+buf_size;
			buf_i[i++]=5+buf_size;
			buf_i[i++]=1+buf_size;
			buf_i[i++]=2+buf_size;
			buf_i[i]=2+buf_size;
			buf_size+=9;
			}
	}

	MAP()
	{
		init();
	}

	void destroy()
	{
		if(global_path) {
			delete [] global_path[0];
			delete [] global_path;
			}
		if(global_hmap) {
			delete [] global_hmap[0];
			delete [] global_hmap;
			}
		if(lava_map)
			glDeleteTextures(1,&lava_map);
		if(path && bloc_w && bloc_h) {
			free(path[0]);
			free(path);
			}
		if(view && bloc_w && bloc_h) {
			free(view[0]);
			free(view);
			}
		if(map_data && bloc_w && bloc_h) {
			free(map_data[0]);
			free(map_data);
			}
		if(ph_map && bloc_w && bloc_h) {
			free(ph_map[0]);		// la carte est alloue d'un seul bloc
			free(ph_map);
			}
		if(ph_map_2 && bloc_w && bloc_h) {
			free(ph_map_2[0]);		// la carte est alloue d'un seul bloc
			free(ph_map_2);
			}
		if(h_map && bloc_w && bloc_h) {
			free(h_map[0]);		// la carte est alloue d'un seul bloc
			free(h_map);
			}
		if(bmap && bloc_w && bloc_h) {
			free(bmap[0]);		// la carte est alloue d'un seul bloc
			free(bmap);
			}
		if(ntex>0) {
			for(int i=0;i<ntex;i++)
				glDeleteTextures(1,&(tex[i]));
			free(tex);
			}
		if(lvl) {
			for(int i=0;i<bloc_h*bloc_w;i++)
				free(lvl[i]);
			free(lvl);
			}
		if(bloc && nbbloc>0) {
//			free(bloc[0].index);
			for(int i=0;i<nbbloc;i++) {
				bloc[i].point=NULL;
//				bloc[i].index=NULL;
				bloc[i].destroy();
				}
			free(bloc);
			}
		if(mini) {
			glDeleteTextures(1,&glmini);
			destroy_bitmap(mini);
			}
		init();
	}

	~MAP()
	{
		destroy();
	}

	void build_global_path();

	bool update_player_visibility(int player_id,int px,int py,int r);

/*	void build(BITMAP *hmap,BITMAP *bmptex=NULL,bool lmap=true)
	{
		destroy();
		if(lmap) {				// Cre la texture de luminosit
			BITMAP *lmap=create_bitmap_ex(8,hmap->w,hmap->h);
			VECTOR light;				// Vecteur lumire
			light.x=1.0f;	light.y=-1.0f;	light.z=1.0f;	light.Unit();
			VECTOR AB,AC,N;				// Vecteur pour le calcul des normales
			for(int y=0;y<hmap->h;y++)
				for(int x=0;x<hmap->w;x++) {
					AB.z=0.0f;
					AB.x=1.0f;
					if(x<hmap->w-1)
						AB.y=(hmap->line[y][x]-hmap->line[y][x+1])*HMAP_RESIZE;
					else
						AB.y=(hmap->line[y][x-1]-hmap->line[y][x])*HMAP_RESIZE;
					AC.z=1.0f;
					AC.x=0.0f;
					if(y<hmap->h-1)
						AC.y=(hmap->line[y][x]-hmap->line[y+1][x])*HMAP_RESIZE;
					else
						AC.y=(hmap->line[y-1][x]-hmap->line[y][x])*HMAP_RESIZE;
					N=AB*AC;
					N.Unit();
					int l=255.0f*(N%light);
					if(l<0) l=0;
					lmap->line[y][x]=l;
					}
			dltex=true;
			allegro_gl_set_texture_format(GL_LUMINANCE8);
			lightmap=allegro_gl_make_texture(lmap);
			glBindTexture(GL_TEXTURE_2D,lightmap);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
			destroy_bitmap(lmap);
			}
		allegro_gl_set_texture_format(-1);		// Format de texture par defaut
		if(bmptex) {			// Cre la texture de surface
			ntex=1;
			tex=(GLuint*) malloc(sizeof(GLuint));
			tex[0]=allegro_gl_make_texture(bmptex);
			glBindTexture(GL_TEXTURE_2D,tex[0]);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
			}
		int x,y,i;
		map_w=hmap->w;			// Dimensions de la carte
		map_h=hmap->h;
		bloc_w=(map_w-1>>4)+1;
		bloc_h=(map_h-1>>4)+1;
		nbbloc=bloc_w*bloc_h;
		bloc=(BLOC*) malloc(sizeof(BLOC)*nbbloc);
		bmap=(short**) malloc(sizeof(short*)*bloc_h);
		bmap[0]=(short*) malloc(sizeof(short)*bloc_w*bloc_h);
		for(i=1;i<bloc_h;i++)
			bmap[i]=&(bmap[0][bloc_w*i]);
		h_map=(byte**) malloc(sizeof(byte*)*bloc_h<<1);
		h_map[0]=(byte*) malloc(sizeof(byte)*bloc_w*bloc_h<<1);
		for(i=1;i<bloc_h;i++)
			h_map[i]=&(h_map[0][bloc_w*i<<1]);

		for(y=0;y<bloc_h;y++)			// Cre les blocs
			for(x=0;x<bloc_w;x++) {
				i=y*bloc_w+x;
				bmap[y<<1][x<<1]=i;
				h_map[(y<<1)][(x<<1)]=0;
				h_map[(y<<1)][(x<<1)+1]=0;
				h_map[(y<<1)+1][(x<<1)]=0;
				h_map[(y<<1)+1][(x<<1)+1]=0;
				bloc[i].init();
				if(ntex>0)
					bloc[i].tex=tex[0];
				int mx,my;
				mx=map_w-x*16;	my=map_h-y*16;
				if(mx>16) mx=16;
				if(my>16) my=16;
				bloc[i].nbpoint=(mx+1)*(my+1);
				bloc[i].nbindex=((mx+1)*2-1)*my+1;
				bloc[i].point=new POINTF[bloc[i].nbpoint];
				bloc[i].index=new GLuint[bloc[i].nbindex];
				if(bmptex)
					bloc[i].texcoord=new float[bloc[i].nbpoint<<1];
				int pos=0,X,Y;
				for(Y=0;Y<=my;Y++)										// Cre le tableau des coordonnes
					for(X=0;X<=mx;X++) {
						bloc[i].point[pos].x=X;
						bloc[i].point[pos].z=Y;
//						bloc[i].point[pos].x=x*16+X-0.5f*hmap->w;
//						bloc[i].point[pos].z=y*16+Y-0.5f*hmap->h;
						if(X+16*x<map_w && Y+16*y<map_h)
							bloc[i].point[pos].y=hmap->line[y*16+Y][x*16+X]*HMAP_RESIZE;	// Utilise la heightmap
						else
							bloc[i].point[pos].y=bloc[i].point[pos-1].y;			// Utilise le point d'avant
						bloc[i].texcoord[pos<<1]=(x*16.0f+X)/hmap->w;
						bloc[i].texcoord[(pos<<1)+1]=(y*16.0f+Y)/hmap->h;
						pos++;
						}
				pos=0;
				bloc[i].index[pos++]=0;									// Cre le tableau d'indices
				for(Y=0;Y<my;Y++)
					for(X=0;X<=mx;X++) {
						int MX=X;
						if(Y&1) MX=mx-X;
						if(X!=0)
							bloc[i].index[pos++]=Y*(mx+1)+MX;
						bloc[i].index[pos++]=(Y+1)*(mx+1)+MX;
						}
				}
	}*/

	void draw_mini(int x1=0,int y1=0,int w=252,int h=252,CAMERA *cam=NULL);			// Dessine la mini-carte

	inline float get_unit_h(float x,float y)
	{
		x=(x+0.5f*map_w)*0.125f;		// Convertit les coordonnes
		y=(y+0.5f*map_h)*0.125f;
		if(x<0.0f) x=0.0f;
		if(y<0.0f) y=0.0f;
		int lx=(bloc_w<<1)-1;
		int ly=(bloc_h<<1)-1;
		if(x>=lx) x=lx-1;
		if(y>=ly) y=ly-1;
		float h[4];
		int X=(int)x,Y=(int)y;
		float dx=x-X;
		float dy=y-Y;
		h[0]=h_map[Y][X];
		if(X+1<lx)
			h[1]=h_map[Y][X+1];
		else
			h[1]=h[0];
		if(Y+1<ly) {
			h[2]=h_map[Y+1][X];
			if(X+1<lx)
				h[3]=h_map[Y+1][X+1];
			else
				h[3]=h[2];
			}
		else {
			h[2]=h[0];
			h[3]=h[1];
			}
		h[0]=h[0]+(h[1]-h[0])*dx;
		h[2]=h[2]+(h[3]-h[2])*dx;
		return h[0]+(h[2]-h[0])*dy;
	}

	inline float get_h(int x,int y)
	{
		if(x<0) x=0;
		if(y<0) y=0;
		if(x>=(bloc_w<<1)-1) x=(bloc_w<<1)-2;
		if(y>=(bloc_h<<1)-1) y=(bloc_h<<1)-2;
		return h_map[y][x];
	}

	inline float get_zdec(int x,int y)
	{
		if(x<0) x=0;
		if(y<0) y=0;
		if(x>=(bloc_w<<1)-1) x=(bloc_w<<1)-2;
		if(y>=(bloc_h<<1)-1) y=(bloc_h<<1)-2;
		return ph_map[y][x]*transform_H_DIV;
	}

	inline int get_zdec_notest(int x,int y)
	{
		return ph_map_2[y][x];
	}

	inline float get_nh(int x,int y)
	{
		if(x<0) x=0;
		if(y<0) y=0;
		if(x>=(bloc_w<<1)-1) x=(bloc_w<<1)-2;
		if(y>=(bloc_h<<1)-1) y=(bloc_h<<1)-2;
		return ph_map[y][x];
	}

	inline void rect(int x1,int y1,int w,int h,short c,char *yardmap=NULL,bool open=false)
	{
		if(yardmap==NULL) {
			int y2=y1+h;
			int x2=x1+w;
			if(y1<0)	y1=0;
			if(y2>(bloc_h<<1)-1)	y2=(bloc_h<<1)-1;
			if(x1<0)	x1=0;
			if(x2>(bloc_w<<1)-1)	x2=(bloc_w<<1)-1;
			for(int y=y1;y<y2;y++)
				for(int x=x1;x<x2;x++)
					map_data[y][x].unit_idx=c;
			}
		else {
			int i=0;
			int y2=y1+h;
			int x2=x1+w;
			if(y1<0)	{	i-=y1*w;	y1=0;	}
			if(y2>(bloc_h<<1)-1)	y2=(bloc_h<<1)-1;
			if(x1<0)	x1=0;
			if(x2>(bloc_w<<1)-1)	x2=(bloc_w<<1)-1;
			int dw=w-(x2-x1);
			for(int y=y1;y<y2;y++) {
				for(int x=x1;x<x2;x++) {
					if(yardmap[i]=='G' || yardmap[i]=='o' || yardmap[i]=='w' || yardmap[i]=='f' || (yardmap[i]=='c' && !open) || (yardmap[i]=='C' && !open) || (yardmap[i]=='O' && open))
						map_data[y][x].unit_idx=c;
					i++;
					}
				i+=dw;
				}
/*				if(y>=0 && y<(bloc_h<<1)-1) {
					for(int x=x1;x<x2;x++) {
						if(x>=0 && x<(bloc_w<<1)-1 &&
							(yardmap[i]=='G' || yardmap[i]=='o' || yardmap[i]=='w' || yardmap[i]=='f' || (yardmap[i]=='c' && !open) || (yardmap[i]=='C' && !open) || (yardmap[i]=='O' && open)))
							map_data[y][x].unit_idx=c;
						i++;
						}
					}
				else
					i+=w;*/
			}
	}

	inline bool check_rect(int x1,int y1,int w,int h,short c)
	{
		int y2=y1+h;
		int x2=x1+w;
		if(y1<0)	y1=0;
		if(y2>(bloc_h<<1)-1)	y2=(bloc_h<<1)-1;
		if(x1<0)	x1=0;
		if(x2>(bloc_w<<1)-1)	x2=(bloc_w<<1)-1;
		for(int y=y1;y<y2;y++)
			for(int x=x1;x<x2;x++)
				if(map_data[y][x].unit_idx!=c && map_data[y][x].unit_idx!=-1)
					return false;
		return true;
	}

	inline float check_rect_dh(int x1,int y1,int w,int h)
	{
		int y2=y1+h;
		int x2=x1+w;
		if(y1<0)	y1=0;
		if(y2>(bloc_h<<1)-1)	y2=(bloc_h<<1)-1;
		if(x1<0)	x1=0;
		if(x2>(bloc_w<<1)-1)	x2=(bloc_w<<1)-1;
		float max_dh=0.0f;
		bool on_water=false;
		for(int y=y1;y<y2;y++)
			for(int x=x1;x<x2;x++) {
				if(map_data[y][x].dh>max_dh)
					max_dh=map_data[y][x].dh;
				on_water|=map_data[y][x].underwater;
				}
		if(on_water)
			max_dh=-max_dh;
		return max_dh;
	}

	inline float check_depth(int x1,int y1,int w,int h)
	{
		int y2=y1+h;
		int x2=x1+w;
		if(y1<0)	y1=0;
		if(y2>(bloc_h<<1)-1)	y2=(bloc_h<<1)-1;
		if(x1<0)	x1=0;
		if(x2>(bloc_w<<1)-1)	x2=(bloc_w<<1)-1;
		float depth=0.0f;
		for(int y=y1;y<y2;y++)
			for(int x=x1;x<x2;x++) {
				float d = sealvl-h_map[y][x];
				if(d>depth)
					depth=d;
				}
		return depth;
	}

	inline bool check_vents(int x1,int y1,int w,int h,char *yard_map)
	{
		if(yard_map==NULL)	return true;
		int y2=y1+h;
		int x2=x1+w;
		if(y1<0)	y1=0;
		if(y2>(bloc_h<<1)-1)	y2=(bloc_h<<1)-1;
		if(x1<0)	x1=0;
		if(x2>(bloc_w<<1)-1)	x2=(bloc_w<<1)-1;
		int dw = w - (x2-x1);
		int i = 0;
		bool ok = true;
		for(int y=y1;y<y2;y++) {
			for(int x=x1;x<x2;x++) {
				if(yard_map[i]=='G') {
					ok = false;
					if(map_data[y][x].stuff>=0) {
						int feature_id = map_data[y][x].stuff;
						if(feature_manager.feature[features.feature[feature_id].type].geothermal)
							return true;
						}
					}
				i++;
				}
			i+=dw;
			}
		return ok;
	}

	void draw(CAMERA *cam,byte player_mask,bool FLAT=false,float niv=0.0f,float t=0.0f,float dt=1.0f,bool depth_only=false,bool check_visibility=true);

	VECTOR hit(VECTOR Pos,VECTOR Dir,bool water=true);			// Calcule l'intersection d'un rayon avec la carte(le rayon partant du dessus de la carte)
};

class WATER
{
public:
	float		map_w;
	float		map_h;
	int			nb_vtx;
	int			nb_idx;
	VECTOR		*point;
	VECTOR		*N;
	float		*texcoord;
	GLuint		*index;
	int			s;
	float		w;

	void init()
	{
		map_w=0.0f;
		map_h=0.0f;
		nb_vtx=0;
		nb_idx=0;
		point=NULL;
		N=NULL;
		texcoord=NULL;
		index=NULL;
		s=0;
		w=1.0f;
	}

	void destroy()
	{
		if(point)		free(point);
		if(N)			free(N);
		if(texcoord)	free(texcoord);
		if(index)		free(index);
		init();
	}

	WATER()
	{
		init();
	}

	~WATER()
	{
		destroy();
	}

	void build(float m_w,float m_h,int d,float size)
	{
		destroy();
		map_w=m_w;
		map_h=m_h;
		s=d;
		w=size;

		nb_vtx=(d+1)*(d+1);
		nb_idx=d*d*6;

		point=(VECTOR*) malloc(sizeof(VECTOR)*nb_vtx);
		N=(VECTOR*) malloc(sizeof(VECTOR)*nb_vtx);
		texcoord=(float*) malloc(sizeof(float)*nb_vtx*2);
		index=(GLuint*) malloc(sizeof(GLuint)*nb_idx);

		int i=0;
		for(int y=0;y<=d;y++)
			for(int x=0;x<=d;x++) {
				point[i].x=(x-0.5f*d)*size/d;
				point[i].y=0.0f;
				point[i].z=(y-0.5f*d)*size/d;
				texcoord[i<<1]=(float)x/d;
				texcoord[(i<<1)+1]=(float)y/d;
				i++;
				}
		i=0;
		for(int y=0;y<d;y++)
			for(int x=0;x<d;x++) {
				index[i++]=y*(d+1)+x;
				index[i++]=y*(d+1)+x+1;
				index[i++]=(y+1)*(d+1)+x;
				index[i++]=(y+1)*(d+1)+x;
				index[i++]=y*(d+1)+x+1;
				index[i++]=(y+1)*(d+1)+x+1;
				}
		normalise();
	}

	void normalise()
	{
		for(int i=0;i<nb_vtx;i++)
			N[i].x=N[i].y=N[i].z=0.0f;
		for(int i=0;i<nb_idx;i+=3) {
			VECTOR n = (point[index[i+1]]-point[index[i]])*(point[index[i+2]]-point[index[i]]);
			n.Unit();
			N[index[i]]=N[index[i]]+n;
			N[index[i+1]]=N[index[i+1]]+n;
			N[index[i+2]]=N[index[i+2]]+n;
			}
		for(int i=0;i<nb_vtx;i++)
			N[i].Unit();
	}

	void draw(float t,float X,float Y,bool flat=false);
};

#include "ia.h"					// Pour l'intelligence artificielle

extern int NB_PLAYERS;

#define PLAYER_CONTROL_LOCAL_HUMAN	0x0
#define PLAYER_CONTROL_REMOTE_HUMAN	0x1
#define PLAYER_CONTROL_LOCAL_AI		0x2
#define PLAYER_CONTROL_REMOTE_AI	0x3
#define PLAYER_CONTROL_NONE			0x4

class PLAYERS			// Classe pour grer les joueurs et leurs statistiques de partie
{
public:
	int			nb_player;		// Nombre de joueurs (maximum 10 joueurs)
	int			local_human_id;	// Quel est le joueur qui commande depuis cette machine??
	byte		control[10];	// Qui controle ce joueur??
	char		*nom[10];		// Noms des joueurs
	char		*side[10];		// Camp des joueurs
	float		energy[10];		// Energie des joueurs
	float		metal[10];		// Metal des joueurs
	float		metal_u[10];	// Metal utilis
	float		energy_u[10];	// Energie utilise
	float		metal_t[10];	// Metal extrait
	float		energy_t[10];	// Energie produite
	int			kills[10];		// Victimes
	int			losses[10];		// Pertes
	int			energy_s[10];	// Capacits de stockage d'nergie
	int			metal_s[10];	// Capacits de stockage de metal
	int			com_metal;		// Stockage fournit par le commandeur
	int			com_energy;
	bool		commander[10];	// Indique s'il y a un commandeur
	bool		annihilated[10];// Le joueur a perdu la partie??
	AI_PLAYER	ai_command[10];	// Controleurs d'intelligence artificielle
	int			nb_unit[10];	// Nombre d'units de chaque joueur

	inline void player_control(MAP *map)
	{
		for(int i=0;i<nb_player;i++)
			if(control[i]==PLAYER_CONTROL_LOCAL_AI)
				ai_command[i].think(map);
	}

	inline void clear()		// Remet  0 la taille des stocks
	{
		for(int i=0;i<nb_player;i++) {
			energy_s[i]=metal_s[i]=0;			// Stocks
			metal_t[i]=energy_t[i]=0.0f;		// Production
			metal_u[i]=energy_u[i]=0.0f;		// Consommation
			commander[i]=false;
			annihilated[i]=true;
			nb_unit[i]=0;
			}
	}

	inline int add(char *NOM,char *SIDE,byte _control)
	{
		if(nb_player>=10)	return -1;		// Trop de joueurs dj
		metal_u[nb_player]=0;
		energy_u[nb_player]=0;
		metal_t[nb_player]=0;
		energy_t[nb_player]=0;
		kills[nb_player]=0;
		losses[nb_player]=0;
		nom[nb_player]=strdup(NOM);
		control[nb_player]=_control;
		nb_unit[nb_player]=0;
		side[nb_player++]=strdup(SIDE);
		if(_control==PLAYER_CONTROL_LOCAL_HUMAN)
			local_human_id=NB_PLAYERS;
		if(_control==PLAYER_CONTROL_LOCAL_AI) {
			char filename[100];
			filename[0]=0;
			strcat(filename,"ai/");
			strcat(filename,NOM);
			strcat(filename,".ai");
			if(file_exists(filename,FA_RDONLY | FA_ARCH,NULL))						// Charge un joueur s'il existe
				ai_command[NB_PLAYERS].load(filename,NB_PLAYERS);
			else													// Sinon cre un nouveau joueur
				ai_command[NB_PLAYERS].change_name(NOM);
			ai_command[NB_PLAYERS].player_id=NB_PLAYERS;
			}
		return NB_PLAYERS++;
	}

	inline void show_resources()
	{
		glEnable(GL_TEXTURE_2D);
		glEnable(GL_BLEND);
		glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_COLOR);
		allegro_gl_printf(aglfont,130.0f,8.0f,0.0f,0xFFFFFF,"METAL");
		allegro_gl_printf(aglfont,130+(SCREEN_W-130>>1),8.0f,0.0f,0xFFFFFF,"ENERGY");
		allegro_gl_printf(aglfont,65+(SCREEN_W-130>>1),8.0f,0.0f,0xFFFFFF,"%d",(int)metal[0]);
		allegro_gl_printf(aglfont,65+(SCREEN_W-130>>1),16.0f,0.0f,0xFFFFFF,"+%f",metal_t[0]);
		allegro_gl_printf(aglfont,65+(SCREEN_W-130>>1),24.0f,0.0f,0xFFFFFF,"-%f",metal_u[0]);
		allegro_gl_printf(aglfont,SCREEN_W-65,8.0f,0.0f,0xFFFFFF,"%d",(int)energy[0]);
		allegro_gl_printf(aglfont,SCREEN_W-65,16.0f,0.0f,0xFFFFFF,"+%f",energy_t[0]);
		allegro_gl_printf(aglfont,SCREEN_W-65,24.0f,0.0f,0xFFFFFF,"-%f",energy_u[0]);

		glDisable(GL_TEXTURE_2D);

		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
		glBegin(GL_QUADS);			// Dessine les barres de metal et d'nergie
			glColor4f(0.7f,0.7f,0.7f,0.5f);

			glVertex2f(180.0f,8.0f);					// Fond de la barre de metal
			glVertex2f(60+(SCREEN_W-130>>1),8.0f);
			glVertex2f(60+(SCREEN_W-130>>1),16.0f);
			glVertex2f(180.0f,16.0f);

			glVertex2f(186+(SCREEN_W-130>>1),8.0f);		// Fond de la barre d'nergie
			glVertex2f(SCREEN_W-70,8.0f);
			glVertex2f(SCREEN_W-70,16.0f);
			glVertex2f(186+(SCREEN_W-130>>1),16.0f);
		glEnd();
		glDisable(GL_BLEND);
		glBegin(GL_QUADS);			// Dessine les barres de metal et d'nergie
			glColor3f(1.0f,1.0f,0.0f);

			if(metal_s[0]) {
				glVertex2f(180.0f,8.0f);					// Barre de metal
				glVertex2f(180+((SCREEN_W-130>>1)-120)*metal[0]/metal_s[0],8.0f);
				glVertex2f(180+((SCREEN_W-130>>1)-120)*metal[0]/metal_s[0],16.0f);
				glVertex2f(180.0f,16.0f);
				}

			if(energy_s[0]) {
				glVertex2f(186+(SCREEN_W-130>>1),8.0f);		// Barre d'nergie
				glVertex2f(186+(SCREEN_W-130>>1)+(SCREEN_W-256-(SCREEN_W-130>>1))*energy[0]/energy_s[0],8.0f);
				glVertex2f(186+(SCREEN_W-130>>1)+(SCREEN_W-256-(SCREEN_W-130>>1))*energy[0]/energy_s[0],16.0f);
				glVertex2f(186+(SCREEN_W-130>>1),16.0f);
				}
		glEnd();
		glColor4f(1.0f,1.0f,1.0f,1.0f);
	}

	inline void init(int E=10000,int M=10000)		// Initialise les donnes des joueurs
	{
		com_metal=M;
		com_energy=E;
		nb_player=0;
		NB_PLAYERS=0;
		local_human_id=-1;
		for(int i=0;i<10;i++) {
			control[i]=PLAYER_CONTROL_NONE;
			ai_command[i].init();
			nom[i]=NULL;
			side[i]=NULL;
			energy[i]=E;
			metal[i]=M;
			metal_u[i]=0;
			energy_u[i]=0;
			metal_t[i]=0;
			energy_t[i]=0;
			kills[i]=0;
			losses[i]=0;
			energy_s[i]=E;
			metal_s[i]=M;
			nb_unit[i]=0;
			}
	}

	inline PLAYERS()
	{
		init();
	}

	inline void destroy()
	{
		for(int i=0;i<10;i++) {
			if(control[i]==PLAYER_CONTROL_LOCAL_AI)		// Enregistre les donnes de l'IA
				ai_command[i].save();
			if(nom[i])	free(nom[i]);
			if(side[i])	free(side[i]);
			ai_command[i].destroy();
			}
		init();
	}

	inline ~PLAYERS()
	{
		destroy();
	}
};

#include "fbi.h"
#include "weapons.h"

extern PLAYERS	players;		// Objet contenant les donnes sur les joueurs

void *create_unit(int type_id,int owner,VECTOR pos);

struct MISSION			// Structure pour stocker les ordres
{
	int			mission;
	PATH_NODE	*path;		// Chemin emprunt par l'unit si besoin pour la mission
	VECTOR		target;
	MISSION 	*next;		// Mission suivante
	bool		step;		// Etape d'une mission
	float		time;		// Temps coul depuis la cration de l'ordre
	int			data;		// Donnes de l'ordre
	byte		flags;		// Donnes supplmentaires
	float		last_d;		// Dernire distance enregistre
	void		*p;
};

class SCRIPT_ENV_STACK			// Pile pour la gestion des scripts
{
public:
	int					var[15];
	int					signal_mask;
	int					cur;
	SCRIPT_ENV_STACK	*next;

	inline void init()
	{
		for(int i=0;i<15;i++)
			var[i]=0;
		cur=0;
		signal_mask=0;
		next=NULL;
	}

	inline SCRIPT_ENV_STACK()
	{
		init();
	}
};

class SCRIPT_ENV			// Classe pour la gestion de l'environnement des scripts
{
public:
	SCRIPT_STACK		*stack;			// Pile utilise par les scripts
	SCRIPT_ENV_STACK	*env;			// Pile d'xecution des scripts
	float				wait;
	bool				running;

	inline void init()
	{
		stack=NULL;
		env=NULL;
		wait=0.0f;
		running=false;
	}

	SCRIPT_ENV()
	{
		init();
	}

	inline void destroy()
	{
		while(stack) {
			SCRIPT_STACK *next=stack->next;
			delete stack;
			stack=next;
			}
		while(env) {
			SCRIPT_ENV_STACK *next=env->next;
			delete env;
			env=next;
			}
		init();
	}

	inline ~SCRIPT_ENV()
	{
		destroy();
	}

	inline void push(int v)
	{
		SCRIPT_STACK *new_stack=new SCRIPT_STACK;
		new_stack->next=stack;
		stack=new_stack;
		stack->val=v;
	}

	inline int pop()
	{
		if(stack==NULL) {			// Si la pile est vide, renvoie 0 et un message pour le dbuggage
#ifdef DEBUG_MODE
			Console->AddEntry("interprteur de scripts: pile vide\n");
#endif
			return 0;
			}
		int v=stack->val;
		SCRIPT_STACK *old=stack;
		stack=stack->next;
		delete old;
		return v;
	}
};

#define SCRIPT_QueryPrimary		0x00
#define SCRIPT_AimPrimary		0x01
#define SCRIPT_FirePrimary		0x02
#define SCRIPT_QuerySecondary	0x03
#define SCRIPT_AimSecondary		0x04
#define SCRIPT_FireSecondary	0x05
#define SCRIPT_QueryTertiary	0x06
#define SCRIPT_AimTertiary		0x07
#define SCRIPT_FireTertiary		0x08
#define SCRIPT_TargetCleared	0x09
#define SCRIPT_stopbuilding		0x0A
#define SCRIPT_stop				0x0B
#define SCRIPT_startbuilding	0x0C
#define SCRIPT_go				0x0D
#define SCRIPT_killed			0x0E
#define SCRIPT_StopMoving		0x0F
#define SCRIPT_Deactivate		0x10
#define SCRIPT_Activate			0x11
#define SCRIPT_create			0x12
#define SCRIPT_MotionControl	0x13
#define SCRIPT_startmoving		0x14
#define SCRIPT_MoveRate1		0x15
#define SCRIPT_MoveRate2		0x16
#define SCRIPT_MoveRate3		0x17
#define SCRIPT_RequestState		0x18
#define SCRIPT_TransportPickup	0x19
#define SCRIPT_TransportDrop	0x1A
#define SCRIPT_QueryTransport	0x1B
#define SCRIPT_BeginTransport	0x1C
#define SCRIPT_EndTransport		0x1D

class UNIT				// Classe pour la gestion des units
{
public:
	SCRIPT			*script;		// Scripts concernant l'unit
	int				s_var[25];		// Tableau de variables pour les scripts
	MODEL			*model;			// Modle reprsentant l'objet
	byte			owner_id;		// Numro du propritaire de l'unit
	int				type_id;		// Type d'unit
	float			hp;				// Points de vide restant  l'unit
	VECTOR			Pos;			// Vecteur position
	VECTOR			V;				// Vitesse de l'unit
	VECTOR			Angle;			// Orientation dans l'espace
	VECTOR			V_Angle;		// Variation de l'orientation dans l'espace
	bool			sel;			// Unit slectionne?
	SCRIPT_DATA		data;			// Donnes pour l'animation de l'unit par le script
	float			port[21];		// Ports
	MISSION			*mission;		// Ordres de l'unit
	int				nb_running;		// Nombre de scripts lancs en mme temps
	SCRIPT_ENV		script_env[25];	// Environnements des scripts
	byte			flags;			// Pour indiquer entre autres au gestionnaire d'units si l'unit existe
	float			c_time;			// Compteur de temps entre 2 missions de particules par une unit de construction
	bool			compute_coord;	// Indique s'il est ncessaire de recalculer les coordonnes du modle 3d
	int				script_val[25];	// Tableau de valeurs retournes par les scripts
	int				idx;			// Indice dans le tableau d'unit
	VECTOR			aim_dir;		// Vecteur de vise
	float			h;				// Altitude (par rapport au sol)
	bool			visible;		// Indique si l'unit est visible
	int				burst[3];		// Pour la gestion des tirs
	float			b_delay[3];
	int				stock[3];		// Stock de munitions si besoin d'en fabriquer / ammo stock
	short			groupe;			// Indique si l'unit fait partie d'un groupe
	bool			built;			// Indique si l'unit est en cours de construction (par une autre unit)
	bool			attacked;		// Indique si l'unit est attaque
	float			planned_weapons;	// Armes en construction / all is in the name
	int				memory[100];	// Pour se rappeler sur quelles armes on a dj tir
	int				mem_size;
	int				script_idx[30];	// Index of scripts to prevent multiple search
	bool			attached;
	int				attached_list[20];
	int				link_list[20];
	int				nb_attached;

	inline bool do_nothing()
	{
		return mission==NULL || ((mission->mission==MISSION_STOP || mission->mission==MISSION_STANDBY || mission->mission==MISSION_VTOL_STANDBY) && mission->next==NULL);
	}

	inline void start_mission_script(int mission_type)
	{
		if(script==NULL)	return;
		switch(mission_type)
		{
		case MISSION_ATTACK:
		case MISSION_AIM:
		case MISSION_SHOOT:
//			activate();
			break;
		case MISSION_PATROL:
		case MISSION_MOVE_CALC_PATH:
		case MISSION_MOVE:
			launch_script(get_script_index(SCRIPT_MotionControl));
			launch_script(get_script_index(SCRIPT_startmoving));
			if(nb_attached==0)
				launch_script(get_script_index(SCRIPT_MoveRate1));		// Pour l'atlas
/*			launch_script(get_script_index("MotionControl"));
			launch_script(get_script_index("startmoving"));
			launch_script(get_script_index("MoveRate1"));		// Pour l'atlas*/
			break;
		case MISSION_BUILD_2:
			break;
		case MISSION_RECLAIM:
			break;
		};
		if(nb_attached>0)
			launch_script(get_script_index(SCRIPT_MoveRate2));
	}

	inline void next_mission()
	{
		if(mission==NULL) {
			set_mission(unit_manager.unit_type[type_id].DefaultMissionType,false,0,false);
			return;
			}
		switch(mission->mission)		// Commandes de fin de mission
		{
		case MISSION_REPAIR:
		case MISSION_RECLAIM:
		case MISSION_BUILD_2:
			launch_script(get_script_index(SCRIPT_stopbuilding));
			launch_script(get_script_index(SCRIPT_stop));
/*			launch_script(get_script_index("stopbuilding"));
			launch_script(get_script_index("stop"));*/
			break;
		case MISSION_ATTACK:
		case MISSION_AIM:
		case MISSION_SHOOT:
			deactivate();
			break;
		};
		if(mission->mission==MISSION_STOP && mission->next==NULL) {
			mission->data=0;
			return;
			}
		MISSION *old=mission;
		mission=mission->next;
		if(old->path)				// Dtruit le chemin si ncessaire
			destroy_path(old->path);
		free(old);
		if(mission==NULL)
			set_mission(unit_manager.unit_type[type_id].DefaultMissionType);
		start_mission_script(mission->mission);
		c_time=0.0f;
	}

	inline void clear_mission()
	{
		if(mission==NULL)	return;
		MISSION *old=mission;
		mission=mission->next;
		if(old->path)				// Dtruit le chemin si ncessaire
			destroy_path(old->path);
		free(old);
	}

	void add_mission(int mission_type,VECTOR *target=NULL,bool step=false,int dat=0,void *pointer=NULL,PATH_NODE *path=NULL,byte flags=0);

	void set_mission(int mission_type,VECTOR *target=NULL,bool step=false,int dat=0,bool stopit=true,void *pointer=NULL,PATH_NODE *path=NULL,byte flags=0);

	inline void compute_model_coord()
	{
		if(!compute_coord)	return;
		if(model==NULL)	return;		// S'il n'y a pas de modle associ, on quitte la fonction
		compute_coord=false;
		MATRIX_4x4 M;
		float scale=unit_manager.unit_type[type_id].Scale;
		M=RotateZ(Angle.z*DEG2RAD)*RotateY(Angle.y*DEG2RAD)*RotateX(Angle.x*DEG2RAD)*Scale(scale);			// Matrice pour le calcul des positions des lments du modle de l'unit
		model->compute_coord(&data,&M);
	}

	inline void raise_signal(int signal)		// Tue les processus associs
	{
		SCRIPT_ENV_STACK *tmp;
		for(int i=0;i<nb_running;i++) {
			tmp=script_env[i].env;
			while(tmp) {
				if(tmp->signal_mask==signal) {
					tmp=script_env[i].env;
					while(tmp!=NULL) {
						script_env[i].env=tmp->next;
						delete tmp;
						tmp=script_env[i].env;
						}
					}
				if(tmp)
					tmp=tmp->next;
				}
			if(script_env[i].env==NULL)
				script_env[i].running=false;
			}
	}

	inline void init(int unit_type=-1,int owner=-1)
	{
		for(int i=0;i<30;i++)	script_idx[i]=-2;	// Not yet searched
		attached=false;
		nb_attached=0;
		mem_size=0;
		planned_weapons=0.0f;
		attacked=false;
		groupe=0;
		burst[0]=0;
		burst[1]=0;
		burst[2]=0;
		b_delay[0]=0.0f;
		b_delay[1]=0.0f;
		b_delay[2]=0.0f;
		stock[0]=0;
		stock[1]=0;
		stock[2]=0;
		h=0.0f;
		compute_coord=true;
		c_time=0.0f;
		flags=1;
		sel=false;
		script=NULL;
		model=NULL;
		owner_id=owner;
		type_id=-1;
		hp=0.0f;
		V.x=V.y=V.z=0.0f;
		Pos=V;
		data.init();
		Angle.x=Angle.y=Angle.z=0.0f;
		V_Angle=Angle;
		nb_running=0;
		int i;
		for(i=0;i<25;i++) {
			script_env[i].init();
			script_env[i].running=false;
			}
		for(i=0;i<21;i++)
			port[i]=0;
		for(i=0;i<25;i++)
			s_var[i]=0;
		if(unit_type<0 || unit_type>=unit_manager.nb_unit)
			unit_type=-1;
		port[ACTIVATION]=0;
		mission=NULL;
		port[BUILD_PERCENT_LEFT]=0.0f;
		if(unit_type!=-1) {
			set_mission(MISSION_STANDBY);
			type_id=unit_type;
			model=unit_manager.unit_type[type_id].model;
			hp=unit_manager.unit_type[type_id].MaxDamage;
			script=unit_manager.unit_type[type_id].script;
			port[STANDINGMOVEORDERS]=unit_manager.unit_type[type_id].StandingMoveOrder;
			port[STANDINGFIREORDERS]=unit_manager.unit_type[type_id].StandingFireOrder;
			set_mission(unit_manager.unit_type[type_id].DefaultMissionType);
			if(script) {
				data.load(script->nb_piece);
				launch_script(get_script_index(SCRIPT_create));
//				launch_script(get_script_index("create"));
				}
			}
	}

	inline UNIT()
	{
		init();
	}

	inline void destroy()
	{
		for(int i=0;i<nb_running;i++)
			script_env[i].destroy();
		while(mission) clear_mission();
		init();
		flags=0;
		groupe=0;
	}

	inline ~UNIT()
	{
		destroy();
	}

	void draw(CAMERA *cam,MAP *map,bool cullface=true);

	void draw_shadow(CAMERA *cam,VECTOR Dir,MAP *map);

	void draw_shadow_basic(CAMERA *cam,VECTOR Dir,MAP *map);

	inline int get_script_index(int id)
	{
		if(script_idx[id]!=-2)	return script_idx[id];
		char *script_name[]= {	"QueryPrimary","AimPrimary","FirePrimary",
								"QuerySecondary","AimSecondary","FireSecondary",
								"QueryTertiary","AimTertiary","FireTertiary",
								"TargetCleared","stopbuilding","stop",
								"startbuilding","go","killed",
								"StopMoving","Deactivate","Activate",
								"create","MotionControl","startmoving",
								"MoveRate1","MoveRate2","MoveRate3",
								"RequestState","TransportPickup","TransportDrop",
								"QueryTransport","BeginTransport","EndTransport"};
		script_idx[id]=get_script_index(script_name[id]);
		return script_idx[id];
	}

	inline int get_script_index(char *script_name)			// Cherche l'indice du script dont on fournit le nom
	{
		if(script)
			for(int i=0;i<script->nb_script;i++)
				if(strcasecmp(script->name[i],script_name)==0)
					return i;
		return -1;
	}

	inline int launch_script(int id,int nb_param=0,int *param=NULL)
	{
		if(script==NULL)	return -2;
		if(id<0 || id>=script->nb_script)
			return -2;
		if(is_running(id))			// le script tourne dj / script already running
			return -1;
		script_env[nb_running].init();
		script_env[nb_running].env=new SCRIPT_ENV_STACK;
		script_env[nb_running].env->init();
		script_env[nb_running].env->cur=id;
		script_env[nb_running].running=true;
		if(nb_param>0 && param!=NULL)
			for(int i=0;i<nb_param;i++)
				script_env[nb_running].env->var[i]=param[i];
		return nb_running++;
	}

	inline bool is_running(int script_index)
	{
		if(script==NULL)	return false;
		if(script_index<0 || script_index>=script->nb_script)	return false;
		for(int i=0;i<nb_running;i++)
			if(script_env[i].running && script_env[i].env!=NULL) {
				SCRIPT_ENV_STACK *current=script_env[i].env;
				while(current) {
					if((current->cur&0xFF)==script_index)
						return true;
					current=current->next;
					}
				}
		return false;
	}

	inline void kill_script(int script_index)		// Fait un peu de mnage
	{
		if(script==NULL)	return;
		if(script_index<0 || script_index>=script->nb_script)	return;
		for(int i=0;i<nb_running;i++)
			if(script_env[i].running && script_env[i].env!=NULL) {
				SCRIPT_ENV_STACK *current=script_env[i].env;
				while(current) {
					if((current->cur&0xFF)==script_index) {		// Tue le script trouv
						current=script_env[i].env;
						script_env[i].running=false;
						while(current) {
							script_env[i].env=current->next;
							delete current;
							current=script_env[i].env;
							}
						break;
						}
					current=current->next;
					}
				}
		int e=0;
		for(int i=0;i+e<nb_running;) {				// Efface les scripts qui se sont arrts
			if(script_env[i+e].running) {
				script_env[i]=script_env[i+e];
				i++;
				}
			else {
				script_env[i+e].destroy();
				e++;
				}
			}
		nb_running-=e;
	}

	inline void reset_script()
	{
		for(int i=0;i<nb_running;i++)
			script_env[i].destroy();
		nb_running=0;
	}

	void run_script(float dt,int id,MAP *map);			// Interprte les scripts lis  l'unit

	int move(float dt,MAP *map,int *path_exec,int key_frame=0);

	void show_orders();				// Dessine les ordres reus

	inline void activate()
	{
		if(port[ACTIVATION]==0.0f) {
			sound_manager.play(unit_manager.unit_type[type_id].activate,Pos);
			kill_script(get_script_index(SCRIPT_Deactivate));
			launch_script(get_script_index(SCRIPT_Activate));
/*			kill_script(get_script_index("Deactivate"));
			launch_script(get_script_index("Activate"));*/
			port[ACTIVATION]=1.0f;
			}
	}

	inline void deactivate()
	{
		if(port[ACTIVATION]!=0.0f) {
			sound_manager.play(unit_manager.unit_type[type_id].deactivate,Pos);
			kill_script(get_script_index(SCRIPT_Activate));
			launch_script(get_script_index(SCRIPT_Deactivate));
/*			kill_script(get_script_index("Activate"));
			launch_script(get_script_index("Deactivate"));*/
			port[ACTIVATION]=0.0f;
			}
	}

	void shoot(int target,VECTOR startpos,VECTOR Dir,int w_id=0);

	bool hit(VECTOR P,VECTOR Dir,VECTOR *hit_vec);
};

class INGAME_UNITS		// Classe pour la gestion massive des units pendant la partie
{
public:
/*----------------------- Variables gnrales ----------------------------------------------*/
	int		nb_unit;		// Nombre d'units
	int		max_unit;		// Nombre maximum d'units stockables dans le tableau
	UNIT	*unit;			// Tableau contenant les rfrences aux units
	int		last_index;		// Dernier indice contenant une unit

/*----------------------- Variables rserves au joueur courant ----------------------------*/

	float	nb_attacked;
	float	nb_built;

	int		page;

	inline void init()
	{
		page=0;
		nb_unit=0;
		unit=NULL;
		max_unit=0;
		last_index=-1;
		nb_attacked=0.0f;
		nb_built=0.0f;
	}

	inline INGAME_UNITS()
	{
		init();
	}

	inline void destroy()
	{
		if(max_unit>0 && unit)			// Dtruit une  une les units
			for(int i=0;i<max_unit;i++)
				unit[i].destroy();
		if(unit)
			free(unit);
		init();
	}

	inline ~INGAME_UNITS()
	{
		destroy();
	}

	inline void kill(int index,MAP *map)			// Dtruit une unit
	{
		if(index<0 || index>last_index)	return;		// On ne peut pas dtruire une unit qui n'existe pas
		if(unit[index].flags) {
			map->rect((((int)(unit[index].Pos.x+map->map_w*0.5f))>>3)-(unit_manager.unit_type[unit[index].type_id].FootprintX>>1),(((int)(unit[index].Pos.z+map->map_h*0.5f))>>3)-(unit_manager.unit_type[unit[index].type_id].FootprintZ>>1),unit_manager.unit_type[unit[index].type_id].FootprintX,unit_manager.unit_type[unit[index].type_id].FootprintZ,-1);
			players.nb_unit[unit[index].owner_id]--;
			players.losses[unit[index].owner_id]++;		// Statistiques
			if(unit_manager.unit_type[unit[index].type_id].canload && unit[index].nb_attached>0)
				for(int i=0;i<unit[index].nb_attached;i++)
					unit[unit[index].attached_list[i]].hp=0.0f;
			unit[index].destroy();		// Dtruit l'unit
			}
		if(index==last_index)			// Redtermine le dernier indice
			while(last_index>=0 && unit[last_index].flags==0)	last_index--;
		nb_unit--;		// Unit dtruite
	}

	inline void draw(CAMERA *cam,MAP *map,bool underwater=false,bool limit=false,bool cullface=true)					// Dessine les units visibles
	{
		if(nb_unit<=0 || unit==NULL)	return;		// Pas d'units  dessiner

		glEnable(GL_LIGHTING);
		glDisable(GL_BLEND);
		glColor4f(1.0f,1.0f,1.0f,1.0f);
		float sea_lvl = limit ? map->sealvl-5.0f : map->sealvl;
		for(int i=0;i<=last_index;i++)
			if(unit[i].flags && ((unit[i].Pos.y<=map->sealvl && underwater) || (unit[i].Pos.y>=sea_lvl && !underwater))) {				// Si il y a une unit
				unit[i].draw(cam,map,cullface);
				players.annihilated[unit[i].owner_id]=false;
				}
	}

	inline void draw_shadow(CAMERA *cam,VECTOR Dir,MAP *map,float alpha=0.5f)					// Dessine les ombres des units visibles
	{
		if(nb_unit<=0 || unit==NULL)	return;		// Pas d'units  dessiner

		if(g_useStencilTwoSide) {					// Si l'extension GL_EXT_stencil_two_side est disponible
			glEnable(GL_STENCIL_TEST);
			glEnable(GL_STENCIL_TEST_TWO_SIDE_EXT);
			glDisable(GL_CULL_FACE);
			glClear(GL_STENCIL_BUFFER_BIT);
			glDepthMask(GL_FALSE);
			glColorMask(0,0,0,0);
			glStencilFunc(GL_ALWAYS,128, 0xffffffff);

			glActiveStencilFaceEXT(GL_FRONT);
			glStencilOp(GL_KEEP, GL_KEEP, GL_INCR_WRAP_EXT);
			glActiveStencilFaceEXT(GL_BACK);
			glStencilOp(GL_KEEP, GL_KEEP, GL_DECR_WRAP_EXT);

			for(int i=0;i<=last_index;i++)
				if(unit[i].flags)				// Si il y a une unit
					unit[i].draw_shadow(cam,Dir,map);
			}
		else {										// Si elle ne l'est pas
			glDepthMask(GL_FALSE);
			glColorMask(0,0,0,0);

			glClear(GL_STENCIL_BUFFER_BIT);
			glEnable(GL_STENCIL_TEST);
			glStencilFunc(GL_ALWAYS,128, 0xffffffff);
			glEnable(GL_CULL_FACE);

			for(int i=0;i<=last_index;i++)
				if(unit[i].flags)					// Si il y a une unit
					unit[i].draw_shadow_basic(cam,Dir,map);
			}
		features.draw_shadow(cam,Dir);

		glColorMask(0xFF,0xFF,0xFF,0xFF);
		allegro_gl_set_allegro_mode();
		glEnable(GL_STENCIL_TEST);
		glStencilFunc(GL_NOTEQUAL,0, 0xffffffff);
		glStencilOp(GL_KEEP,GL_KEEP,GL_KEEP);
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
		glEnable(GL_BLEND);
		glColor4f(0.0f,0.0f,0.0f,alpha);
		glBegin(GL_QUADS);
			glVertex2i(0,0);
			glVertex2i(SCREEN_W,0);
			glVertex2i(SCREEN_W,SCREEN_H);
			glVertex2i(0,SCREEN_H);
		glEnd();
		allegro_gl_unset_allegro_mode();
		glDepthMask(GL_TRUE);
		glDisable(GL_STENCIL_TEST);
		glDisable(GL_BLEND);
		glColor4f(1.0f,1.0f,1.0f,1.0f);
	}

	inline void draw_mini(float map_w,float map_h,int mini_w,int mini_h,SECTOR **map_data)				// Repre les units sur la mini-carte
	{
		if(nb_unit<=0 || unit==NULL)	return;		// Pas d'units  dessiner

		float rw=128.0f*mini_w/252/map_w;
		float rh=128.0f*mini_h/252/map_h;

		glDisable(GL_TEXTURE_2D);
		glPointSize(3.0f);
		glBegin(GL_POINTS);
		int cur_id=-1;
		byte mask=1<<players.local_human_id;
		int b_w=(int)map_w>>3;
		int b_h=(int)map_h>>3;
		for(int i=0;i<=last_index;i++)
			if(unit[i].flags!=0) {
				int px=((int)(unit[i].Pos.x+map_w*0.5f))>>3;
				int py=((int)(unit[i].Pos.z+map_h*0.5f))>>3;
				if(px<0 || py<0 || px>=b_w || py>=b_h)	continue;
				if(!(map_data[py][px].view&mask) || !map_data[py][px].timer) continue;	// Unit non visible / Unit is not visible
				unit[i].flags|=0x10;
				if(cur_id!=unit[i].owner_id) {
					cur_id=unit[i].owner_id;
					glColor3f(0.5f*player_color[3*cur_id],0.5f*player_color[3*cur_id+1],0.5f*player_color[3*cur_id+2]);
					}
				float pos_x=unit[i].Pos.x*rw+64.0f;
				float pos_y=unit[i].Pos.z*rh+64.0f;
				glVertex2f(pos_x-1.0f,pos_y);
				glVertex2f(pos_x+1.0f,pos_y);
				glVertex2f(pos_x,pos_y-1.0f);
				glVertex2f(pos_x,pos_y+1.0f);
				}
		cur_id=-1;
		for(int i=0;i<=last_index;i++)
			if(unit[i].flags!=0 && (unit[i].flags&0x10)==0x10) {
				unit[i].flags^=0x10;
				if(cur_id!=unit[i].owner_id) {
					cur_id=unit[i].owner_id;
					glColor3f(player_color[3*cur_id],player_color[3*cur_id+1],player_color[3*cur_id+2]);
					}
				glVertex2f(unit[i].Pos.x*rw+64.0f,unit[i].Pos.z*rh+64.0f);
				}
		cur_id=players.local_human_id;
		for(int i=0;i<=last_index;i++)
			if(unit[i].flags!=0 && unit[i].owner_id==players.local_human_id && unit[i].sel) {
				glColor3f(1.0f,1.0f,1.0f);
				float pos_x=unit[i].Pos.x*rw+64.0f;
				float pos_y=unit[i].Pos.z*rh+64.0f;
				glVertex2f(pos_x-1.0f,pos_y);
				glVertex2f(pos_x+1.0f,pos_y);
				glVertex2f(pos_x,pos_y-1.0f);
				glVertex2f(pos_x,pos_y+1.0f);

				glColor3f(player_color[3*cur_id],player_color[3*cur_id+1],player_color[3*cur_id+2]);
				glVertex2f(pos_x,pos_y);
				}
		glEnd();
		glPointSize(1.0f);
		glEnable(GL_TEXTURE_2D);
	}

	inline void move(float dt,MAP *map=NULL,int key_frame=0)
	{
		if(nb_unit<=0 || unit==NULL)	return;		// Pas d'units  dessiner

		players.clear();		// Rinitialise le compteur de ressources

		int i;
		for(i=0;i<=last_index;i++)		// Compte les stocks de ressources et les productions
			if(unit[i].flags && unit[i].port[BUILD_PERCENT_LEFT]==0.0f) {
				players.metal_s[unit[i].owner_id]+=unit_manager.unit_type[unit[i].type_id].MetalStorage;
				players.energy_s[unit[i].owner_id]+=unit_manager.unit_type[unit[i].type_id].EnergyStorage;
				players.energy_t[unit[i].owner_id]+=unit_manager.unit_type[unit[i].type_id].EnergyMake;
				players.commander[unit[i].owner_id]|=(unit_manager.unit_type[unit[i].type_id].TEDclass==CLASS_COMMANDER);
				if((unit[i].port[ACTIVATION] || !unit_manager.unit_type[unit[i].type_id].onoffable)
				&& unit_manager.unit_type[unit[i].type_id].EnergyUse<=players.energy[unit[i].owner_id]) {
					players.metal_t[unit[i].owner_id]+=unit_manager.unit_type[unit[i].type_id].MakesMetal+unit_manager.unit_type[unit[i].type_id].MetalMake;
					if(unit_manager.unit_type[unit[i].type_id].ExtractsMetal) {	// Extracteur de mtal
						int metal_base;
						int px,py;
						px=(int)((unit[i].Pos.x+map->map_w*0.5f)/map->map_w*(map->bloc_w<<1));
						py=(int)((unit[i].Pos.z+map->map_h*0.5f)/map->map_h*(map->bloc_h<<1));
						if(map->map_data[py][px].stuff>=0)
							metal_base=feature_manager.feature[features.feature[map->map_data[py][px].stuff].type].metal;
						else
							metal_base=map->map_data[py][px].metal;
						players.metal_t[unit[i].owner_id]+=10.0f*metal_base*unit_manager.unit_type[unit[i].type_id].ExtractsMetal;
						}
					if(unit_manager.unit_type[unit[i].type_id].EnergyUse<0)
						players.energy_t[unit[i].owner_id]-=unit_manager.unit_type[unit[i].type_id].EnergyUse;
					else
						players.energy_u[unit[i].owner_id]+=unit_manager.unit_type[unit[i].type_id].EnergyUse;
					}
				}
		for(i=0;i<players.nb_player;i++)
			if(players.commander[i]) {
				players.metal_s[i]+=players.com_metal;
				players.energy_s[i]+=players.com_energy;
				}

		int path_exec=0;
		for(i=0;i<=last_index;i++)
			if(unit[i].flags) {
				if(unit[i].owner_id==players.local_human_id) {
					if(unit[i].attacked
					|| (unit[i].mission!=NULL && unit[i].mission->mission==MISSION_SHOOT))	nb_attacked+=100;
					if(unit[i].built)		nb_built++;
					}
				players.nb_unit[unit[i].owner_id]++;			// Compte les units de chaque joueur
				if(unit[i].move(dt,map,&path_exec,key_frame)==-1)			// Vrifie si l'unit a t dtruite
					kill(i,map);
				}
		nb_attacked*=exp(-dt*0.1f);
		nb_built*=exp(-dt*0.1f);

		for(i=0;i<players.nb_player;i++) {
			players.metal[i]+=dt*(players.metal_t[i]-players.metal_u[i]);
			players.energy[i]+=dt*(players.energy_t[i]-players.energy_u[i]);
			if(players.metal[i]<0.0f)
				players.metal[i]=0.0f;
			else if(players.metal[i]>players.metal_s[i])
				players.metal[i]=players.metal_s[i];
			if(players.energy[i]<0.0f)
				players.energy[i]=0.0f;
			else if(players.energy[i]>players.energy_s[i])
				players.energy[i]=players.energy_s[i];
			}
	}

	inline int create(int type_id,int owner)
	{
		if(type_id<0 || type_id>=unit_manager.nb_unit)	return -1;
		if(owner<0 || owner>=NB_PLAYERS)	return -1;
		if(nb_unit>=MAX_UNIT_PER_PLAYER*NB_PLAYERS)		return -1;
		nb_unit++;
		if(nb_unit>max_unit) {
			max_unit=MAX_UNIT_PER_PLAYER*NB_PLAYERS;
			UNIT *n_unit=(UNIT*) malloc(sizeof(UNIT)*max_unit);
			for(int i=0;i<max_unit;i++) {
				n_unit[i].init();
				n_unit[i].flags=0;
				n_unit[i].idx=i;
				}
			if(unit) {
				memcpy(n_unit,unit,sizeof(UNIT)*(nb_unit-1));
				free(unit);
				}
			unit=n_unit;
			}
		if(unit==NULL)
			printf("erreur: allocation de mmoire choue\n");
		int unit_index=owner*MAX_UNIT_PER_PLAYER;
		while(unit_index<max_unit && unit[unit_index].flags) unit_index++;		// La nouvelle unit occupe la premire place libre
		if(unit_index>=(owner+1)*MAX_UNIT_PER_PLAYER) {	// Dfense de dborder
			printf("nombre limite d'unit atteint!\n");
			return -1;
			}
		unit[unit_index].init(type_id,owner);
		unit[unit_index].Angle.y=((rand()%2001)-1000)*0.01f;	// Angle de 10 maximum
		if(unit_index>last_index)
			last_index=unit_index;
		players.nb_unit[owner]++;
		return unit_index;
	}

	bool select(CAMERA *cam,int sel_x[],int sel_y[]);

	int pick(CAMERA *cam,int sensibility=1);

	inline void give_order_move(int player_id,VECTOR target,bool set=true,byte flags=0)
	{
		for(int i=0;i<=last_index;i++)
			if(unit[i].flags!=0 && unit[i].owner_id==player_id && unit[i].sel && unit[i].port[BUILD_PERCENT_LEFT]==0.0f && unit_manager.unit_type[unit[i].type_id].canmove) {
				if(set)
					unit[i].set_mission(MISSION_MOVE_CALC_PATH,&target,false,0,true,NULL,NULL,flags);
				else
					unit[i].add_mission(MISSION_MOVE_CALC_PATH,&target,false,0,NULL,NULL,flags);
				if(player_id==players.local_human_id)
					sound_manager.play(unit_manager.unit_type[unit[i].type_id].ok1,unit[i].Pos);
				}
	}

	inline void give_order_patrol(int player_id,VECTOR target,bool set=true)
	{
		for(int i=0;i<=last_index;i++)
			if(unit[i].flags!=0 && unit[i].owner_id==player_id && unit[i].sel && unit[i].port[BUILD_PERCENT_LEFT]==0.0f && unit_manager.unit_type[unit[i].type_id].canpatrol) {
				if(set)
					unit[i].set_mission(MISSION_PATROL,&target,false,0,true,NULL,NULL);
				else
					unit[i].add_mission(MISSION_PATROL,&target,false,0,NULL,NULL);
				if(player_id==players.local_human_id)
					sound_manager.play(unit_manager.unit_type[unit[i].type_id].ok1,unit[i].Pos);
				}
	}

	inline void give_order_guard(int player_id,int target,bool set=true)
	{
		for(int i=0;i<=last_index;i++)
			if(unit[i].flags!=0 && unit[i].owner_id==player_id && unit[i].sel && unit[i].port[BUILD_PERCENT_LEFT]==0.0f && unit_manager.unit_type[unit[i].type_id].canguard) {
				if(set)
					unit[i].set_mission(MISSION_GUARD,&unit[target].Pos,false,0,true,&(unit[target]),NULL);
				else
					unit[i].add_mission(MISSION_GUARD,&unit[target].Pos,false,0,&(unit[target]),NULL);
				if(player_id==players.local_human_id)
					sound_manager.play(unit_manager.unit_type[unit[i].type_id].ok1,unit[i].Pos);
				}
	}

	inline void give_order_load(int player_id,int target,bool set=true)
	{
		if(unit[target].flags==0 || !unit_manager.unit_type[unit[target].type_id].canmove)	return;
		switch(unit_manager.unit_type[unit[target].type_id].TEDclass)
		{
		case CLASS_UNDEF:
		case CLASS_WATER:
		case CLASS_SHIP:
		case CLASS_PLANT:
		case CLASS_SPECIAL:
		case CLASS_FORT:
			return;
			break;
		};
		for(int i=0;i<=last_index;i++)
			if(unit[i].flags!=0 && unit[i].owner_id==player_id && unit[i].sel && unit[i].port[BUILD_PERCENT_LEFT]==0.0f && unit_manager.unit_type[unit[i].type_id].canload) {
				if(set)
					unit[i].set_mission(MISSION_LOAD,&unit[target].Pos,false,0,true,&(unit[target]),NULL);
				else
					unit[i].add_mission(MISSION_LOAD,&unit[target].Pos,false,0,&(unit[target]),NULL);
				if(player_id==players.local_human_id)
					sound_manager.play(unit_manager.unit_type[unit[i].type_id].ok1,unit[i].Pos);
				}
	}

	inline void give_order_build(int player_id,int unit_type_id,VECTOR target,bool set=true)
	{
		for(int i=0;i<=last_index;i++)
			if(unit[i].flags!=0 && unit[i].owner_id==player_id && unit[i].sel && unit[i].port[BUILD_PERCENT_LEFT]==0.0f && unit_manager.unit_type[unit[i].type_id].Builder) {
				if(player_id==players.local_human_id)
					sound_manager.play(unit_manager.unit_type[unit[i].type_id].build,unit[i].Pos);
				if(set)
					unit[i].set_mission(MISSION_BUILD,&target,false,unit_type_id);
				else
					unit[i].add_mission(MISSION_BUILD,&target,false,unit_type_id);
				}
	}

#define SIGNAL_ORDER_NONE	0x0
#define SIGNAL_ORDER_MOVE	0x1
#define SIGNAL_ORDER_PATROL	0x2
#define SIGNAL_ORDER_GUARD	0x3
#define SIGNAL_ORDER_ATTACK	0x4
#define SIGNAL_ORDER_RECLAM	0x5
#define SIGNAL_ORDER_STOP	0x6
#define SIGNAL_ORDER_ONOFF	0x7
#define SIGNAL_ORDER_LOAD	0x8
#define SIGNAL_ORDER_UNLOAD	0x9

	inline int menu_order(int omb,int player_id)							// Menu permettant de donner des ordres
	{
		bool onoffable=false;
		bool canstop=false;
		bool canpatrol=false;
		bool canmove=false;
		bool canguard=false;
		bool canattack=false;
		bool canreclam=false;
		bool builders=false;
		bool canload=false;

		int signal=SIGNAL_ORDER_NONE;

		int i;
		for(i=0;i<=last_index;i++)
			if(unit[i].flags && unit[i].owner_id==player_id && unit[i].sel) {
				onoffable|=unit_manager.unit_type[unit[i].type_id].onoffable;
				canstop|=unit_manager.unit_type[unit[i].type_id].canstop;
				canmove|=unit_manager.unit_type[unit[i].type_id].canmove;
				canpatrol|=unit_manager.unit_type[unit[i].type_id].canpatrol;
				canguard|=unit_manager.unit_type[unit[i].type_id].canguard;
				canattack|=unit_manager.unit_type[unit[i].type_id].canattack;
				canreclam|=unit_manager.unit_type[unit[i].type_id].CanReclamate;
				builders|=unit_manager.unit_type[unit[i].type_id].Builder;
				canload|=unit_manager.unit_type[unit[i].type_id].canload;
				}

		bool chg_page=false;
		if(mouse_b==1 && mouse_y>=420 && mouse_y<=476)
			if(mouse_x>=5 && mouse_x<30)
				chg_page=true;
		if(page==0)
			glbutton("1",5.0f,420.0f,30.0f,476.0f,chg_page);
		else
			glbutton("2",5.0f,420.0f,30.0f,476.0f,chg_page);
		if(chg_page && omb!=1)
			page=++page%2;

	if(page==0) {
		if(onoffable) {					// Fonction On/Off
			bool on=false,off=false;

			if(mouse_b==1 && mouse_y>=340 && mouse_y<=356) {
				if(mouse_x>=5 && mouse_x<62)
					on=true;
				if(mouse_x>66 && mouse_x<=123)
					off=true;
				}

			if(on && omb!=1) {
				signal=SIGNAL_ORDER_ONOFF;
				for(i=0;i<=last_index;i++)
					if(unit[i].flags!=0 && unit[i].owner_id==player_id && unit[i].sel)
						unit[i].activate();
				}
			if(off && omb!=1) {
				signal=SIGNAL_ORDER_ONOFF;
				for(i=0;i<=last_index;i++)
					if(unit[i].flags!=0 && unit[i].owner_id==player_id && unit[i].sel)
						unit[i].deactivate();
				}

			glbutton("on",5.0f,340.0f,61.5f,356.0f,on);
			glbutton("off",66.5f,340.0f,123.0f,356.0f,off);
			}

		if(canstop) {					// Fonction Stop
			bool stop=false;

			if(mouse_b==1 && mouse_y>=360 && mouse_y<=376) {
				if(mouse_x>=5 && mouse_x<62)
					stop=true;
				}

			if(stop && omb!=1) {
				signal=SIGNAL_ORDER_STOP;
				for(i=0;i<=last_index;i++)
					if(unit[i].flags!=0 && unit[i].owner_id==player_id && unit[i].sel && unit_manager.unit_type[unit[i].type_id].canstop)
						unit[i].set_mission(MISSION_STOP);
				}

			glbutton("stop",5.0f,360.0f,61.5f,376.0f,stop);
			}

		if(canmove) {					// Fonction Move
			bool move=false;

			if(mouse_b==1 && mouse_y>=360 && mouse_y<=376) {
				if(mouse_x>66 && mouse_x<=123)
					move=true;
				}

			if(move && omb!=1)
				signal=SIGNAL_ORDER_MOVE;

			glbutton("move",66.5f,360.0f,123.0f,376.0f,move);
			}

		if(canpatrol) {					// Fonction Patrol
			bool patrol=false;

			if(mouse_b==1 && mouse_y>=380 && mouse_y<=396) {
				if(mouse_x>66 && mouse_x<=123)
					patrol=true;
				}

			if(patrol && omb!=1)
				signal=SIGNAL_ORDER_PATROL;

			glbutton("patrol",66.5f,380.0f,123.0f,396.0f,patrol);
			}

		if(canguard) {					// Fonction Guard
			bool guard=false;

			if(mouse_b==1 && mouse_y>=380 && mouse_y<=396) {
				if(mouse_x>=5 && mouse_x<62)
					guard=true;
				}

			if(guard && omb!=1)
				signal=SIGNAL_ORDER_GUARD;

			glbutton("guard",5.0f,380.0f,61.5f,396.0f,guard);
			}

		if(canattack) {					// Fonction Attack
			bool attack=false;

			if(mouse_b==1 && mouse_y>=400 && mouse_y<=416) {
				if(mouse_x>66 && mouse_x<=123)
					attack=true;
				}

			if(attack && omb!=1)
				signal=SIGNAL_ORDER_ATTACK;

			glbutton("attack",66.5f,400.0f,123.0f,416.0f,attack);
			}

		if(canreclam) {					// Fonction Reclamate
			bool reclam=false;

			if(mouse_b==1 && mouse_y>=400 && mouse_y<=416) {
				if(mouse_x>=5 && mouse_x<62)
					reclam=true;
				}

			if(reclam && omb!=1)
				signal=SIGNAL_ORDER_RECLAM;

			glbutton("reclam",5.0f,400.0f,61.5f,416.0f,reclam);
			}
		}
	else if(page==1) {
		if(canload) {					// Fonction load
			bool load=false;
			bool unload=false;

			if(mouse_b==1 && mouse_y>=340 && mouse_y<=356) {
				if(mouse_x>=5 && mouse_x<62)
					load=true;
				if(mouse_x>66 && mouse_x<=123)
					unload=true;
				}

			if(load && omb!=1)
				signal=SIGNAL_ORDER_LOAD;
			if(unload && omb!=1)
				signal=SIGNAL_ORDER_UNLOAD;

			glbutton("load",5.0f,340.0f,61.5f,356.0f,load);
			glbutton("unload",66.5f,340.0f,123.0f,356.0f,unload);
			}
		}
		return signal;
	}

	inline void complete_menu(int index)
	{
		if(index<0 || index>last_index) return;		// On n'affiche que des donnes sur les units EXISTANTES
		if(unit[index].flags==0) return;		// L'unit n'existe plus

		UNIT *target=unit[index].mission!=NULL ? (UNIT*) unit[index].mission->p : NULL;
		if(target && target->flags==0)
			target=NULL;

		glEnable(GL_TEXTURE_2D);
		glEnable(GL_BLEND);
		glBlendFunc(GL_ZERO,GL_ONE_MINUS_SRC_COLOR);
		allegro_gl_printf(aglfont,149.0f,SCREEN_H-28.0f,0.0f,0x0,unit_manager.unit_type[unit[index].type_id].name);
		allegro_gl_printf(aglfont,151.0f,SCREEN_H-28.0f,0.0f,0x0,unit_manager.unit_type[unit[index].type_id].name);
		allegro_gl_printf(aglfont,150.0f,SCREEN_H-29.0f,0.0f,0x0,unit_manager.unit_type[unit[index].type_id].name);
		allegro_gl_printf(aglfont,150.0f,SCREEN_H-27.0f,0.0f,0x0,unit_manager.unit_type[unit[index].type_id].name);
		glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_COLOR);
		allegro_gl_printf(aglfont,150.0f,SCREEN_H-28.0f,0.0f,0xFFFFFF,unit_manager.unit_type[unit[index].type_id].name);
		if(target && (unit[index].mission->flags&4)!=4) {
			glBlendFunc(GL_ZERO,GL_ONE_MINUS_SRC_COLOR);
			allegro_gl_printf(aglfont,174.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0x0,unit_manager.unit_type[target->type_id].name);
			allegro_gl_printf(aglfont,176.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0x0,unit_manager.unit_type[target->type_id].name);
			allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-29.0f,0.0f,0x0,unit_manager.unit_type[target->type_id].name);
			allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-27.0f,0.0f,0x0,unit_manager.unit_type[target->type_id].name);
			glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_COLOR);
			allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0xFFFFFF,unit_manager.unit_type[target->type_id].name);
			}
		else if(unit[index].planned_weapons>0.0f)
			switch(LANG)
			{
			case LANG_FRENCH:
				glBlendFunc(GL_ZERO,GL_ONE_MINUS_SRC_COLOR);
				allegro_gl_printf(aglfont,174.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0x0,"arme");
				allegro_gl_printf(aglfont,176.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0x0,"arme");
				allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-29.0f,0.0f,0x0,"arme");
				allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-27.0f,0.0f,0x0,"arme");
				glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_COLOR);
				allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0xFFFFFF,"arme");	break;
			case LANG_SPANISH:
				glBlendFunc(GL_ZERO,GL_ONE_MINUS_SRC_COLOR);
				allegro_gl_printf(aglfont,174.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0x0,"arma");
				allegro_gl_printf(aglfont,176.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0x0,"arma");
				allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-29.0f,0.0f,0x0,"arma");
				allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-27.0f,0.0f,0x0,"arma");
				glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_COLOR);
				allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0xFFFFFF,"arma");	break;
			default:
				glBlendFunc(GL_ZERO,GL_ONE_MINUS_SRC_COLOR);
				allegro_gl_printf(aglfont,174.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0x0,"weapon");
				allegro_gl_printf(aglfont,176.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0x0,"weapon");
				allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-29.0f,0.0f,0x0,"weapon");
				allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-27.0f,0.0f,0x0,"weapon");
				glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_COLOR);
				allegro_gl_printf(aglfont,175.0f+(SCREEN_W-200>>1),SCREEN_H-28.0f,0.0f,0xFFFFFF,"weapon");
			};

		int stock=0;
		if(unit_manager.unit_type[unit[index].type_id].weapon[0] && unit_manager.unit_type[unit[index].type_id].weapon[0]->stockpile)	stock=unit[index].stock[0];
		else if(unit_manager.unit_type[unit[index].type_id].weapon[1] && unit_manager.unit_type[unit[index].type_id].weapon[1]->stockpile)	stock=unit[index].stock[1];
		else if(unit_manager.unit_type[unit[index].type_id].weapon[2] && unit_manager.unit_type[unit[index].type_id].weapon[2]->stockpile)	stock=unit[index].stock[2];
		if((unit_manager.unit_type[unit[index].type_id].Builder && unit_manager.unit_type[unit[index].type_id].TEDclass==CLASS_PLANT)
		|| unit[index].planned_weapons>0.0f || stock>0) {		// Affiche la liste de construction
			int page=unit_manager.unit_type[unit[index].type_id].page;

			glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_COLOR);
			for(int i=6*page;i<min(unit_manager.unit_type[unit[index].type_id].nb_unit,6*page+6);i++) {		// Affiche les diffrentes images d'units constructibles
				int nb=0;
				MISSION *m=unit[index].mission;
				while(m) {
					if((m->mission==MISSION_BUILD || m->mission==MISSION_BUILD_2) && m->data==unit_manager.unit_type[unit[index].type_id].BuildList[i])
						nb++;
					m=m->next;
					}
				if(nb>0) {
					char buf[10];
					uszprintf(buf,10,"%d",nb);
					glBlendFunc(GL_ZERO,GL_ONE_MINUS_SRC_COLOR);
					allegro_gl_printf(aglfont,(i&1)*64+31-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+32-0.5f*text_height(font),0.0f,0x0,buf);
					allegro_gl_printf(aglfont,(i&1)*64+33-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+32-0.5f*text_height(font),0.0f,0x0,buf);
					allegro_gl_printf(aglfont,(i&1)*64+32-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+31-0.5f*text_height(font),0.0f,0x0,buf);
					allegro_gl_printf(aglfont,(i&1)*64+32-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+33-0.5f*text_height(font),0.0f,0x0,buf);
					glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_COLOR);
					allegro_gl_printf(aglfont,(i&1)*64+32-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+32-0.5f*text_height(font),0.0f,0xFFFFFF,buf);
					}
				else if(unit_manager.unit_type[unit[index].type_id].BuildList[i]==-1) {		// Il s'agit d'une arme / It's a weapon
					char buf[10];
					if((int)unit[index].planned_weapons==unit[index].planned_weapons)
						uszprintf(buf,10,"%d(%d)",(int)unit[index].planned_weapons,stock);
					else
						uszprintf(buf,10,"%d(%d)",(int)unit[index].planned_weapons+1,stock);
					glBlendFunc(GL_ZERO,GL_ONE_MINUS_SRC_COLOR);
					allegro_gl_printf(aglfont,(i&1)*64+31-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+32-0.5f*text_height(font),0.0f,0x0,buf);
					allegro_gl_printf(aglfont,(i&1)*64+33-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+32-0.5f*text_height(font),0.0f,0x0,buf);
					allegro_gl_printf(aglfont,(i&1)*64+32-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+31-0.5f*text_height(font),0.0f,0x0,buf);
					allegro_gl_printf(aglfont,(i&1)*64+32-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+33-0.5f*text_height(font),0.0f,0x0,buf);
					glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_COLOR);
					allegro_gl_printf(aglfont,(i&1)*64+32-0.5f*text_length(font,buf),128+(i-6*page>>1)*64+32-0.5f*text_height(font),0.0f,0xFFFFFF,buf);
					}
				}
			}

		glDisable(GL_TEXTURE_2D);

		glDisable(GL_BLEND);

		glBegin(GL_QUADS);			// Dessine les barres de metal et d'nergie
			glColor4f(1.0f,0.0f,0.0f,1.0f);

			glVertex2f(150.0f,SCREEN_H-20.0f);					// Fond de la barre de vie de l'unit
			glVertex2f(125+(SCREEN_W-200>>1),SCREEN_H-20.0f);
			glVertex2f(125+(SCREEN_W-200>>1),SCREEN_H-12.0f);
			glVertex2f(150.0f,SCREEN_H-12.0f);

			if(target && (unit[index].mission->flags&4)!=4 && !unit_manager.unit_type[target->type_id].HideDamage) {			// Si l'unit a une cible
				glVertex2f(175+(SCREEN_W-200>>1),SCREEN_H-20.0f);		// Fond de la barre de vie
				glVertex2f(SCREEN_W-50,SCREEN_H-20.0f);
				glVertex2f(SCREEN_W-50,SCREEN_H-12.0f);
				glVertex2f(175+(SCREEN_W-200>>1),SCREEN_H-12.0f);
				}
			else if(unit[index].planned_weapons>0.0f) {
				glVertex2f(175+(SCREEN_W-200>>1),SCREEN_H-20.0f);		// Fond de la barre de vie
				glVertex2f(SCREEN_W-50,SCREEN_H-20.0f);
				glVertex2f(SCREEN_W-50,SCREEN_H-12.0f);
				glVertex2f(175+(SCREEN_W-200>>1),SCREEN_H-12.0f);
				}

			glColor3f(1.0f,1.0f,0.0f);

			if(unit[index].hp>0) {
				glVertex2f(150.0f,SCREEN_H-20.0f);					// Barre de vie
				glVertex2f(150+((SCREEN_W-200>>1)-25)*unit[index].hp/unit_manager.unit_type[unit[index].type_id].MaxDamage,SCREEN_H-20.0f);
				glVertex2f(150+((SCREEN_W-200>>1)-25)*unit[index].hp/unit_manager.unit_type[unit[index].type_id].MaxDamage,SCREEN_H-12.0f);
				glVertex2f(150.0f,SCREEN_H-12.0f);
				}

			if(target && (unit[index].mission->flags&4)!=4 && !unit_manager.unit_type[target->type_id].HideDamage) {			// Si l'unit a une cible
				if(target->hp>0) {
					glVertex2f(175+(SCREEN_W-200>>1),SCREEN_H-20.0f);		// Barre de vie
					glVertex2f(175+(SCREEN_W-200>>1)+((SCREEN_W-200>>1)-25)*target->hp/unit_manager.unit_type[target->type_id].MaxDamage,SCREEN_H-20.0f);
					glVertex2f(175+(SCREEN_W-200>>1)+((SCREEN_W-200>>1)-25)*target->hp/unit_manager.unit_type[target->type_id].MaxDamage,SCREEN_H-12.0f);
					glVertex2f(175+(SCREEN_W-200>>1),SCREEN_H-12.0f);
					}
				}
			else if(unit[index].planned_weapons>0.0f) {						// construit une arme / build a weapon
				float p=1.0f-(unit[index].planned_weapons-(int)unit[index].planned_weapons);
				if(p==1.0f)	p=0.0f;
				glVertex2f(175+(SCREEN_W-200>>1),SCREEN_H-20.0f);
				glVertex2f(175+(SCREEN_W-200>>1)+((SCREEN_W-200>>1)-25)*p,SCREEN_H-20.0f);
				glVertex2f(175+(SCREEN_W-200>>1)+((SCREEN_W-200>>1)-25)*p,SCREEN_H-12.0f);
				glVertex2f(175+(SCREEN_W-200>>1),SCREEN_H-12.0f);
				}

		glEnd();
		glColor4f(1.0f,1.0f,1.0f,1.0f);
	}
};

extern INGAME_UNITS units;

bool can_be_built(VECTOR Pos,MAP *map,int unit_type_id);

#endif						// Fin de la dclaration des classes
