/*  TA3D, a remake of Total Annihilation
    Copyright (C) 2005  Roland BROCHARD

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA*/

/*-----------------------------------------------------------------------------------\
|                                       weapons.h                                    |
|   Ce module contient les structures, classes et fonctions nécessaires à la lecture |
| des fichiers tdf du jeu totalannihilation concernant les armes utilisées par les   |
| unités du jeu.                                                                     |
|                                                                                    |
\-----------------------------------------------------------------------------------*/

#include "gaf.h"

#ifndef __WEAPON_CLASSES
#define __WEAPON_CLASSES

#define RENDER_TYPE_LASER		0x0
#define RENDER_TYPE_MISSILE		0x1
#define RENDER_TYPE_GUN			0x2
#define RENDER_TYPE_DGUN		0x3
#define RENDER_TYPE_BITMAP		0x4
#define RENDER_TYPE_PARTICLES	0x5
#define RENDER_TYPE_BOMB		0x6
#define RENDER_TYPE_LIGHTNING	0x7

class WEAPON_DEF				// Classe définissant un type d'arme
{
public:
	int		weapon_id;			// Numéro identifiant l'arme
	char	*internal_name;		// Nom interne de l'arme
	char	*name;				// Nom de l'arme
	int		rendertype;
	bool	ballistic;
	bool	dropped;
	bool	turret;
	int		range;				// portée
	float	reloadtime;			// temps de rechargement
	float	weaponvelocity;
	int		areaofeffect;		// zone d'effet
	bool	startsmoke;
	bool	endsmoke;
	int		damage;				// Dégats causés par l'arme
	int		firestarter;
	int		accuracy;
	int		aimrate;
	int		tolerance;
	int		holdtime;
	int		energypershot;
	int		metalpershot;
	int		minbarrelangle;
	bool	unitsonly;
	float	edgeeffectiveness;
	bool	lineofsight;
	int		color[2];
	int		burst;
	float	burstrate;
	float	duration;
	bool	beamweapon;
	bool	burnblow;
	float	startvelocity;			// Pour les missiles
	float	weapontimer;			// Pour les missiles
	float	weaponacceleration;		// Pour les missiles
	int		turnrate;				// Pour les missiles
	MODEL	*model;					// Modèle 3D
	float	smokedelay;
	bool	guidance;				// Guidage
	bool	tracks;
	bool	selfprop;
	bool	smoketrail;				// Laisse de la fumée en passant
	bool	noautorange;
	int		flighttime;
	bool	vlaunch;
	bool	stockpile;
	bool	targetable;				// On peut viser
	bool	commandfire;			// ne tire pas seul
	bool	cruise;
	bool	propeller;
	bool	twophase;
	int		shakemagnitude;
	float	shakeduration;
	char	*soundstart;			// Son à jouer lorsque l'arme tire
	char	*soundhit;				// Son à jouer lorsque l'arme explose (si l'arme explose)
	char	*soundwater;			// Son à jouer lorsque l'arme touche l'eau
	char	*soundtrigger;			// Son à jouer lorsque l'arme tire en mode rapide
	char	*explosiongaf;
	char	*explosionart;
	char	*waterexplosiongaf;
	char	*waterexplosionart;
	char	*lavaexplosiongaf;
	char	*lavaexplosionart;
	int		nb_id;
	bool	waterweapon;
	int		metal;
	int		energy;
	bool	interceptor;			// Prend pour cible des armes / Target weapons only
	float	coverage;				// Zone de protection

	inline void init()
	{
		burnblow=false;
		coverage=0.0f;
		interceptor=false;
		metal=0;
		energy=0;
		waterweapon=false;
		explosiongaf=NULL;
		explosionart=NULL;
		waterexplosiongaf=NULL;
		waterexplosionart=NULL;
		lavaexplosiongaf=NULL;
		lavaexplosionart=NULL;
		dropped=false;
		nb_id=0;
		soundstart=NULL;
		soundhit=NULL;
		soundwater=NULL;
		soundtrigger=NULL;
		burst=1;
		weapon_id=0;			// Numéro identifiant l'arme
		internal_name=NULL;		// Nom interne de l'arme
		name=NULL;				// Nom de l'arme
		rendertype=0;
		ballistic=false;
		turret=false;
		range=100;				// portée
		reloadtime=0.0f;		// temps de rechargement
		weaponvelocity=0.0f;
		areaofeffect=10;		// zone d'effet
		startsmoke=false;
		endsmoke=false;
		damage=100;				// Dégats causés par l'arme
		firestarter=0;
		accuracy=0;
		aimrate=0;
		tolerance=0;
		holdtime=0;
		energypershot=0;
		metalpershot=0;
		minbarrelangle=0;
		unitsonly=false;
		edgeeffectiveness=1.0f;
		lineofsight=false;
		color[0]=0;
		color[1]=0xFFFFFF;
		burstrate=1.0f;
		duration=-1.0f;
		beamweapon=false;
		startvelocity=0;			// Pour les missiles
		weapontimer=0.0f;			// Pour les missiles
		weaponacceleration=0.0f;	// Pour les missiles
		turnrate=1;					// Pour les missiles
		model=NULL;					// Modèle 3D
		smokedelay=0.1f;
		guidance=false;				// Guidage
		tracks=false;
		selfprop=false;
		smoketrail=false;			// Laisse de la fumée en passant
		noautorange=false;
		flighttime=10;
		vlaunch=false;
		stockpile=false;			// Nécessite de fabriquer des munitions / need to make ammo??
		targetable=false;			// On peut viser
		commandfire=false;			// ne tire pas seul
		cruise=false;
		propeller=false;
		twophase=false;
		shakemagnitude=0;
		shakeduration=0.1f;
	}

	inline void destroy()
	{
		if(internal_name)	free(internal_name);
		if(name)	free(name);
		init();
	}

	WEAPON_DEF()
	{
		init();
	}

	~WEAPON_DEF()
	{
		destroy();
	}
};

class WEAPON_MANAGER			// Gestionnaire de types d'armes
{
public:
	int			nb_weapons;
	WEAPON_DEF	*weapon;
	ANIM		cannonshell;	// Animation pour les tirs de cannons

	void init()
	{
		nb_weapons=0;
		weapon=NULL;
		cannonshell.init();
	}

	WEAPON_MANAGER()
	{
		init();
	}

	void destroy()
	{
		cannonshell.destroy();
		if(nb_weapons>0 && weapon)			// Détruit les éléments
			for(int i=0;i<nb_weapons;i++)
				weapon[i].destroy();
		if(weapon)
			free(weapon);
		init();
	}

	~WEAPON_MANAGER()
	{
		destroy();
	}

	int add_weapon(char *name)			// Ajoute un élément
	{
		nb_weapons++;
		WEAPON_DEF *n_weapon=(WEAPON_DEF*) malloc(sizeof(WEAPON_DEF)*nb_weapons);
		if(weapon && nb_weapons>1)
			for(int i=0;i<nb_weapons-1;i++)
				n_weapon[i]=weapon[i];
		if(weapon)	free(weapon);
		weapon=n_weapon;
		weapon[nb_weapons-1].init();
		weapon[nb_weapons-1].internal_name=strdup(name);
		weapon[nb_weapons-1].nb_id=nb_weapons-1;
		return nb_weapons-1;
	}

private:
	inline char *get_line(char *data)
	{
		int pos=0;
		while(data[pos]!=0 && data[pos]!=13 && data[pos]!=10)	pos++;
		char *d=new char[pos+1];
		memcpy(d,data,pos);
		d[pos]=0;
		return d;
	}
public:

	void load_tdf(char *data,int size=99999999);					// Charge un fichier tdf

	int get_weapon_index(char *name)
	{
		if(name==NULL)	return -1;
		if(nb_weapons<=0)	return -1;
		for(int i=0;i<nb_weapons;i++)
			if(strcasecmp(name,weapon[i].internal_name)==0)
				return i;
		return -1;
	}

	int get_weapon_index(int weapon_id)
	{
		if(nb_weapons<=0)	return -1;
		for(int i=0;i<nb_weapons;i++)
			if(weapon[i].weapon_id==weapon_id)
				return i;
		return -1;
	}
};

extern WEAPON_MANAGER		weapon_manager;

void load_all_weapons_in_hpi(char *filename);

void load_weapons();				// Charge tout les éléments

class WEAPON						// Objet arme utilisé pendant le jeu
{
public:
	int			weapon_id;			// Identifiant de l'arme
	VECTOR		Pos;				// Position
	VECTOR		V;					// Vitesse
	VECTOR		Ac;					// Accélération
	VECTOR		target_pos;			// Position ciblée
	int			target;				// Unité ciblée (dans le tableau des unités)
	float		stime;				// Temps écoulé depuis le lancement
	float		killtime;			// Temps écoulé depuis la destruction de l'arme
	bool		dying;
	float		smoke_time;			// Temps écoulé depuis la dernière émission de fumée
	float		f_time;				// Temps de vol
	float		a_time;				// Temps d'activité
	int			anim_sprite;		// Position dans l'animation
	int			shooter_idx;		// Unité qui a tiré l'arme (ne peut pas se tirer dessus)
	int			phase;
	byte		owner;
	int			idx;

	inline void init()
	{
		idx=0;
		phase=1;
		a_time=0.0f;
		f_time=0.0f;
		shooter_idx=-1;
		anim_sprite=0;
		weapon_id=-1;		// Non défini
		Pos.x=Pos.y=Pos.z=0.0f;
		Ac=V=Pos;
		target_pos=Pos;
		target=-1;			// Pas de cible
		stime=0.0f;
		killtime=0.0f;
		dying=false;
		smoke_time=0.0f;
		owner=0;
	}

	WEAPON()
	{
		init();
	}

	void move(float dt,MAP *map);

	void draw(CAMERA *cam=NULL,MAP *map=NULL);
};

class INGAME_WEAPONS				// Gestionnaire d'armes
{
public:
	int			nb_weapon;			// Nombre d'armes
	int			max_weapon;			// Nombre maximum d'armes stockables dans le tableau
	WEAPON		*weapon;			// Tableau regroupant les armes
	ANIM		nuclogo;			// Logos des armes atomiques sur la minicarte / Logo of nuclear weapons on minimap

	void init(bool real=true)
	{
		nb_weapon=0;
		max_weapon=0;
		weapon=NULL;
		nuclogo.init();
		if(real) {
			byte *data=load_file("/anims/fx.gaf");
			if(data) {
				nuclogo.load_gaf(data,get_gaf_entry_index(data,"nuclogo"));
				nuclogo.convert();
				nuclogo.clean();
				free(data);
				}
			}
	}

	void destroy()
	{
		if(max_weapon>0 && weapon)
			free(weapon);
		nuclogo.destroy();
		init(false);
	}

	INGAME_WEAPONS()
	{
		init(false);
	}

	~INGAME_WEAPONS()
	{
		destroy();
	}

	int add_weapon(int weapon_id,int shooter)
	{
		if(weapon_id<0)	return -1;
		if(nb_weapon<max_weapon) {		// S'il y a encore de la place
			for(int i=0;i<max_weapon;i++)
				if(weapon[i].weapon_id==-1) {
					nb_weapon++;
					weapon[i].init();
					weapon[i].weapon_id=weapon_id;
					weapon[i].shooter_idx=shooter;
					weapon[i].idx=i;
					weapon[i].f_time=weapon_manager.weapon[weapon_id].flighttime;
					return i;
					}
			}
		else {
			max_weapon+=100;			// Augmente la taille du tableau
			WEAPON *new_weapon=(WEAPON*) malloc(sizeof(WEAPON)*max_weapon);
			for(int i=0;i<max_weapon;i++)
				new_weapon[i].weapon_id=-1;
			if(weapon && nb_weapon>0)
				for(int i=0;i<nb_weapon;i++)
					new_weapon[i]=weapon[i];
			if(weapon)
				free(weapon);
			weapon=new_weapon;
			int index=nb_weapon;
			nb_weapon++;
			weapon[index].init();
			weapon[index].weapon_id=weapon_id;
			weapon[index].shooter_idx=shooter;
			weapon[index].idx=index;
			weapon[index].f_time=weapon_manager.weapon[weapon_id].flighttime;
			return index;
			}
	}

	void move(float dt,MAP *map)
	{
		if(nb_weapon<=0 || max_weapon<=0) return;
		for(int i=0;i<max_weapon;i++)
			if(weapon[i].weapon_id>=0)
				weapon[i].move(dt,map);
	}

	void draw(CAMERA *cam=NULL,MAP *map=NULL,bool underwater=false)
	{
		if(nb_weapon<=0 || max_weapon<=0) return;
		for(int i=0;i<max_weapon;i++)
			if(weapon[i].weapon_id>=0 && ((weapon[i].Pos.y<map->sealvl && underwater) || (weapon[i].Pos.y>=map->sealvl && !underwater)))
				weapon[i].draw(cam,map);
	}

	inline void draw_mini(float map_w,float map_h,int mini_w,int mini_h)				// Repère les unités sur la mini-carte
	{
		if(nb_weapon<=0 || max_weapon<=0)	return;		// Pas d'unités à dessiner

		float rw=128.0f*mini_w/252;
		float rh=128.0f*mini_h/252;

		glColor4f(1.0f,1.0f,1.0f,1.0f);
		glDisable(GL_TEXTURE_2D);
		glBegin(GL_POINTS);
		int cur_id=0;
		for(int i=0;i<max_weapon;i++)
			if(weapon[i].weapon_id>=0) {
				if(weapon_manager.weapon[weapon[i].weapon_id].cruise || weapon_manager.weapon[weapon[i].weapon_id].interceptor) {
					glEnd();
					glEnable(GL_TEXTURE_2D);
					glEnable(GL_BLEND);
					glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
					int idx=weapon[i].owner;
					PutTex(nuclogo.glbmp[idx],weapon[i].Pos.x/map_w*rw+64.0f-nuclogo.ofs_x[idx],weapon[i].Pos.z/map_h*rh+64.0f-nuclogo.ofs_y[idx],weapon[i].Pos.x/map_w*rw+63.0f-nuclogo.ofs_x[idx]+nuclogo.w[idx],weapon[i].Pos.z/map_h*rh+63.0f-nuclogo.ofs_y[idx]+nuclogo.h[idx]);
					glDisable(GL_BLEND);
					glDisable(GL_TEXTURE_2D);
					glBegin(GL_POINTS);
					}
				else
					glVertex2f(weapon[i].Pos.x/map_w*rw+64.0f,weapon[i].Pos.z/map_h*rh+64.0f);
				}
		glEnd();
		glEnable(GL_TEXTURE_2D);
	}
};

extern INGAME_WEAPONS weapons;

class FX
{
public:
	float		time;		// Durée d'éxistence de l'effet
	bool		playing;	// Si l'effet est joué
	VECTOR		Pos;		// Position
	float		size;		// Taille (proportion de l'image d'origine)
	int			anm;		// Animation

public:
	inline void init()
	{
		time=0.0f;
		playing=false;
		Pos.x=Pos.y=Pos.z=0.0f;
		size=1.0f;
		anm=0;
	}

	inline void destroy()
	{
		init();
	}

	FX()
	{
		init();
	}

	~FX()
	{
		destroy();
	}

	inline bool move(float dt,ANIM **anims)
	{
		if(!playing)	return false;
		time+=dt;
		if(time*15.0f>=anims[anm]->nb_bmp) {
			playing=false;
			return true;
			}
		return false;
	}

	inline void draw(CAMERA *cam, MAP *map, ANIM **anims)
	{
		if(!playing)	return;
		if(map!=NULL) {
			int px=(int)(Pos.x+map->map_w*0.5f)>>4;
			int py=(int)(Pos.z+map->map_h*0.5f)>>4;
			if(px<0 || py<0 || px>=map->bloc_w || py>=map->bloc_h)	return;
			if(map->view[py][px]!=1
			|| (!map->map_data[py<<1][px<<1].timer && !map->map_data[py<<1][(px<<1)+1].timer && !map->map_data[(py<<1)+1][px<<1].timer && !map->map_data[(py<<1)+1][(px<<1)+1].timer))	return;
			}
		if(anims[anm]==NULL) {
			playing=false;
			return;
			}
		int img=(int)(time*15.0f);
		float wsize=size*anims[anm]->w[img];
		float hsize=size*anims[anm]->h[img];
		float dx=anims[anm]->ofs_x[img];
		float dy=anims[anm]->ofs_y[img];
		VECTOR P=Pos;
		glBindTexture(GL_TEXTURE_2D,anims[anm]->glbmp[img]);
		float hux=hsize*cam->Up.x;
		float wsx=wsize*cam->Side.x;
		float huy=hsize*cam->Up.y;
		float wsy=wsize*cam->Side.y;
		float huz=hsize*cam->Up.z;
		float wsz=wsize*cam->Side.z;
		if(cam->mirror) {
			glBegin(GL_QUADS);
				glTexCoord2f(0.0f,0.0f);	glVertex3f(P.x-hux-wsx,P.y+huy+wsy,P.z-huz-wsz);
				glTexCoord2f(1.0f,0.0f);	glVertex3f(P.x-hux+wsx,P.y+huy-wsy,P.z-huz+wsz);
				glTexCoord2f(1.0f,1.0f);	glVertex3f(P.x+hux+wsx,P.y-huy-wsy,P.z+huz+wsz);
				glTexCoord2f(0.0f,1.0f);	glVertex3f(P.x+hux-wsx,P.y-huy+wsy,P.z+huz-wsz);
			glEnd();
/*			glBegin(GL_QUADS);
				glTexCoord2f(0.0f,0.0f);	glVertex3f(P.x-hsize*cam->Up.x-wsize*cam->Side.x,P.y+hsize*cam->Up.y+wsize*cam->Side.y,P.z-hsize*cam->Up.z-wsize*cam->Side.z);
				glTexCoord2f(1.0f,0.0f);	glVertex3f(P.x-hsize*cam->Up.x+wsize*cam->Side.x,P.y+hsize*cam->Up.y-wsize*cam->Side.y,P.z-hsize*cam->Up.z+wsize*cam->Side.z);
				glTexCoord2f(1.0f,1.0f);	glVertex3f(P.x+hsize*cam->Up.x+wsize*cam->Side.x,P.y-hsize*cam->Up.y-wsize*cam->Side.y,P.z+hsize*cam->Up.z+wsize*cam->Side.z);
				glTexCoord2f(0.0f,1.0f);	glVertex3f(P.x+hsize*cam->Up.x-wsize*cam->Side.x,P.y-hsize*cam->Up.y+wsize*cam->Side.y,P.z+hsize*cam->Up.z-wsize*cam->Side.z);
			glEnd();*/
			}
		else {
			glBegin(GL_QUADS);
				glTexCoord2f(0.0f,0.0f);	glVertex3f(P.x-hux-wsx,P.y-huy-wsy,P.z-huz-wsz);
				glTexCoord2f(1.0f,0.0f);	glVertex3f(P.x-hux+wsx,P.y-huy+wsy,P.z-huz+wsz);
				glTexCoord2f(1.0f,1.0f);	glVertex3f(P.x+hux+wsx,P.y+huy+wsy,P.z+huz+wsz);
				glTexCoord2f(0.0f,1.0f);	glVertex3f(P.x+hux-wsx,P.y+huy-wsy,P.z+huz-wsz);
			glEnd();
/*			glBegin(GL_QUADS);
				glTexCoord2f(0.0f,0.0f);	glVertex3f(P.x-hsize*cam->Up.x-wsize*cam->Side.x,P.y-hsize*cam->Up.y-wsize*cam->Side.y,P.z-hsize*cam->Up.z-wsize*cam->Side.z);
				glTexCoord2f(1.0f,0.0f);	glVertex3f(P.x-hsize*cam->Up.x+wsize*cam->Side.x,P.y-hsize*cam->Up.y+wsize*cam->Side.y,P.z-hsize*cam->Up.z+wsize*cam->Side.z);
				glTexCoord2f(1.0f,1.0f);	glVertex3f(P.x+hsize*cam->Up.x+wsize*cam->Side.x,P.y+hsize*cam->Up.y+wsize*cam->Side.y,P.z+hsize*cam->Up.z+wsize*cam->Side.z);
				glTexCoord2f(0.0f,1.0f);	glVertex3f(P.x+hsize*cam->Up.x-wsize*cam->Side.x,P.y+hsize*cam->Up.y-wsize*cam->Side.y,P.z+hsize*cam->Up.z-wsize*cam->Side.z);
			glEnd();*/
			}
	}

	inline void load(int anim,VECTOR P,float s)
	{
		destroy();

		anm=anim;
		Pos=P;
		size=s*0.25f;
		time=0.0f;
		playing=true;
	}
};

class FX_MANAGER
{
private:
	int			max_fx;
	int			nb_fx;
	FX			*fx;

	int			cache_size;			// Cache
	int			max_cache_size;
	char		**cache_name;
	ANIM		**cache_anm;
	int			*use;

public:
	inline void init()
	{
		max_fx=0;
		nb_fx=0;
		fx=NULL;

		max_cache_size=0;
		cache_size=0;
		cache_name=NULL;
		cache_anm=NULL;
		use=NULL;
	}

	inline void destroy()
	{
		if(fx) {
			for(int i=0;i<max_fx;i++)
				fx[i].destroy();
			free(fx);
			}
		if(cache_size>0) {
			if(cache_name) {
				for(int i=0;i<max_cache_size;i++)
					if(cache_name[i])
						free(cache_name[i]);
				free(cache_name);
				}
			if(cache_anm) {
				for(int i=0;i<max_cache_size;i++)
					if(cache_anm[i]) {
						cache_anm[i]->destroy();
						delete cache_anm[i];
						}
				free(cache_anm);
				}
			}
		init();
	}

	FX_MANAGER()
	{
		init();
	}

	~FX_MANAGER()
	{
		destroy();
	}

	int is_in_cache(char *filename)
	{
		if(cache_size<=0)	return -1;
		for(int i=0;i<max_cache_size;i++)
			if(cache_anm[i]!=NULL && cache_name[i]!=NULL)
				if(strcasecmp(filename,cache_name[i])==0)
					return i;
		return -1;
	}

	int put_in_cache(char *filename,ANIM *anm)
	{
		int is_in=is_in_cache(filename);
		if(is_in>=0)	return is_in;		// On ne le garde pas 2 fois
		int idx=-1;
		if(cache_size+1>max_cache_size) {
			max_cache_size+=100;
			char **n_name=(char**)malloc(sizeof(char*)*max_cache_size);
			ANIM **n_anm=(ANIM**)malloc(sizeof(ANIM*)*max_cache_size);
			int *n_use=(int*)malloc(sizeof(int)*max_cache_size);
			for(int i=max_cache_size-100;i<max_cache_size;i++) {
				n_use[i]=0;
				n_name[i]=NULL;
				n_anm[i]=NULL;
				}
			if(cache_size>0) {
				for(int i=0;i<max_cache_size-100;i++) {
					n_name[i]=cache_name[i];
					n_anm[i]=cache_anm[i];
					n_use[i]=use[i];
					}
				free(cache_name);
				free(cache_anm);
				free(use);
				}
			use=n_use;
			cache_name=n_name;
			cache_anm=n_anm;
			idx=cache_size;
			}
		else {
			idx=0;
			for(int i=0;i<max_cache_size;i++)
				if(cache_anm[i]==NULL)
					idx=i;
			}
		use[idx]=1;
		cache_anm[idx]=anm;
		cache_name[idx]=strdup(filename);
		cache_size++;
		return idx;
	}

	inline void move(float dt)
	{
		for(int i=0;i<max_fx;i++)
			if(fx[i].move(dt,cache_anm)) {
				use[fx[i].anm]--;
				nb_fx--;
				if(use[fx[i].anm]<=0) {		// Animation inutilisée
					free(cache_name[fx[i].anm]);		cache_name[fx[i].anm]=NULL;
					cache_anm[fx[i].anm]->destroy();	delete cache_anm[fx[i].anm];	cache_anm[fx[i].anm]=NULL;
					cache_size--;
					}
				}
	}

	inline void draw(CAMERA *cam, MAP *map, float w_lvl=0.0f, bool UW=false)
	{
		glEnable(GL_TEXTURE_2D);
		glDisable(GL_LIGHTING);
		glDisable(GL_CULL_FACE);
		glEnable(GL_BLEND);
		glDepthMask(GL_FALSE);
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
		cam->SetView();
		glEnable(GL_POLYGON_OFFSET_FILL);
		glPolygonOffset(0.0f,-1600.0f);
		if(UW)
			for(int i=0;i<max_fx;i++)
				if(fx[i].Pos.y<w_lvl)
					fx[i].draw(cam,map,cache_anm);
		else
			for(int i=0;i<max_fx;i++)
				if(fx[i].Pos.y>=w_lvl)
					fx[i].draw(cam,map,cache_anm);
		glDisable(GL_POLYGON_OFFSET_FILL);
		glPolygonOffset(0.0f,0.0f);
		glDisable(GL_BLEND);
		glDepthMask(GL_TRUE);
	}

	int add(char *filename,char *entry_name,VECTOR Pos,float size);
};

extern FX_MANAGER	fx_manager;
#endif
