/*  TA3D, a remake of Total Annihilation
    Copyright (C) 2005  Roland BROCHARD

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA*/

/*-----------------------------------------------------------------------------\
|                                      ia.h                                    |
|       Ce module est responsable de l'intelligence artificielle               |
|                                                                              |
\-----------------------------------------------------------------------------*/

#include <string.h>
#include <stdio.h>
#include <math.h>

#ifndef ARTIFICIAL_INTELLIGENCE
#define ARTIFICIAL_INTELLIGENCE

struct NEURON
{
	float	var;			// Variable pour les opérations
	float	*weight;		// Poids des différents neurones sources
};

class BRAIN		// Réseau de NEURON à n entrées et p sorties
{
public:
	int		nb_neuron;		// Nombre de NEURON
	NEURON	*neuron;		// Tableau de NEURON
	int		n;				// Nombre d'entrées dans le réseau
	int		p;				// Nombre de sorties du réseau
	int		q;				// Taille des rangs
	float	*n_out;			// Tableau de sortie

	inline void init()
	{
		nb_neuron=0;
		neuron=NULL;
		n=p=q=0;
		n_out=NULL;
	}

	inline void destroy()
	{
		if(nb_neuron>0 && neuron) {
			for(int i=0;i<nb_neuron;i++)
				if(neuron[i].weight!=NULL)
					free(neuron[i].weight);
			free(neuron);
			}
		if(n_out)	free(n_out);
		init();
	}

	BRAIN()
	{
		init();
	}

	~BRAIN()
	{
		destroy();
	}

	void build(int nb_in,int nb_out,int rg)				// Crée le réseau de neurones
	{
		destroy();
		q=rg;
		nb_neuron=q+nb_in+nb_out;		// Nombre de couches x nombre d'entrées + nombre de sorties
		n=nb_in;
		p=nb_out;
		neuron=(NEURON*) malloc(sizeof(NEURON)*nb_neuron);
		n_out=(float*) malloc(sizeof(float)*nb_out);
		for(int i=0;i<nb_neuron;i++) {
			neuron[i].var=0.0f;
			neuron[i].weight=NULL;
			if(i<nb_out)
				n_out[i]=0.0f;
			if(i>=n && i<nb_neuron-p) {
				neuron[i].weight=(float*) malloc(sizeof(float)*n);
				for(int e=0;e<n;e++)
					neuron[i].weight[e]=(rand()%2001)*0.001f-1.0f;
				}
			else if(i>=n) {
				neuron[i].weight=(float*) malloc(sizeof(float)*q);
				for(int e=0;e<q;e++)
					neuron[i].weight[e]=(rand()%2001)*0.001f-1.0f;
				}
			}
	}

	inline void active_neuron(int i)
	{
		if(neuron[i].weight==NULL)	return;
		if(i<n)
			return;
		neuron[i].var=0.0f;
		if(i<nb_neuron-p)
			for(int e=0;e<n;e++)
				neuron[i].var+=neuron[e].var*neuron[i].weight[e];
		else
			for(int e=0;e<q;e++)
				neuron[i].var+=neuron[n+e].var*neuron[i].weight[e];
		neuron[i].var=1.0f/(1.0f+exp(-neuron[i].var));
	}

	inline float *work(float entry[],bool seuil=false)			// Fait bosser un peu le réseau de NEURON et renvoie les valeurs calculées par un NEURON
	{
		if(nb_neuron<0)	return NULL;		// Pas de NEURON à faire bosser
		int i;
		for(i=0;i<n;i++)		// Prépare le réseau au calcul
			neuron[i].var=entry[i];
		for(i=n;i<nb_neuron;i++)
			active_neuron(i);
		if(!seuil)
			for(i=0;i<p;i++)		// Récupère le résultat du calcul
				n_out[i]=neuron[n+q+i].var;
		else
			for(i=0;i<p;i++)		// Récupère le résultat du calcul
				n_out[i]=neuron[n+q+i].var>=0.5f ? 1.0f : 0.0f;
		return n_out;
	}

	inline void mutation()			// Déclenche une mutation dans le réseau de neurones
	{
		int index=(rand()%(nb_neuron-n))+n;
		int mod_w=0;
		if(index<nb_neuron-p)	mod_w=rand()%n;
		else mod_w=rand()%q;
		neuron[index].weight[mod_w]+=((rand()%200001)-100000)*0.00001f;
	}

	inline void learn(float *result,float coef=1.0f)		// Corrige les défauts
	{
		for(int i=0;i<p;i++)
			n_out[i]=(result[i]-n_out[i])*(n_out[i]+0.01f)*(1.01f-n_out[i]);

		float diff[q];
		for(int i=0;i<q;i++)
			diff[i]=0.0f;

		for(int i=0;i<p;i++)		// Neurones de sortie
			for(int e=0;e<q;e++) {
				diff[e]+=(n_out[i]+0.01f)*(1.01f-n_out[i])*neuron[n+q+i].weight[e]*neuron[n+e].var;
				neuron[n+q+i].weight[e]+=coef*n_out[i]*neuron[n+e].var;
				}

									// Neurones des couches intermédiaires
		for(int i=0;i<q;i++)
			for(int e=0;e<n;e++)
				neuron[n+i].weight[e]+=coef*diff[i]*neuron[e].var;
	}

	void save(FILE *file);		// Enregistre le réseau de neurones

	int load(FILE *file);		// Charge le réseau de neurones
};

BRAIN *copy_brain(BRAIN *brain,BRAIN *dst=NULL);		// Copie un réseau de neurones

#define ORDER_ATTACK		0x0			// Ordre d'attaquer une zone
#define ORDER_MOVE			0x1			// Ordre de déplacer des troupes
#define ORDER_RECLAIM		0x2			// Ordre de récupérer des cadavres
#define ORDER_BUILD			0x3			// Ordre de construire des bâtiments
#define ORDER_PATROL		0x4			// Ordre de patrouiller(surveiller une zone)
#define ORDER_SPY			0x5			// Ordre d'espionner une zone
#define ORDER_BUILD_ARMY	0x6			// Ordre de construire une armée
#define ORDER_RESSOURCE		0x7			// Ordre de récolter des ressources
#define ORDER_EXPLORE		0x8			// Ordre d'explorer la carte (pas assez d'infos...)

#define NB_ORDERS			0x9

#define BRAIN_VALUE_NULL	0x0
#define BRAIN_VALUE_LOW		0x1
#define BRAIN_VALUE_MEDIUM	0x2
#define BRAIN_VALUE_HIGH	0x4
#define BRAIN_VALUE_MAX		0x8

#define BRAIN_VALUE_BITS	0x4			// Nombre de bits nécessaires pour coder une valeur pour un réseau de neurones

struct LIST			// Liste de données
{
	int		data;
	LIST	*next;
};

class S_ORDER		// Ordre pour l'IA
{
public:
	byte	type;			// Type d'ordre
	VECTOR	pos;			// Position de la zone ciblée
	float	size;			// Taille de la zone ciblée
	LIST	*list;			// Liste de données supplémentaires (liste d'unités par exemple)
	S_ORDER	*next;			// Ordre suivant

	S_ORDER()
	{
		type=ORDER_ATTACK;
		size=0.0f;
		list=NULL;
		next=NULL;
		pos.x=pos.y=pos.z=0.0f;
	}
};

class STRATEGY					// Classe pour la gestion des stratégies de l'ordinateur
{
public:
	S_ORDER	*orders;		// Ordres en cours de validité
	S_ORDER *last;

	inline void init()
	{
		orders=NULL;
		last=NULL;
	}

	void destroy()
	{
		S_ORDER *cur=orders;
		S_ORDER *tmp;

		while(cur) {
			if(cur->list) {
				LIST *c_list=cur->list;
				LIST *t_list;
				while(c_list) {
					t_list=c_list;
					c_list=c_list->next;
					delete t_list;
					}
				}
			tmp=cur;
			cur=cur->next;
			delete tmp;
			}

		init();
	}

	STRATEGY()
	{
		init();
	}

	~STRATEGY()
	{
		destroy();
	}

	inline void add_order(S_ORDER *order)				// Pile FIFO
	{
		if(orders==NULL) {
			orders=order;
			last=order;
			return;
			}
		order->next=NULL;
		last->next=order;
		last=order;
	}

	inline int next_order()
	{
		if(orders==NULL)	return 1;		// Si on a atteint la fin de la liste
		S_ORDER *old=orders;
		orders=orders->next;
		LIST *l=old->list;
		while(l) {
			LIST *ol=l;
			l=l->next;
			delete ol;
			}
		delete old;
		if(orders==NULL)
			last=NULL;
		return 0;
	}
};

#define ZONE_METAL			0x0		// Cette zone contient du métal
#define ZONE_ENERGY			0x1		// Cette zone contient des ressources énergétiques
#define ZONE_ENEMY			0x2		// Cette zone est controlée par l'ennemi
#define ZONE_VITAL			0x3		// Cette zone est un point de controle stratégique
#define ZONE_UNKNOWN		0x4		// Cette zone est inexplorée
#define ZONE_CAMP			0X5		// Cette zone est un camp du joueur IA

#define UNIT_TYPE_NONE		0x0		// Aucune
#define UNIT_TYPE_TANK		0x1		// Tank
#define UNIT_TYPE_KBOT		0x2		// Kbot
#define UNIT_TYPE_VTOL		0x4		// Avions
#define UNIT_TYPE_SHIP		0x8		// Bateau
#define UNIT_TYPE_PLANT		0x10	// Bâtiment
#define UNIT_TYPE_UNKN		0x20	// Inconnu

struct ZONE						// Structure définissant les zones de la cartes
{
	VECTOR	Pos;
	float	size;
	byte	type;
	byte	unit_type;			// Types d'unités présents dans la zone
	byte	brain_value;		// Variable quantitative pour réseau de neurones
};

class S_DATA					// Classe pour mémoriser les informations stratégiques sur la partie
{
public:
	int		max_data;		// Quantité de mémoire allouée pour le stockage d'éléments
	int		nb_zone;		// Nombre de zones
	ZONE	*zone;			// Zones stratégiques

	inline void show_zones(float map_w,float map_h,float mini_w,float mini_h)	// Affiche les zones sur la mini-map (pour le débogage pas pour tricher)
	{
		if(nb_zone<=0)	return;

		float rw=128.0f*mini_w/252/map_w;
		float rh=128.0f*mini_h/252/map_h;
		glColor4f(1.0f,1.0f,1.0f,1.0f);
		glDisable(GL_BLEND);
		glDisable(GL_TEXTURE_2D);

		for(int i=0;i<nb_zone;i++) {
			int c=0;
			switch(zone[i].type)
			{
			case ZONE_METAL:
				c=makecol(0x7F,0x7F,0x7F);
				break;
			case ZONE_ENERGY:
				c=makecol(0xFF,0xFF,0x0);
				break;
			case ZONE_ENEMY:
				c=makecol(0xFF,0x0,0x0);
				break;
			case ZONE_VITAL:
				c=makecol(0x0,0x0,0xFF);
				break;
			case ZONE_UNKNOWN:
				c=makecol(0x0,0x0,0x0);
				break;
			case ZONE_CAMP:
				c=makecol(0xFF,0xFF,0xFF);
				break;
			};
			circle(screen,(int)(zone[i].Pos.x*rw+64.0f),(int)(zone[i].Pos.z*rh+64.0f),(int)(zone[i].size*rw),c);
			}
		glEnable(GL_TEXTURE_2D);
	}

	inline int min_zone_number()		// bilan de la situation
	{
		int nb[6];
		for(int i=0;i<6;i++)	nb[i]=0;
		for(int i=0;i<nb_zone;i++)
			nb[zone[i].type]++;
		int r=nb_zone;
		for(int i=0;i<6;i++)
			if(nb[i]<r)	r=nb[i];
		return r;
	}

	inline int enemy_zone_number()		// bilan de la situation (ennemis)
	{
		int nb=0;
		int n=0;
		for(int i=0;i<nb_zone;i++)
			if(zone[i].type==ZONE_ENEMY) {
				nb+=zone[i].brain_value;
				n++;
				}
		if(n>0)
			nb=(nb+(n>>1))/n;
		return nb;
	}

	inline int energy_zone_number()		// bilan de la situation (énergie)
	{
		int nb=0;
		int n=0;
		for(int i=0;i<nb_zone;i++)
			if(zone[i].type==ZONE_ENERGY) {
				nb+=zone[i].brain_value;
				n++;
				}
		if(n>0)
			nb=(nb+(n>>1))/n;
		return nb;
	}

	inline int metal_zone_number()		// bilan de la situation (métal)
	{
		int nb=0;
		int n=0;
		for(int i=0;i<nb_zone;i++)
			if(zone[i].type==ZONE_METAL) {
				nb+=zone[i].brain_value;
				n++;
				}
		if(n>0)
			nb=(nb+(n>>1))/n;
		return nb;
	}

	inline void init()
	{
		max_data=0;
		nb_zone=0;
		zone=NULL;
	}

	inline void destroy()
	{
		if(max_data>0 && zone!=NULL)
			free(zone);
		init();
	}

	inline void add_zone(ZONE z)
	{
		if(nb_zone==max_data) {
			max_data+=100;
			ZONE *n_zone=(ZONE*) malloc(sizeof(ZONE)*max_data);
			if(zone) {
				for(int i=0;i<nb_zone;i++)
					n_zone[i]=zone[i];
				free(zone);
				}
			zone=n_zone;
			}
		zone[nb_zone++]=z;
	}

	inline void del_zone(int idx)
	{
		if(idx<0 || idx>=nb_zone)	return;

		nb_zone--;
		for(int i=idx;i<nb_zone;i++)
			zone[i]=zone[i+1];
	}

	inline bool fusion_zone(ZONE z)		// Met à jour la base de données
	{
		int near_idx=-1;
		float best=999999999.0f;
		bool mod=false;
		for(int i=0;i<nb_zone;i++) {			// Fusionne avec la meilleure zone qui recoupe z
			if(zone[i].type!=z.type)	continue;
			float m=z.size+zone[i].size;
			float d=(zone[i].Pos-z.Pos).Norm();
			if(d+z.size<=zone[i].size || (z.size<=zone[i].size && d<zone[i].size))
				return false;
			if(d<zone[i].size*0.5f && zone[i].size<=z.size) {
				mod=true;
				del_zone(i);
				i--;
				continue;
				}
			if(d<=m*0.72f && d<best && m<=11.2f) {
				best=d;
				near_idx=i;
				}
			}

		if(near_idx<0) {
			add_zone(z);
			return true;
			}

		float d=best;
		if(d+z.size>zone[near_idx].size) {
			VECTOR P=zone[near_idx].Pos;
			float size=zone[near_idx].size;
			zone[near_idx]=z;
			zone[near_idx].Pos=0.5f*(P+z.Pos);
			zone[near_idx].size=d+max(size,z.size);
			return true;
			}
		if(zone[near_idx].type!=z.type || zone[near_idx].unit_type!=z.unit_type || zone[near_idx].brain_value!=z.brain_value) {
			zone[near_idx].type=z.type;
			zone[near_idx].unit_type=z.unit_type;
			zone[near_idx].brain_value=z.brain_value;
			return true;
			}
		else
			return mod;
	}

	inline bool remove_data(ZONE z)			// Met à jour les données (efface ce qui n'est plus vrai)
	{
		int near_idx=-1;
		float best=999999999.0f;
		for(int i=0;i<nb_zone;i++) {			// Fusionne avec la meilleure zone qui recoupe z
			if(zone[i].type!=z.type)	continue;
			float m=z.size+zone[i].size;
			float d=(zone[i].Pos-z.Pos).Sq();
			if(d<=m*m*0.25f && d<best) {
				best=d;
				near_idx=i;
				}
			}

		if(near_idx<0)
			return false;

		del_zone(near_idx);				// Met à jour
		return true;
	}

	S_DATA()
	{
		init();
	}

	~S_DATA()
	{
		destroy();
	}
};

class AI_PLAYER					// Classe gérant les joueurs controlés par l'IA
{
public:
	char		*name;			// Attention faudrait pas qu'il se prenne pour quelqu'un!! -> indique aussi le fichier correspondant à l'IA (faut sauvegarder les cervelles)
	BRAIN		decider;		// Réseau de neurones d'analyse de la partie et de décision
	BRAIN		anticiper;		// Réseau de neurones voué à l'analyse et à l'anticipation des mouvements ennemis
	STRATEGY	strategy;		// Stratégie en cours de validité
	S_DATA		strategic_data;	// Données stratégiques sur la partie
	int			player_id;		// Identifiant du joueur
	int			unit_id;		// Identifiant d'unité pour parcourir les unités

	inline void init()
	{
		name=strdup("default ai");
		decider.init();
		anticiper.init();
		strategy.init();
		strategic_data.init();
		player_id=0;
		unit_id=0;
	}

	inline void destroy()
	{
		if(name)	free(name);
		decider.destroy();
		anticiper.destroy();
		strategy.destroy();
		strategic_data.destroy();
		player_id=0;
		unit_id=0;
	}

	AI_PLAYER()
	{
		init();
	}

	~AI_PLAYER()
	{
		destroy();
	}

	inline void change_name(char *new_name)		// Change le nom de l'IA (conduit à la création d'un nouveau fichier)
	{
		if(name)
			free(name);
		name=strdup(new_name);
	}

	void save()
	{
		char filename[100];
		filename[0]=0;
		strcat(filename,"ai/");
		strcat(filename,name);
		strcat(filename,".ai");
		FILE *file=fopen(filename,"wb");

		byte l=strlen(name);
		fwrite(&l,1,1,file);		// Nom de l'IA
		fwrite(name,l,1,file);

		decider.save(file);			// Réseau de décision

		anticiper.save(file);		// Réseau d'analyse

		fclose(file);
	}

	void load(char *filename,int id=0)
	{
		FILE *file=fopen(filename,"rb");

		byte l;
		fread(&l,1,1,file);
		if(name)
			free(name);
		name=(char*) malloc(l+1);
		name[l]=0;
		fread(name,l,1,file);

		decider.load(file);

		anticiper.load(file);

		fclose(file);

		player_id=id;
	}

	void think(MAP *map);
};

#endif
