/*!
**	\file	random.cpp
**
**		copied from http://www.humble-programmer.com/download/Random.cpp
**
**		Random number generator for the DGS application.  This is the 
**		"Mersenne Twister" random number generator MT19937, which generates 
**		pseudorandom integers uniformly distributed in 0..(2^32 - 1) starting
**		from any odd seed in 0..(2^32 - 1).
**
**		According to the URL <http://www.math.keio.ac.jp/~matumoto/emt.html>
**		the Mersenne Twister is "designed with consideration of the flaws of 
**		various existing generators," has a period of 2^19937 - 1, gives a 
**		sequence that is 623-dimensionally equidistributed, and "has passed 
**		many stringent tests, including the die-hard test of G. Marsaglia and 
**		the load test of P. Hellekalek and S. Wegenkittl."  It is efficient in 
**		memory usage (typically using 2506 to 5012 bytes of static data) and 
**		the code is quite short as well).  It generates random numbers in 
**		batches of 624 at a time, so the caching and pipelining of modern 
**		systems is exploited.  It is also divide- and mod-free.
**

**		Copyright (C) 1997 Makoto Matsumoto and Takuji Nishimura.  When you 
**		use this, send an e-mail to <matumoto@math.keio.ac.jp> with an 
**		appropriate reference to your work.  This version is a recode by 
**		Shawn Cokus (Cokus@math.washington.edu) on March 8, 1998 of a version 
**		by Takuji Nishimura (who had suggestions from Topher Cooper and Marc 
**		Rieffel in July-August 1997).
*/
#include <limits.h>
#include <stddef.h>
typedef unsigned long DWORD;

/*	-----------------------------------------------------------------------
	CONSTANTS / DEFINITIONS
	-----------------------------------------------------------------------	*/
static const int		kN = 624;				// length of state vector
static const int		kM = 397;				// a period parameter
static const DWORD		kMagic = 0x9908B0DFU;	// a magic constant
///
///	\def	HI_BIT(u)
///			Masks all but highest bit of 32-bit value u
///	\def	LO_BIT(u)
///			Masks all but lowest bit of 32-bit value u
///	\def	LOW_BITS(u)
///			Masks only the highest bit of 32-bit value u
///	\def	MIX_BITS(u,v)
///			Moves the high bit of u to high bit of v
///
#define	HI_BIT(u)		((u) & 0x80000000U)
#define	LO_BIT(u)		((u) & 0x00000001U)
#define	LOW_BITS(u)		((u) & 0x7FFFFFFFU)
#define	MIX_BITS(u,v)	(HI_BIT(u)|LOW_BITS(v))
/*	-----------------------------------------------------------------------
	LOCAL (STATIC) VARIABLES
	-----------------------------------------------------------------------	*/
static int				s_nLeft = -1;		// this many values before reloading
static DWORD			s_dwState[kN+1];	// state vector + 1 extra for ANSI C
static DWORD *			s_pdwNext = NULL;	// ptr to next random value
///
///	\brief	Reloads the table of random numbers
///	\return	First random number
///
static DWORD
reloadMT(void)
	{
#if	_DEBUG	// PRE-CONDITION
	ATLASSERT(s_nLeft >= -1);
#endif

	int			ndx;

	DWORD		s0, 
				s1;
	DWORD *		p0 = s_dwState;
	DWORD *		p2 = &s_dwState[2];
	DWORD *		pM = &s_dwState[kM];

	s_nLeft = kN - 1;
	s_pdwNext = s_dwState + 1;

	for (s0 = s_dwState[0], s1 = s_dwState[1], ndx = (kN - kM) + 1; --ndx; s0 = s1, s1 = *p2++)
		*p0++ = *pM++ ^ (MIX_BITS(s0, s1) >> 1) ^ (LO_BIT(s1) ? kMagic : 0U);

	for (pM = s_dwState, ndx = kM; --ndx; s0 = s1, s1 = *p2++)
		*p0++ = *pM++ ^ (MIX_BITS(s0, s1) >> 1) ^ (LO_BIT(s1) ? kMagic : 0U);

	s1 = s_dwState[0];
	*p0 = *pM ^ (MIX_BITS(s0, s1) >> 1) ^ (LO_BIT(s1) ? kMagic : 0U);
	s1 ^= (s1 >> 11);
	s1 ^= (s1 <<  7) & 0x9D2C5680U;
	s1 ^= (s1 << 15) & 0xEFC60000U;

#if	_DEBUG	// POST-CONDITION
	ATLASSERT(s_nLeft > 0);
	ASSERT_PTR(s_pdwNext);
#endif

	return s1 ^ (s1 >> 18);
	}	/* end of reloadMT() */
///
///	\brief	Bounded random number generator
///	\return	Random 32-bit value up to a given value.
///	\param	dwUpperBound
///			Maximum allowable value (defaults to ULONG_MAX)
///
DWORD
Random(const DWORD dwUpperBound)
	{
#if	_DEBUG	// PRE-CONDITION
	ATLASSERT(dwUpperBound > 0);
	ATLASSERT(s_nLeft >= 0);
#endif

	DWORD	dwRnd;

	if (--s_nLeft < 0)
		dwRnd = reloadMT();
	else
		{
		dwRnd  = *s_pdwNext++;
		dwRnd ^= (dwRnd >> 11);
		dwRnd ^= (dwRnd <<  7) & 0x9D2C5680U;
		dwRnd ^= (dwRnd << 15) & 0xEFC60000U;
		dwRnd ^= (dwRnd >> 18);
		}

	if (dwUpperBound == 1L)	// special case is a 0/1 coin flip
		dwRnd &= 0x01U;
	else if (dwUpperBound < ULONG_MAX)
		dwRnd %= dwUpperBound + 1;

#if	_DEBUG	// POST-CONDITION
	ATLASSERT(dwRnd <= dwUpperBound);
#endif

	return dwRnd;
	}	/* end of Random() */
///
///	\brief	Sets the random number generator seed
///
///	We initialize s_dwState[0..(kN-1)] via the generator
///
///		x_new = (69069 * x_old) mod 2^32
///
///	from Line 15 of Table 1, p. 106, Sec. 3.3.4 of Knuth Volume 2, 3rd ed.
///	The generator's potency (min. s>=0 with (69069-1)^s = 0 mod 2^32) is  16, 
///	which seems to be alright by p. 25, Sec. 3.2.1.3 of Knuth.  It  also does 
///	well in the dimension 2..5 spectral tests, but it could be  better in 
///	dimension 6 (Line 15, Table 1, p. 106, Sec. 3.3.4, Knuth).
///
///	Note that the random number user does not see the values generated here 
///	directly since reloadMT() will always munge them first, so maybe none of 
///	all of this matters.  In fact, the seed values made here could even be 
///	extra-special desirable if the Mersenne Twister theory says so--that's why 
///	the only change I made is to restrict to odd seeds.
///
///	NOTES:
///		1.	(SJC) I do not know what the initial s_dwState requirements of the 
///			Mersenne Twister are, but it seems this seeding generator could be 
///			better.  It achieves the maximum period for its modulus (2^30) iff 
///			x_initial is odd (p. 20-21, Sec. 3.2.1.2, Knuth); if x_initial can 
///			be even, you have sequences like 
///				0, 0, 0, ...
///				2^31, 2^31, 2^31, ...
///				2^30, 2^30, 2^30, ...
///				2^29, 2^29 + 2^31, 2^29, 2^29 + 2^31, ..., etc. 
///			so I force seed to be odd.  Even if x_initial is odd, if x_initial 
///			is 1 mod 4 then:
///				the lowest bit of x is always 1,
///				the s_pdwNext-to-lowest bit of x is always 0,
///				the 2nd-from-lowest bit of x alternates      ... 0 1 0 1 0 1 0 1 ... ,
///				the 3rd-from-lowest bit of x 4-cycles        ... 0 1 1 0 0 1 1 0 ... ,
///				the 4th-from-lowest bit of x has the 8-cycle ... 0 0 0 1 1 1 1 0 ... ,
///			and if x_initial is 3 mod 4 then
///				the lowest bit of x is always 1,
///				the s_pdwNext-to-lowest bit of x is always 1,
///				the 2nd-from-lowest bit of x alternates      ... 0 1 0 1 0 1 0 1 ... ,
///				the 3rd-from-lowest bit of x 4-cycles        ... 0 0 1 1 0 0 1 1 ... ,
///				the 4th-from-lowest bit of x has the 8-cycle ... 0 0 1 1 1 1 0 0 ... ,
///
///	\param	dwSeed
///			Seed value (any non-zero integer)
///
void
RandomSeed(DWORD dwSeed)
	{
	DWORD *	pdx = s_dwState;
	DWORD	dwx = dwSeed | 1U;

	//~ DEBUG_OUT("random seed value is 0x%08lX", dwSeed);
	
	*pdx++ = dwx;
	for (int idx = kN; --idx; *pdx++ = dwx)
		dwx *= 69069U;

	s_nLeft = 0;
	}	/* end of RandomSeed() */
/****************************************************************************
**
**	End of Random.cpp
**
**	-------------------------------------------------------------------------
**
**	$History: $
**
****************************************************************************/
