#include <assert.h>
#include "object.h"
#include "engine.h"
#include "anim.h"
#include "player.h"

using namespace std;

TEG_MAP *Object::getMap()
{ 
	assert (room);
	return room->map;
}

// tilestack -> check the tiles in all 3 layers at the same time
int Object::getTileStackFlags(int mx, int my)
{
	int result = 0;
	TEG_MAP *map = getMap();
	if (!map) return 0; // no map found !!!
	
	if (mx < 0 || my < 0 || mx >= map->w || my >= map->h)
	{
		return 0;
	}
	else
	{
		int i1, i2, f1, f2;
		i1 = teg_mapget (map, 0, mx, my);
		i2 = teg_mapget (map, 1, mx, my);
		if (i1 >= 0) f1 = map->tilelist->tiles[i1].flags; else f1 = 0;
		if (i2 >= 0) f2 = map->tilelist->tiles[i2].flags; else f2 = 0;
		
		// check for solids
		if (f1 == 1 || f2 == 1) result |= TS_SOLID;
		
		return result;
	}
}

void Object::setDir(Dir _dir) 
{ 
	if (_dir != dir) 
	{ 
		dir = _dir; 
		setAnim (current); // reset anim, reinitalize hotx, hoty
	} 
}

void Object::setAnim (Anim *a)
{
	assert (a);
	current = a;
	frame = 0;
	animcounter = 0;
	sprdx = a->getFrame(0, dir)->hotx;
	sprdy = a->getFrame(0, dir)->hoty;
	w = a->getFrame(0, dir)->w;
	h = a->getFrame(0, dir)->h;
}

Resources *Object::getResources()
{
	if (e)
		return e->getResources();
	else
		return NULL;
	
}

Object::Object (Room *r)
{
	assert (r);
	current = NULL;
	frame = 0;
	animcounter = 0;
	x = 0;
	y = 0;
	w = 8;
	h = 8;
	sprdx = 0;
	sprdy = 0;
	visible = false;
	alive = true;
	solid = true;
	parent = NULL;
	dir = RIGHT;
	room = r;
}

void Object::setParent (Objects *p)
{
	assert (p);
	parent = p;
	e = parent->getParent();
	assert (e);
}

RLE_SPRITE *Object::getSprite ()
{
	if (!current) return NULL;
	return current->getFrameRle (frame, dir);
}

void Object::try_move (fix dx, fix dy)
{
	int dxleft = dx, dyleft= dy;
	int ddx = dx > 0 ? 1 : -1;
	int ddy = dy > 0 ? 1 : -1;	
	int trydx, trydy;
	bool valid = true;
	while ((abs(dxleft) > 0 || abs (dyleft) > 0) && valid)
	{
		if (abs(dxleft) > abs(dyleft))
		{
			trydy = 0;
			if (abs(dxleft) >= 1)
				trydx = ddx;
			else
				trydx = dxleft;
		}
		else
		{
			trydx = 0;
			if (abs(dyleft) >= 1)
				trydy = ddy;
			else
				trydy = dyleft;
		}

		// check with tilemap background, but only if object is solid.
		if (solid)
		{
			
			// check if (x +  |trydx, y + trydy) is valid
			int mx1, my1, mx2, my2;
			int ix, iy;
			TEG_MAP *map = getMap();
			mx1 = ((int)x + trydx) / map->tilelist->tilew;
			my1 = ((int)y + trydy) / map->tilelist->tileh;
			mx2 = ((int)x + trydx + w - 1) / map->tilelist->tilew;
			my2 = ((int)y + trydy + h - 1) / map->tilelist->tileh;
					
			// loop through all map positions we touch with the solid region
			for (ix = mx1; ix <= mx2; ++ix)
			{
				for (iy = my1; iy <= my2; ++iy)
				{
					// see if there is a solid tile at this position
					if (getTileStackFlags (ix, iy) & TS_SOLID)
					{
							valid = false;
					}
				}
			}
		}
		
		if (valid)
		{
			x += trydx;
			dxleft -= trydx;
			y += trydy;
			dyleft -= trydy;
		}		
	}
}
   
void Object::setLocation (fix nx, fix ny)
{
	x = nx;
	y = ny;
}

void Object::setRoom (Room *_room)
{
	assert (_room);	
	room = _room;
}
void Objects::killAll()
{
	list<Object*>::iterator i;
	for (i = objects.begin(); i != objects.end(); ++i)
	{
		delete (*i);
		(*i) = NULL;
	}
	objects.clear();	
}

void Object::update()
{
	assert (parent);
	assert (room);
	if (!current) return;
	
	// ask current time
	int counter = e->getCounter();
	
	// if we're lagging behind too much, catch up at once
	if (counter - animcounter > 1000) animcounter = counter;
	
	// if frametime is 0, loop same frame forever
	if (current->getFrameTime (frame, dir) > 0) 
	// otherwise, see if we need to go to the next frame.
	{
		while (counter - animcounter > current->getFrameTime (frame, dir))
		{
			animcounter += current->getFrameTime (frame, dir);
			frame++;
			if (frame >= current->getFrameCount(dir)) frame = 0;
		}
	}
}


void Objects::add(Object *o)
{
	objects.push_back (o);
	o->setParent (this);
}

class MyObjectRemover
{
   public:
	  bool operator()(Object *o)
	  {
		 if (!o->isAlive())
		 {
			delete o;
			return 1;
		 }
		 return 0;
	  }
};

void Objects::update()
{
	list<Object*>::iterator i;
	for (i = objects.begin(); i != objects.end(); i++)
	{
		if ((*i)->isAlive()) (*i)->update();
	}
	
	// collission detection!	
	list<Object*>::iterator j;
	for (i = objects.begin(); i != objects.end(); i++)
		for (j = objects.begin(); j != i; j++)
	{
		// see if bb interesect
		if ((*i)->isAlive() && (*j)->isAlive() && (*i)->getRoom() == (*j)->getRoom())
		{
			int x1 = (*i)->getx();
			int y1 = (*i)->gety();
			int w1 = (*i)->getw();
			int h1 = (*i)->geth();
			int x2 = (*j)->getx();
			int y2 = (*j)->gety();
			int w2 = (*j)->getw();
			int h2 = (*j)->geth();
			if(!((x1 >= x2+w2) || (x2 >= x1+w1) || (y1 >= y2+h2) || (y2 >= y1+h1)))
			{
				(*i)->handleCollission ((*j));
				(*j)->handleCollission ((*i));
			}
		}		
	}
	
	// remove all that are not alive!
	objects.remove_if (MyObjectRemover());
}

void Objects::draw (BITMAP *buffer, Room *room, int cx, int cy, int cw, int ch, int xofst, int yofst)
{
    set_clip_rect (buffer, cx, cy, cx + cw, cy + ch);
    
	list<Object*>::iterator i;
	for (i = objects.begin(); i != objects.end(); i++)
	{
		if ((*i)->isVisible() && (*i)->isAlive() && 
			(*i)->getRoom() == room)
		{
			RLE_SPRITE *rle = (*i)->getSprite();
			if (rle)
			{
				draw_rle_sprite (buffer, (*i)->getSprite(), 
					(*i)->getSprx() - xofst, (*i)->getSpry() -yofst);
			}
#ifdef DEBUG
			if (parent->isDebug())
			{
				rect (buffer, 
					(*i)->getx() - xofst, 
					(*i)->gety() - yofst,
					(*i)->getx() - xofst + (*i)->getw(), 
					(*i)->gety() - yofst + (*i)->geth(),
					GREEN);
						
				//~ textprintf_ex (buffer, font, 
						//~ (*i)->getx() - xofst, 
						//~ (*i)->gety() - yofst, GREEN, -1, "%i",
						//~ (*i)->getRoom()
					//~ );
			}			
#endif
		}
	}
	
	set_clip_rect (buffer, 0, 0, buffer->w, buffer->h);
}
