#include <assert.h>
#include "engine.h"
#include "main.h"
#include "object.h"
#include <stdio.h>
#include "player.h"
#include "door.h"
#include "monster.h"

int Engine::getCounter () 
{ 
	return parent->getCounter(); 
}

// fill up target bmp with the same map over and over
void teg_draw_tiled_rle (BITMAP *bmp, const TEG_MAP* map, int layer, int xview, int yview, int frame)
{
	int x, y;
	int tilex, tiley;
	int tilew, tileh;
	tileh = map->tilelist->tileh;
	tilew = map->tilelist->tilew;
	tiley = (yview / tileh) % map->h;
	while (tiley < 0) tiley += map->h;
	for (y = -(yview % tileh); y < bmp->h; y += tileh)
	{
		
		//~ int tempx = ((xview/tilew) + 1);		
		//~ tilex = map->w - (tempx % map->w) - 1;
		tilex = (xview / tilew) % map->w;
		while (tilex < 0) tilex += map->w;
		for (x = - (xview % tilew) ; x < bmp->w; x += tilew)
		{
			RLE_SPRITE *s;
			int i;
			i = teg_mapget(map, layer, tilex, tiley);
			if (i >= 0 && i < map->tilelist->tilenum)
			{
					i += frame * map->tilelist->tilenum;
				s = map->tilelist->tiles[i].rle;
				if (s != NULL)
					draw_rle_sprite (bmp, s, x, y);
			}
			tilex++;
			if (tilex >= map->w) tilex = 0;
		}
		tiley++;
		if (tiley >= map->h) tiley = 0;
	}
}

Engine::Engine (Main *p) : objects (this)
{
	parent = p;
	player[0] = NULL;
	player[1] = NULL;
	debug = false;
	resources = p->getResources();
	settings = p->getSettings();
	dp = NULL;
}

void Engine::doneLevel()
{
	objects.killAll();
}

Button* Engine::getButton(int p)
{ 
	return settings->button[p]; 
}

void Engine::drawStatus (BITMAP *buffer, int x, int y, PlayerState *xps)
{
	int bananasGot = 0;
	for (int i = 0; i < settings->numPlayers; ++i) bananasGot += ps[i].bananas;
	textprintf_ex (buffer, gamefont, x + 8, y + 8, YELLOW, -1, "BANANAS %02i LEFT %02i", xps->bananas, bananaCount - bananasGot);
	textprintf_ex (buffer, gamefont, x + 8, y + 24, RED, -1, "HEALTH %03i MAX %03i", xps->hp, xps->hpMax);
	textprintf_ex (buffer, gamefont, x + 8, y + 40, WHITE, -1, "XP %05i", xps->xp);	
}

void Engine::init()
{
	Player::init(resources);	
	Bullet::init(resources);
	Door::init(resources);
	Monster::init(resources);	
	Banana::init(resources);	
	Teleport::init(resources);
	
	gamefont = resources->getFont("METRO");
 	srand(time(0));	
}

void Engine::initGame ()
{
	// initialize game and player stats.
	// to start a new game.
	currentLevel = 0;
	gameTimer = defaultGameTime;

	// create a level
	// TODO: dynamic	
	ps[0] = PlayerState();
	ps[1] = PlayerState();
	
	monsterHp = defaultMonsterHp;
	initLevel();	
}

void Engine::initLevel()
{
	// initialize objects
	
 	level = createLevel(resources, &objects, currentLevel + (settings->numPlayers == 1 ? 4 : 6), monsterHp);
	
	Room *startRoom = level->getStartRoom();
	
	player[0] = new Player (&ps[0], startRoom, 0);
	player[0]->setLocation ((fix)120, (fix)160);
	objects.add (player[0]);
	if (settings->numPlayers == 2)
	{
		player[1] = new Player (&ps[1], startRoom, 1);
		player[1]->setLocation ((fix)160, (fix)120);
		player[1]->setRoom (startRoom);
		objects.add (player[1]);		
	}
	else
	{
		player[1] = NULL;
	}
	
 	view[0].init (settings->numPlayers, 0);
	view[0].player = player[0]; 
	view[1].init (2, 1);
	view[1].player = player[1];

	bananaCount = level->getBananaCount();
	
	state = GS_INTRO; // show cutscene
	introTimer = 40;
}

void Engine::resume ()
{
	state = oldState;
}

// called when exit found
void Engine::nextLevel()
{
	// go on to next level
	state = GS_INTRO;
	introTimer = 40;
	currentLevel++;
	objects.killAll();
	delete level;
	int max = 0;
	int maxPlayer = 0;
	int i;
	for (i = 0; i < settings->numPlayers; ++i)
	{
		if (ps[i].bananas > max)
		{
			maxPlayer = i;
			max = ps[i].bananas;
		}
		
	}
	ps[maxPlayer].hpMax += 25;	
	for (i = 0; i < settings->numPlayers; ++i)
	{	
		ps[i].hp = ps[i].hpMax;
		ps[i].bananas = 0;
	}
	gameTimer += gameTimeIncrease; // extra minute
	monsterHp += monsterHpIncrease;
	initLevel();
}

// TODO: copy this function back to tegel
void Engine::teg_partdraw_rle (BITMAP *bmp, const TEG_MAP* map, int layer, int cx, int cy, int cw, int ch, int xview, int yview)
{
	int x, y;
    int tilex, tiley;
    
    set_clip_rect (bmp, cx, cy, cx + cw, cy + ch);

    for (tilex = 0; tilex < map->w; tilex++)
    {
        for (tiley = 0; tiley < map->h; tiley++)
        {
            RLE_SPRITE *s;
            int i;
            x = tilex * map->tilelist->tilew - xview;
            y = tiley * map->tilelist->tileh - yview;
            i = teg_mapget(map, layer, tilex, tiley);
            if (i >= 0 && i < map->tilelist->tilenum)
            {
                s = map->tilelist->tiles[i].rle;
                if (s != NULL)
                    draw_rle_sprite (bmp, s, x, y);
            }
        }
    }

    set_clip_rect (bmp, 0, 0, bmp->w, bmp->h);
}

void Engine::draw (BITMAP *buffer)
{
	switch (state)
	{
		case GS_PLAY:
		{
			clear_to_color (buffer, BLACK);
			//~ teg_draw (buffer, level, 0, camera_x, 0);
			//~ objects.draw(buffer, camera_x, 0);
			for (int i = 0; i < settings->numPlayers; ++i)
			{
				if (!ps[i].died) 
				{
					TEG_MAP *map = view[i].player->getMap();
					teg_partdraw_rle (buffer, map, 0,
							view[i].left, view[i].top, view[i].width, view[i].height, 
							view[i].camera_x - view[i].left, view[i].camera_y - view[i].top);
					teg_partdraw_rle (buffer, map, 1,
							view[i].left, view[i].top, view[i].width, view[i].height, 
							view[i].camera_x - view[i].left, view[i].camera_y - view[i].top);
					objects.draw(buffer, view[i].player->getRoom(), view[i].left, view[i].top, 
							view[i].width, view[i].height, 
							view[i].camera_x - view[i].left, view[i].camera_y - view[i].top);
				}
				drawStatus (buffer, view[i].status_x, view[i].status_y, &ps[i]);				
			}
			int min = (gameTimer / 60000);
			int sec = (gameTimer / 1000) % 60;
			int csec = (gameTimer / 10) % 100;
			textprintf_ex (buffer, gamefont, 0, 
				464, WHITE, -1, "%02i:%02i:%02i", min, sec, csec);
		}	break;
		case GS_OUTRO: 
			clear_to_color (buffer, BLACK);
			textprintf_centre_ex (buffer, gamefont, buffer->w / 2, 
				buffer->h / 2, WHITE, -1, gameover_message.c_str());
			break;			
		case GS_INTRO:
			clear_to_color (buffer, BLACK);
			
			textprintf_centre_ex (buffer, gamefont, buffer->w / 2, 
				(buffer->h - text_height (gamefont)) / 2, WHITE, -1, 
				"LEVEL %i", currentLevel+1);
			if (((introTimer / 6) & 1) && currentLevel > 0)
			{
				textprintf_centre_ex (buffer, gamefont, buffer->w / 2, 
					(buffer->h - text_height (gamefont)) * 3 / 4, CYAN, -1, 
					"EXTRA TIME");	
			}
			break;
		case GS_DONE: case GS_MENU_RESUME:
			break;
		default:
			assert (false);
	}
}

void Engine::update ()
{
	int c = 0;
	if (keypressed())
	{	
		c = readkey();
		if (c >> 8 == KEY_ESC)
		{
			oldState = state;
			state = GS_MENU_RESUME;
		}
#ifdef DEBUG
		if (c >> 8 == KEY_F11)
		{
			debug = !debug;
		}
#endif	
		if (c >> 8 == KEY_F12)
		{
			screenshot();
		}		
	}
	switch (state)
	{
		case GS_INTRO: // level intro screen showing
			introTimer--;
			if (introTimer == 0)
			{
				state = GS_PLAY;
				introTimer = 40;
			}
			break;
		case GS_OUTRO:
			outroTimer--;
			if (outroTimer == 0)
			{
				doneLevel();
				state = GS_DONE;	
				break;
			}
			break;
		case GS_PLAY: // game playing
		{
			// add objects based on position
			gameTimer -= Main::update_speed;
			objects.update();
			bool gameover = true;
			int bananasGot = 0;

			for (int i = 0; i < settings->numPlayers; ++i)
			{
				// update camera
				int newx, newy;
				newx = player[i]->getx() - (view[i].width / 2);
				newy = player[i]->gety() - (view[i].height / 2);
// 				if (newx >= 0)
					view[i].camera_x = newx;
// 				if (newy >= 0)
					view[i].camera_y = newy;
					
				if (!ps[i].died) gameover = false;
				bananasGot += ps[i].bananas;
			}
			if (gameover)
			{
				gameover_message = "GAME OVER";
				outroTimer = 100;
				state = GS_OUTRO;
			}
			if (gameTimer <= 0)
			{
				gameover_message = "TIME UP";
				outroTimer = 100;
				state = GS_OUTRO;
			}
 			if (bananaCount - bananasGot == 0)
			{
				nextLevel();
			}
		}
			break;
		case GS_MENU_RESUME: // do nothing, main will switch to resume menu.
			break;
		case GS_DONE: // do nothing, game is finished (won or lost)
			break;
		default:
			assert (false); // shouldn't occur
	}  
}

void Engine::playSample (SAMPLE *s)
{
	if (!settings->isSoundOn()) return;
	play_sample (s, 127, 127, 1000, 0);
}

void Engine::playMusic (const char *id)
{
    if (!(settings->isSoundOn() && settings->isMusicOn())) return;
    if (dp)
    {
        al_stop_duh (dp);
        dp = NULL;
    }
    dp = al_start_duh (resources->getDuh (id), 2, 0, 1.0f, 4096, 22050);
}

void Engine::stopMusic ()
{
    if (dp) 
    {
        al_stop_duh (dp);
        dp = NULL;
    }
}

void Engine::done()
{
	stopMusic();
}

Player *Engine::getNearestPlayer (Object *o)
{
	assert (o);
	if (settings->numPlayers == 2) 
	{
		Room *r1, *r2;
		r1 = player[0]->getRoom();
		r2 = player[1]->getRoom();
		if (r1 == o->getRoom() && r2 == o->getRoom())
		{
			fix dx1 = player[0]->getx() - o->getx();
			fix dy1 = player[0]->gety() - o->gety();
			fix dx2 = player[1]->getx() - o->getx();
			fix dy2 = player[1]->gety() - o->gety();
			if (dx1 * dx1 + dy1 * dy1 < dx2 * dx2 + dy2 * dy2)
			{
				return player[1];
			}
			else
			{
				return player[0];
			}
		}
		else if (r1 == o->getRoom())
		{
			return player[0];
		}
		else if (r2 == o->getRoom())
		{
			return player[1];
		}
		else
		{
			return NULL;
		}
	}
	else
	{
		if (player[0]->getRoom() == o->getRoom())
		{
			return player[0];
		}
		else
		{
			return NULL;
		}
	}
}
