#include <assert.h>
#include "main.h"

// needed for "Sleep()"
#ifdef ALLEGRO_WINDOWS
#include <winalleg.h>
#endif

volatile int Main::counter = 0;

void Main::timer_handler()
{
	Main::counter += 10;
}

END_OF_FUNCTION(timer_handler);

Main::Main() : engine (this), menu (&engine), testview (&resources)
{
		
	buffer = NULL;
	last_fps = 0;
	update_counter = 0;
	frame_count = 0;
	frame_counter = 0;
}

int Main::init(int argc, const char *const *argv)
// returns 1 on success, 0 on failure
{
	LOCK_VARIABLE(Main::counter);
	LOCK_FUNCTION(timer_handler);
	if (allegro_init () < 0)
	{
		allegro_message ("init failed");
		return 0;
	}
	
	atexit(&dumb_exit); // dumb cleanup
    dumb_register_stdfiles();

	override_config_file ("fole1.cfg");
		settings.getFromConfig();
		settings.getFromArgs (argc, argv);
	
	if (install_keyboard () < 0)
	{
		allegro_message ("install keyboard failed");
		return 0;
	}
		initKeyboard(); // install custom keyboard handler
		
	if (install_timer() < 0)
	{
		allegro_message ("install timer failed");
		return 0;
	}
	set_volume_per_voice (1);	
	if (settings.isSoundOn() && install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL) < 0)
	{
		// could not get sound to work
		settings.setSoundOn(false);
		allegro_message ("Could not initialize sound. Sound is turned off.\n%s\n", allegro_error);
	}
	if (install_int (timer_handler, 10) < 0)
	{
		allegro_message ("installation of timer handler failed");
		return 0;
	}
	set_color_depth (16);
	if (settings.windowed)
	{
		if (set_gfx_mode(GFX_AUTODETECT_WINDOWED,640,480,0,0)!=0)
		{
			allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
			return 0;
		}
	}
	else
	{
		if (set_gfx_mode(GFX_AUTODETECT_FULLSCREEN,640,480,0,0)!=0)
		{
			allegro_message("Unable initialize graphics module\n%s\n", allegro_error);
			return 0;
		}
	}
	buffer = create_bitmap(SCREEN_W, SCREEN_H);
	if (!buffer)
	{
		allegro_message ("Error creating background buffer");
		return 0;
	}
	if (install_mouse() == -1)
	{
		allegro_message ("could not install mouse");
		set_gfx_mode (GFX_TEXT, 0, 0, 0, 0);
		allegro_exit();
		return 0;
	}		
	set_window_title ("Fole I - Speedhack 2006 - Max & Amarillion");
	teg_install();
	if (!resources.init())
	{
		allegro_message ("Could not load all resources!");
		return 0;
	}
	engine.init();   
	menu.init(&resources, &settings); 
	return 1;
}

void Main::run()
{
	menu.initStart();
		
	if (settings.runMode == Settings::RM_TESTVIEW)
		state = TESTVIEW;
	else
		state = MENU;	
	
	while (state != QUIT)
	{
		while ((counter - update_counter) > update_speed) // update 25 times/sec
		{
			update_counter += update_speed;
			int update_start = counter;
			al_poll_duh(engine.dp);
			update ();
			
			// check to see if update takes too long
			if (counter - update_start > 40)
			{
				// we probably can't catch up anymore.
				// move update_counter ahead, or we have to
				// catch up even more next loop
				update_counter = counter;
				break;
			}
		}
		
		al_poll_duh(engine.dp);
		draw();
        if (settings.fpsOn)
		{
			textprintf_ex (buffer, font, 0, 0, 
				  WHITE, -1, "fps: %d", last_fps);
		}
		
		blit (buffer, screen, 0, 0, 0, 0, buffer->w, buffer->h);
		
		if ((counter - frame_counter) > 1000)
		{
			last_fps = frame_count;
			frame_count = 0;
			frame_counter = counter;
		}
		frame_count++;

#ifdef ALLEGRO_LINUX
		struct timeval tv = {0, 1};
		select (0, NULL, NULL, NULL, &tv);
#elif defined(ALLEGRO_WINDOWS)
		Sleep(1);
#else
		yield_time_slice();
#endif   
	}			
}

void Main::update()
{
	int result;
	switch (state)
	{
		case MENU: // run menu
			menu.update();
			result = menu.getState();
			switch (result)
			{
				case 0: state = QUIT; break; // quit
				case 1:
					engine.initGame(); 
					engine.playMusic("SYREEN");
					state = GAME;
					break; // start new game
				case 2: 
					engine.resume();
					engine.playMusic("SYREEN");
					state = GAME; 
					break; // resume game
				case -1: break; // continue showing menu
				default: assert (false); // shouldn't occur
			}
			break;
		case QUIT: // quit
			// do nothing
			break;
		case GAME: // game is playing, no matter if at intro screen or play screen
			engine.update();
			{
				Engine::GameState gs = engine.getState();				
				switch (gs)
				{
					case Engine::GS_INTRO:
					case Engine::GS_OUTRO:
					case Engine::GS_PLAY: 
						break; // do nothing
					case Engine::GS_MENU_RESUME: // resume menu
						menu.initResume();
						engine.stopMusic();
						state = MENU;
						break;
					case Engine::GS_DONE: // go back to start menu
						menu.initStart();
						engine.stopMusic();
						state = MENU;
						break;
					default:
						assert (false);
				}
			}
			break;
		case TESTVIEW:
			testview.update();
			if (testview.isFinished()) { state = QUIT; }
			break;
		default:
			assert (false); // shouldn't occur
	}
#ifdef DEBUG
	if (key[KEY_F10]) state = QUIT; // emergency bail out
#endif
}

void Main::draw()
{
	switch (state)
	{
		case MENU:
			menu.draw(buffer);
			break;
		case QUIT:
			break;
		case GAME:
			// draw game
			engine.draw(buffer);
			break;
		case TESTVIEW:
			testview.draw (buffer);
			break;
		default:
			assert (false);
	}
}

Main::~Main()
{
	menu.done();
	teg_uninstall();
	if (buffer) destroy_bitmap (buffer);	
	engine.done();
}

int main(int argc, const char *const *argv)
{
	Main m;
	if (m.init(argc, argv))
	{		
		m.run();
	}
	return 0;
} END_OF_MAIN();
