#include <assert.h>
#include "level.h"
#include <math.h>

#include "door.h"
#include "monster.h"
#include "player.h"

using namespace std;
using namespace xdom;

Level *createTestLevel(Resources *resources, Objects *objects, int monsterHp)
{
	Level *level = new Level();
	RoomSet *roomSet = resources->getRoomSet ("start");

	Room *temp[15];
	for (int i = 0; i < 14; ++i) { temp[i] = NULL; }
	temp[0] = new Room(objects, roomSet->findRoom(true, true, true, true, false), monsterHp);
	temp[1] = new Room(objects, roomSet->findRoom(true, true, false, true, false), monsterHp);
	temp[2] = new Room(objects, roomSet->findRoom(false, true, false, false, true), monsterHp);
	temp[3] = new Room(objects, roomSet->findRoom(false, false, true, true, false), monsterHp);
	temp[4] = new Room(objects, roomSet->findRoom(false, false, true, false, false), monsterHp);
	temp[5] = new Room(objects, roomSet->findRoom(true, true, false, false, false), monsterHp);
	temp[6] = new Room(objects, roomSet->findRoom(true, false, false, false, true), monsterHp);
	temp[7] = new Room(objects, roomSet->findRoom(false, true, true, false, false), monsterHp);
	temp[8] = new Room(objects, roomSet->findRoom(true, false, false, true, true), monsterHp);
	temp[9] = new Room(objects, roomSet->findRoom(false, false, true, false, false), monsterHp);
	temp[10] = new Room(objects, roomSet->findRoom(true, false, true, true, true), monsterHp);
	temp[11] = new Room(objects, roomSet->findRoom(true, false, false, true, false), monsterHp);
	temp[12] = new Room(objects, roomSet->findRoom(false, true, false, true, false), monsterHp);
	temp[13] = new Room(objects, roomSet->findRoom(false, true, true, false, false), monsterHp);
	for (int i = 0; i < 14; ++i) { assert (temp[i]); level->rooms.push_back (temp[i]); }
	
	temp[0]->linkDoor (temp[1], 0);
	temp[0]->linkDoor (temp[5], 1);
	temp[0]->linkDoor (temp[10], 2);
	temp[0]->linkDoor (temp[7], 3);
	temp[1]->linkDoor (temp[2], 0);
	temp[1]->linkDoor (temp[3], 3);
	temp[2]->linkTeleport (temp[6]);
	temp[3]->linkDoor (temp[4], 3);
	temp[5]->linkDoor (temp[6], 1);
	temp[7]->linkDoor (temp[8], 1);
	temp[8]->linkDoor (temp[9], 3);
	temp[8]->linkTeleport (temp[10]);
	temp[10]->linkDoor (temp[13], 0);
	temp[10]->linkDoor (temp[11], 2);
	temp[11]->linkDoor (temp[12], 0);
	temp[12]->linkDoor (temp[13], 3);

	// test level
	// 	level->rooms.push_back (new Room(objects, roomSet->findRoom(true, true, true, true, false), monsterHp));
	
	return level;
}

Level *createLevel(Resources *resources, Objects *objects, unsigned int numRooms, int monsterHp)
{
	const int up = 0x100;
	const int down = 0x200;
	const int left = 0x400;
	const int right = 0x800;
	const int start = 0x1000;
	const int used = 0x2000;
	const int teleport = 0x4000;
	const int side = 8;
	const int maxRooms = side * side;
	const int dr[4] = { -side, side, -1, 1 };
	const int df[4] = { up, down, left, right };
	const int dopp[4] = { 1, 0, 3, 2 };
	vector<int> rooms;
	int matrix[maxRooms];	
	if (numRooms > maxRooms) numRooms = maxRooms;
	for (int i = 0; i < maxRooms; ++i) matrix[i] = 0;
		
	{
		int x = rand () % maxRooms;
		
		matrix[x] |= start | used;
		rooms.push_back (x);
	}
	
	while (rooms.size() < numRooms)
	{
		bool ok = false;
		int src;
		int dest;
		int d;
		// look for a room (src) with a connection (d) to an unoccupied space (dest).
		while (!ok)
		{
			int r = rand() % rooms.size();
			d = (rand() >> 4) % 4;
			src = rooms[r];
			dest = (src + dr[d]);
			while (dest < 0) dest += maxRooms;
			dest %= maxRooms;	
			if (matrix[dest] == 0)
			{
				ok = true;
			}			
		}
		
		// add door
		matrix[dest] |= used;
		rooms.push_back (dest);
		
		// create link
		int o = dopp[d]; // dir opposite to d
		matrix[src] |= df[d]; // flag for this dir
		matrix[dest] |= df[o]; // flag for opposite dir
	}
	
	// add some teleporters
	for (unsigned int i = 0; i < numRooms; i += 4) // one teleporter per 4 rooms
	{
		bool ok = false;
		int src = 0, dest = 0;
		while (!ok)
		{
			int rsrc = rand() % rooms.size();
			src = rooms[rsrc];
			if (!(matrix[src] & teleport))
			{
				int rdest = rand() % rooms.size();
				dest = rooms[rdest];
				if (dest != src && rsrc != rdest && 
						(!(matrix[dest] & teleport))
					)
				{
					ok = true;
				}
			}
		}
		assert (!(matrix[src] & teleport));
		assert (!(matrix[dest] & teleport));
		assert (src < maxRooms);
		assert (dest < maxRooms);
		assert (src >= 0);
		assert (dest >= 0);
		matrix[src] |= dest | teleport;
		matrix[dest] |= src | teleport;
	}
	
	Level *level = new Level();
	RoomSet *roomSet = resources->getRoomSet ("start");

	Room *matrix2[maxRooms];
	for (int i = 0; i < maxRooms; ++i) matrix2[i] = NULL;
	for (vector<int>::iterator i = rooms.begin(); i != rooms.end(); ++i)
	{
		int src = (*i);
		assert (matrix[src] > 0);
		RoomInfo *ri = roomSet->findRoom(
			matrix[src] & up, matrix[src] & down, 
			matrix[src] & left, matrix[src] & right, matrix[src] & teleport);
		if (!ri)
		{
			allegro_message ("Couldn't find roominfo for %i", matrix[src]);
		}
		assert (ri);
		
		Room *temp = new Room(objects, ri, monsterHp);
		assert (temp);
		
		matrix2[src] = temp;
		level->rooms.push_back(temp);
		
		for (int d = 0; d < 4; ++d)
		{
			int dest = (src + dr[d]);
			while (dest < 0) dest += maxRooms;
			dest %= maxRooms;	
			if ((matrix[src] & df[d]) && matrix2[dest])
			{
				assert (matrix[dest] > 0);
				// create door link
				matrix2[src]->linkDoor (matrix2[dest], d);
			}
		}
		if (matrix[src] & teleport)
		{
			int dest = matrix[src] & 0xFF;
			assert ((matrix[dest] & 0xFF) == src);
			if (matrix2[dest])
			{
				// create teleport link
				matrix2[src]->linkTeleport (matrix2[dest]);
			}
		}
	}
	
	return level;
}

void RoomSet::loadFromXml (DomNode *n, Resources *res, map<string, RoomSet*> &result)
{
	assert (n->name == "roomset");
	RoomSet *temp;
	string name = n->attributes["name"];
	vector<DomNode>::iterator i;
	vector<DomNode>::iterator h;
	temp = new RoomSet();
	
	for (h = n->children.begin(); h != n->children.end(); ++h)
	{
		assert (h->name == "room");
		RoomInfo ri;
		ri.map = res->getMap(h->attributes["map"]);
			
		for (i = h->children.begin(); i != h->children.end(); ++i)
		{
			ObjectInfo oi;
			oi.x = atoi(i->attributes["x"].c_str());
			oi.y = atoi(i->attributes["y"].c_str());
			if (i->name == "door")
			{
				oi.type = ObjectInfo::DOOR;
				string dir = i->attributes["dir"];
				if (dir == "up") { oi.doorDir = 0; ri.up = true; }
				else if (dir == "down") { oi.doorDir = 1; ri.down = true; }
				else if (dir == "left") { oi.doorDir = 2; ri.left = true; }
				else if (dir == "right") { oi.doorDir = 3; ri.right = true; }
				else assert (false);
			}
			else if (i->name == "monster")
			{
				oi.type = ObjectInfo::MONSTER;
				oi.monsterType = atoi(i->attributes["type"].c_str());
			}
			else if (i->name == "banana")
			{
				oi.type = ObjectInfo::BANANA;
				ri.bananas++;
			}
			else if (i->name == "teleporter")
			{
				oi.type = ObjectInfo::TELEPORT;
				ri.teleport = true;
			}
			else if (i->name == "player")
			{
				oi.type = ObjectInfo::PLAYER;
				oi.pi = atoi(i->attributes["pi"].c_str());
				ri.playerStart = true;
			}
			else
			{
				assert (false); // wrong tag
			}
			ri.objectInfo.push_back(oi);
		}
		temp->rooms.push_back (ri);
	}	
	result.insert (pair<string, RoomSet*>(name, temp));
}

RoomInfo *RoomSet::findRoom (bool up, bool down, bool left, bool right, bool teleport)
{
	RoomInfo *result = NULL;
	vector <RoomInfo>::iterator i;
	for (i = rooms.begin(); i != rooms.end(); ++i)
	{
		if (i->up == up && i->down == down && i->left == left && i->right == right && i->teleport == teleport)
		{
			result = &(*i);
			break;
		}	
	}
	return result;
}

Room::Room (Objects *o, RoomInfo *ri, int monsterHp) : roomInfo(ri), objects (o), map (NULL)
{
	doors[0] = NULL;
	doors[1] = NULL;
	doors[2] = NULL;
	doors[3] = NULL;
	teleport = NULL;
	map = ri->map;
	
	// add monsters, doors, etc. (but do not link doors yet)
	vector <ObjectInfo>::iterator i;
	for (i = ri->objectInfo.begin(); i != ri->objectInfo.end(); ++i)
	{
		switch (i->type)
		{
			case ObjectInfo::DOOR:
				{
					Door *d = new Door (this);
					d->setDir ((Dir)i->doorDir);
					d->setLocation ((fix)i->x * 32, (fix)i->y * 32);
					objects->add (d);
					doors[i->doorDir] = d;
				}
				break;
			case ObjectInfo::TELEPORT:
				{
					Teleport *t = new Teleport (this);
					t->setLocation ((fix)i->x * 32, (fix)i->y * 32);
					objects->add (t);
					teleport = t;
				}
				break;
			case ObjectInfo::MONSTER:
				{
					Monster *m = new Monster (this, i->monsterType, monsterHp);
					m->setLocation ((fix)i->x * 32, (fix)i->y * 32);
					objects->add (m);
				}
				break;
			case ObjectInfo::BANANA:
				{
					Banana *b = new Banana (this);
					b->setLocation ((fix)i->x * 32, (fix)i->y * 32);
					objects->add (b);
				}
				break;
			case ObjectInfo::PLAYER:
				break;
		}
	}
}

void Room::linkDoor (Room *otherRoom, int dir)
{
	const int opposite[4] = {1, 0, 3, 2};
	int odir = opposite[dir];
	assert (doors[dir]);
	assert (otherRoom);
	assert (otherRoom->doors[odir]);
	doors[dir]->link(otherRoom->doors[odir]);
}

void Room::linkTeleport (Room *otherRoom)
{
	assert (teleport);
	assert (otherRoom);
	assert (otherRoom->teleport);
	teleport->link(otherRoom->teleport);	
}

Level::~Level()
{
	vector<Room *>::iterator i;
	for (i = rooms.begin(); i != rooms.end(); ++i)
	{
		delete (*i);
	}
}

int Level::getBananaCount()
{
	int result = 0;
	vector <Room *>::iterator i;
	for (i = rooms.begin(); i != rooms.end(); ++i)
	{
		result += (*i)->getBananaCount();
	}
	return result;
}

int Room::getBananaCount()
{
	assert (roomInfo);
	return roomInfo->bananas;
}
