/*  TA3D, a remake of Total Annihilation
    Copyright (C) 2005  Roland BROCHARD

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA*/

/*-----------------------------------------------------------------------------------\
|                                        music.h                                     |
|  ce fichier contient les structures, classes et fonctions nécessaires à la lecture |
| des fichiers musicaux (mp3/ogg) pendant le jeu en fonction de l'état de la partie. |
| La liste de lecture est créée automatiquement s'il n'y en a pas et est modifiable. |
|                                                                                    |
\-----------------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>

#if USE_MP3					// Fichiers nécéssaires au décodage de la musique au format MP3
#include "../include/almp3.h"
#endif

#if USE_OGG					// Fichiers nécessaires au décodage de la musique au format OGG
#include <alogg/alogg.h>
#endif

#if USE_MP3 || USE_OGG		// Code du gestionnaire de musique

typedef unsigned char byte;

#define MUSIC_TYPE_NONE		0x0
#define MUSIC_TYPE_BATTLE	0x1

#define MUSIC_NONE			0x0
#define MUSIC_MP3			0x1		// Type de fichier musical
#define MUSIC_OGG			0x2

class MUSIC_STREAM			// Objet pour lire de la musique sous forme de flux depuis les fichiers musicaux
{
public:
	byte			type;		// Type d'objet

	int				voice;

	byte			*data;		// Données nécessaires à la lecture des mp3
	PACKFILE		*file;
#if USE_MP3
	ALMP3_MP3STREAM	*mp3;
#endif

#if USE_OGG
	alogg_stream	*ogg;		// Données nécessaires à la lecture des ogg
#endif

	void init()
	{
		type=MUSIC_NONE;
		data=NULL;
		file=NULL;
#if USE_MP3
		mp3=NULL;
#endif
#if USE_OGG
		ogg=NULL;
#endif
	}

	void destroy()
	{
		switch(type)
		{
		case MUSIC_NONE:
			if(data)	delete data;
			if(file)	pack_fclose(file);
			break;
		case MUSIC_MP3:
#if USE_MP3
			almp3_destroy_mp3stream(mp3);
#endif
			if(data)	delete data;
			if(file)	pack_fclose(file);
			break;
		case MUSIC_OGG:
#if USE_OGG
			alogg_stop_streaming(ogg);
#endif
			break;
		};
		init();
	}

	MUSIC_STREAM()
	{
		init();
	}

	~MUSIC_STREAM()
	{
		destroy();
	}

	void poll();

	void stop()
	{
		if(type!=MUSIC_NONE)
			voice_ramp_volume(voice, 2000, 0);
	}

	void play(char *filename);
};

class MUSIC_MANAGER
{
public:
	int				nb_files;		// Nombre de fichiers lisibles
	char			**list;			// Liste de lecture
	byte			*type;			// Type des morceaux (bataille, calme, ...)
	MUSIC_STREAM	cur;			// Flux en cour de lecture
	MUSIC_STREAM	old;			// Ancien flux, pour faire une transition en douceur
	byte			game_state;		// Etat de la partie pour ne pas lancer n'importe quoi

	void init()
	{
		nb_files=0;
		list=NULL;
		type=NULL;
		cur.init();
		old.init();
		game_state=MUSIC_TYPE_NONE;
	}

	void destroy()
	{
		if(list) {
			for(int i=0;i<nb_files;i++)
				if(list[i])	free(list[i]);
			delete list;
			}
		if(type)	delete type;
		cur.destroy();
		old.destroy();
		nb_files=0;
		init();
	}

	MUSIC_MANAGER()
	{
		init();
	}

	~MUSIC_MANAGER()
	{
		destroy();
	}

	void load_play_list();

	void save_play_list();

	void poll();

	void change_state(byte new_state);
};

extern MUSIC_MANAGER	music_manager;

#endif
