/*  TA3D, a remake of Total Annihilation
    Copyright (C) 2005  Roland BROCHARD

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA*/

/*-----------------------------------------------------------------------------------\
|                                        sound.h                                     |
|   Ce module contient les structures, classes et fonctions nécessaires à la gestion |
| des sons.                                                                          |
|                                                                                    |
\-----------------------------------------------------------------------------------*/

#include <string.h>
#include <stdio.h>
#include "console.h"
#include "hpi.h"
#include "vector.h"

SAMPLE *load_memory_wav(byte *data,int size);

class SOUND_MANAGER
{
private:
	int			nb_sound;		// Nombre de sons joués
	SAMPLE		*sample[128];	// Sons joués par Allegro
	int			voice[128];		// Voix allouées pour jouer les sons
	VECTOR		pos[128];		// Position des sources sonores
	bool		sound3d[128];	// Indique si le son est localisé dans l'espace

	SAMPLE		*cache[256];	// Mémoire cache pour éviter les coups de lag durant les parties chargées
	int			cache_size;
	char		*cache_name[256];

public:
	void init()
	{
		nb_sound=0;
		for(int i=0;i<128;i++)
			sample[i]=NULL;

		cache_size=0;
		for(int i=0;i<256;i++) {
			cache[i]=NULL;
			cache_name[i]=NULL;
			}
	}

	void destroy()
	{
		for(int i=0;i<nb_sound;i++)
			destroy_sample(sample[i]);

		for(int i=0;i<cache_size;i++) {
			destroy_sample(cache[i]);
			free(cache_name);
			}
		init();
	}

	inline SAMPLE *duplicate_sample(SAMPLE *smp)
	{
		if(smp==NULL)	return NULL;
		SAMPLE *copy=create_sample(smp->bits,smp->stereo,smp->freq,smp->len);
		memcpy(copy->data,smp->data,smp->stereo!=0 ? (smp->len*smp->bits>>2) : (smp->len*smp->bits>>3));		// Copie les données sonores
		return copy;
	}

	inline int is_in_cache(char *name)
	{
		for(int i=0;i<cache_size;i++)
			if(strcasecmp(name,cache_name[i])==0)
				return i;
		return -1;
	}

	inline void put_in_cache(char *name,SAMPLE *smp)
	{
		if(is_in_cache(name)==-1)	return;		// Pas la peine de le mettre 2 fois!!
		if(cache_size>=256) {
			cache_size=256;
			destroy_sample(cache[0]);
			free(cache_name[0]);
			cache_size--;
			for(int i=0;i<cache_size;i++) {
				cache[i]=cache[i+1];
				cache_name[i]=cache_name[i+1];
				}
			}
		cache[cache_size]=duplicate_sample(smp);
		cache_name[cache_size++]=strdup(name);
	}

	inline SAMPLE *get_from_cache(char *name)
	{
		int idx=is_in_cache(name);
		if(idx==-1)	return NULL;
		return duplicate_sample(cache[idx]);
	}

	SOUND_MANAGER()
	{
		init();
	}

	~SOUND_MANAGER()
	{
		destroy();
	}

	inline void clean(VECTOR cam_pos,VECTOR side)
	{
		int e=0;
		for(int i=0;i+e<nb_sound;) {
			if(voice_get_position(voice[i+e])==-1) {			// Le son a fini d'être joué
				destroy_sample(sample[i+e]);
				e++;
				}
			else {
				sample[i]=sample[i+e];
				voice[i]=voice[i+e];
				pos[i]=pos[i+e];
				i++;
				}
			}
		nb_sound-=e;
		VECTOR o_d=cam_pos+side;
		VECTOR o_g=cam_pos-side;
		for(int i=0;i<nb_sound;i++) {
			if(sound3d[i]) {
				float i_d=25500000.0f/(pos[i]-o_d).Sq();
				float i_g=25500000.0f/(pos[i]-o_g).Sq();
				int volume=(int)(i_d+i_g);
				int pan= (volume==0) ? 0 : (int)(255*(i_d/(i_d+i_g)));
				if(volume>255)	volume=255;
				voice_set_volume(voice[i],volume);
				voice_set_pan(voice[i],pan);
				}
			else {
				voice_set_volume(voice[i],255);
				voice_set_pan(voice[i],127);
				}
			}
	}

	void play(char *filename,VECTOR Pos,bool s3d=true);
};

extern SOUND_MANAGER sound_manager;
