#include "list.h"

#include <stdlib.h>

/* List Init
 * Prepare the the list for use.
 */
void list_init(list_t *list)
{
   list->data = NULL;
   list->next = NULL;
}

/* List Push Data
 * Push an item onto the list.
 */
void *list_push_data(list_t **list, void *data, list_t *pos)
{
   list_t *node = NULL;

   if ((node = (list_t *)malloc(sizeof(list_t))) == NULL) {
      return NULL;
   }
   node->data = data;
   node->next = NULL;
   list_push(list, node, pos);

   return data;
}

/* List Pop Data
 * Pop an item off the list.
 */
void *list_pop_data(list_t **list, void *data)
{
   list_t *node = NULL;
   void *temp = NULL;

   node = *list;
   while (node) {
      if (node->data == data) {
         break;
      }
      node = (list_t *)node->next;
   }

   if (node != NULL) {
      temp = node->data;
      list_pop(list, node);
      free((list_t *)node);
   }

   return temp;
}

/* List Push
 * Push a list node onto the list.
 */
list_t *list_push(list_t **list, list_t *node, list_t *pos)
{
   list_t *current;
   list_t *prev;

   if (*list) {
      current = *list;
      prev = NULL;
      while (current != pos && current != NULL) {
         prev = current;
         current = current->next;
      }
      if (prev) {
         prev->next = node;
      }
      if (current == *list) {
         *list = node;
      }
      node->next = current;
   }
   else {
      *list = node;
   }

   return node;
}

/* List Pop
 * Pop a list node off the list.
 */
list_t *list_pop(list_t **list, list_t *node)
{
   list_t *pos;

   if (node == *list) {
      *list = node->next;
      node->next = NULL;
   }
   else {
      pos = *list;
      while (pos->next != node) {
         pos = pos->next;
      }
      pos->next = node->next;
      node->next = NULL;
   }

   return node;
}

/* List Clean
 * Clean up list nodes.
 */
void list_clean(list_t **list, void (*cleaner)(void *data))
{
   list_t *current = NULL;
   list_t *temp = NULL;

   current = *list;
   while (current != NULL) {
      temp = (list_t *)current->next;
      if (cleaner) {
         cleaner(current->data);
      }
      list_pop(list, current);
      free(current);
      current = temp;
   }
}

/* List Read
 * Read the contents of the list from the file.
 */
int list_read(list_t **list, FILE *file, int size, int (*reader)(void *data, FILE *file))
{
   int count;
   int i;
   void *data = NULL;
   list_t *node = NULL;

   if (fread(&count, sizeof(int), 1, file) < 1) {
      return 1;
   }

   for (i = 0; i < count; i++) {
      data = NULL;
      if ((data = malloc(size)) == NULL) {
         return 1;
      }
      if (reader(data, file)) {
         free(data);
         return 1;
      }
      node = NULL;
      if ((node = malloc(sizeof(list_t))) == NULL) {
         return 1;
      }
      list_init(node);
      node->data = (void *)data;
      list_push(list, node, NULL);
   }

   return 0;
}

/* List Write
 * Write the contents of the list to the file.
 */
int list_write(list_t **list, FILE *file, int (*writer)(void *data, FILE *file))
{
   int count;
   list_t *node = NULL;

   count = list_count(list);
   if (fwrite(&count, sizeof(int), 1, file) < 1) {
      return 1;
   }

   node = *list;
   while (node != NULL) {
      if (writer(node->data, file)) {
         return 1;
      }
      node = node->next;
   }

   return 0;
}

/* List Count
 * Count the number of list nodes.
 */
int list_count(list_t **list)
{
   list_t *node = NULL;
   int count;

   count = 0;
   node = *list;
   while (node != NULL) {
      count++;
      node = (list_t *)node->next;
   }

   return count;
}

/* List Pos
 * Gets the position of the list node.
 */
int list_pos(list_t **list, list_t *node)
{
   list_t *current = NULL;
   int pos;

   pos = 0;
   current = *list;
   while (current != NULL && current != node) {
      pos++;
      current = (list_t *)current->next;
   }

   if (current == NULL) {
      return -1;
   }

   return pos;
}

/* List Data Pos
 * Gets the position of the item.
 */
int list_data_pos(list_t **list, void *data)
{
   list_t *current = NULL;
   int pos;

   pos = 0;
   current = *list;
   while (current != NULL && current->data != data) {
      pos++;
      current = (list_t *)current->next;
   }

   if (current == NULL) {
      return -1;
   }

   return pos;
}

/* List Create
 * Allocate memory for a new list.
 */
list_t *list_create(void)
{
   list_t *list = NULL;

   if ((list = malloc(sizeof(list_t))) == NULL) {
      return NULL;
   }
   list_init(list);

   return list;
}

/* List Destroy
 * Deallocate memory for a new list.
 */
void list_destroy(list_t **list)
{
   if (*list != NULL) {
      free(*list);
      *list = NULL;
   }
}

