#include "generate.h"

#include <stdlib.h>
#include <math.h>
#include "..\math.h"

#include "perlin.h"
#include "interp.h"

#define RAN(max) (((double)rand() / (double)RAND_MAX) * (double)max)
#define DIST(x1, y1, x2, y2) (sqrt(pow((x1) - (x2), 2) + pow((y1) - (y2), 2)))

void point_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j;

   i = x;
   j = y;

   if (contain_rect(&i, &j, hmap->w, hmap->h, contain)) {
      hmap->squares[(int)i][(int)j] += power * size;
   }
}

void cone_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            d = DIST(x, y, i, j);
            if (fabs(d) <= size) {
               hmap->squares[k][l] += power * (size - d);
            }
         }
      }
   }
}

void bump_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            d = DIST(x, y, i, j);
            if (fabs(d) <= size) {
               hmap->squares[k][l] += (power / 2 + cos((d / size) * M_PI) * power / 2) * size;
            }
         }
      }
   }
}

void peak_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            d = DIST(x, y, i, j);
            if (fabs(d) <= size) {
               hmap->squares[k][l] += (size + cos((d / size) / 2 * M_PI + 0.5 * M_PI) * size) * power;
            }
         }
      }
   }
}

void hemisphere_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            d = DIST(x, y, i, j);
            if (fabs(d) <= size) {
               hmap->squares[k][l] += power * sqrt(pow(size, 2) - pow(d, 2));
            }
         }
      }
   }
}

void inverse_hemisphere_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            d = DIST(x, y, i, j);
            if (fabs(d) <= size) {
               hmap->squares[k][l] += power * (sqrt(pow(size, 2) - pow(size - d, 2)) - size);
            }
         }
      }
   }
}

void crater_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            d = DIST(x, y, i, j);
            if (fabs(d) <= size) {
               hmap->squares[k][l] += (sin(d / size * M_PI) * power) * size;
            }
         }
      }
   }
}

void circle_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            d = DIST(x, y, i, j);
            if (fabs(d) <= size) {
               hmap->squares[k][l] += power * size;
            }
         }
      }
   }
}

void fault_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j;
   float fault_angle;
   float m, c;
   float d_x, d_y;

   fault_angle = RAN(M_PI * 2);

   d_y = 1.0;
   d_x = d_y / tan(fault_angle);

   m = d_x / d_y;

   c = y - m * x;

   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         if (j < (m * i + c)) {
            hmap->squares[i][j] += (float)power * size;
         }
      }
   }
}

void square_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            hmap->squares[k][l] += size * power;
         }
      }
   }
}

void pyramid_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;
   float xx, yy;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            xx = size - fabs(x - i);
            yy = size - fabs(y - j);
            hmap->squares[k][l] += power * ((xx < yy) ? xx : yy);
         }
      }
   }
}

void diamond_collager(hmap_t *hmap, float x, float y, float size, float power, int contain)
{
   int i, j, k, l;
   float d;
   int start_x, start_y, end_x, end_y;
   float xx, yy;

   start_x = x - size;
   start_y = y - size;
   end_x = x + size;
   end_y = y + size;

   for (i = start_x; i <= end_x; i++) {
      for (j = start_y; j <= end_y; j++) {
         k = i;
         l = j;
         if (contain_rect(&k, &l, hmap->w, hmap->h, contain)) {
            xx = size - fabs(x - i);
            yy = size - fabs(y - j);
            if ((xx + yy) / 2 >= size / 2) {
               hmap->squares[k][l] += power * ((xx + yy) / 2 - size / 2);
            }
         }
      }
   }
}

void hmap_collage_reduce(hmap_t *hmap, hmap_collager_t method, int number, float size, int contain)
{
   int i;
   float x, y;
   float power;

   for (i = 0; i < number; i++) {
      power = (rand() % 2 == 0) ? -1 : 1;

      x = rand() % hmap->w;
      y = rand() % hmap->h;

      method(hmap, x, y, ((double)(number - i) / (double)number) * size, power, contain);
   }
}

void hmap_collage(hmap_t *hmap, hmap_collager_t method, int number, float size, int contain)
{
   int i;
   float x, y;
   float power;

   for (i = 0; i < number; i++) {
      power = (rand() % 2 == 0) ? -1 : 1;

      x = rand() % hmap->w;
      y = rand() % hmap->h;

      method(hmap, x, y, size, power, contain);
   }
}

void hmap_subdivide(hmap_t *hmap, float x, float y, float w, float h, float range, float roughness)
{
   float half_w, half_h;
   float half_roughness;
   float mid_val, top_val, right_val, bottom_val, left_val;

   if ((w < 1.0) && (h < 1.0)) {
      return;
   }

   half_w = w / 2;
   half_h = h / 2;

   half_roughness = roughness / 2;

   mid_val = ((hmap->squares[(int)x][(int)y] +
                   hmap->squares[(int)(x + w)][(int)y] +
                   hmap->squares[(int)(x + w)][(int)(y + h)] +
                   hmap->squares[(int)x][(int)(y + h)]) / 4) + (RAN(range * roughness) - (range * roughness / 2.0));
   hmap->squares[(int)(x + half_w)][(int)(y + half_h)] = mid_val;

   top_val = ((hmap->squares[(int)x][(int)y] +
               hmap->squares[(int)(x + w)][(int)y]) / 2);
   hmap->squares[(int)(x + w / 2)][(int)(y)] = top_val;

   left_val = ((hmap->squares[(int)x][(int)(y + h)] +
                hmap->squares[(int)x][(int)y]) / 2);
   hmap->squares[(int)(x)][(int)(y + h / 2)] = left_val;

   right_val = ((hmap->squares[(int)(x + w)][(int)y] +
                 hmap->squares[(int)(x + w)][(int)(y + h)]) / 2);
   hmap->squares[(int)(x + w)][(int)(y + h / 2)] = right_val;

   bottom_val = ((hmap->squares[(int)(x + w)][(int)(y + h)] +
                  hmap->squares[(int)x][(int)(y + h)]) / 2);
   hmap->squares[(int)(x + w / 2)][(int)(y + h)] = bottom_val;

   hmap_subdivide(hmap, x, y, half_w, half_h, range, half_roughness);
   hmap_subdivide(hmap, x + half_w, y, half_w, half_h, range, half_roughness);
   hmap_subdivide(hmap, x + half_w, y + half_h, half_w, half_h, range, half_roughness);
   hmap_subdivide(hmap, x, y + half_h, half_w, half_h, range, half_roughness);
}

void hmap_subdivision(hmap_t *hmap, float range, float roughness)
{
   int x, y;
   int div_w, div_h;
   int x1, y1, x2, y2;
   int mid_x, mid_y;
   int mid_val, top_val, left_val, right_val, bottom_val;

   for (div_w = hmap->w, div_h = hmap->h; div_w > 1 && div_h > 1; div_w /= 2, div_h /= 2)  {
      for (x = 0; x < hmap->w; x += div_w) {
         for (y = 0; y < hmap->h; y += div_h) {
            x1 = x;
            y1 = y;
            x2 = x1 + div_w;
            y2 = y1 + div_h;
            mid_x = x1 + div_w / 2;
            mid_y = y1 + div_h / 2;

            mid_val = ((hmap->squares[x1][y1] +
                        hmap->squares[x2][y1] +
                        hmap->squares[x2][y2] +
                        hmap->squares[x1][y2]) / 4) + (RAN(range * roughness) - (range * roughness / 2.0));
            hmap->squares[mid_x][mid_y] = mid_val;

            top_val = ((hmap->squares[x1][y1] +
                        hmap->squares[x2][y1]) / 2);
            hmap->squares[mid_x][y1] = top_val;

            bottom_val = ((hmap->squares[x1][y2] +
                        hmap->squares[x2][y2]) / 2);
            hmap->squares[mid_x][y2] = bottom_val;

            left_val = ((hmap->squares[x1][y1] +
                        hmap->squares[x1][y2]) / 2);
            hmap->squares[x1][mid_y] = left_val;

            right_val = ((hmap->squares[x2][y1] +
                        hmap->squares[x2][y2]) / 2);
            hmap->squares[x2][mid_y] = right_val;
         }
      }
   }
}

void hmap_deposition3(hmap_t *hmap, hmap_collager_t method, int number, int size, double dist, double bsize, int contain)
{
   int i, j;
   int x, y;
   float power;

   for (i = 0; i < number; i++) {
      x = (float)rand() / (float)RAND_MAX * hmap->w;
      y = (float)rand() / (float)RAND_MAX * hmap->h;
      power = (rand() % 2 == 0) ? -1 : 1;
      for (j = 0; j < size; j++) {
         method(hmap, x, y, bsize, power, contain);
         switch (rand() % 4) {
         case 0:
            x += dist;
            break;
         case 1:
            x -= dist;
            break;
         case 2:
            y += dist;
            break;
         case 3:
            y -= dist;
            break;
         }
         while (x < 0) x += hmap->w;
         while (x >= hmap->w) x -= hmap->w;
         while (y < 0) y += hmap->h;
         while (y >= hmap->h) y -= hmap->h;
      }
   }
}

void hmap_deposition2(hmap_t *hmap, hmap_collager_t method, int number, int size, double dist, double bsize, int contain)
{
   int i, j;
   int x, y;
   double dx, dy;
   double angle;
   float power;

   for (i = 0; i < number; i++) {
      x = (float)rand() / (float)RAND_MAX * hmap->w;
      y = (float)rand() / (float)RAND_MAX * hmap->h;
      power = (rand() % 2 == 0) ? -1 : 1;
      for (j = 0; j < size; j++) {
         method(hmap, x, y, bsize, power, contain);
         angle = RAN(2 * M_PI);
         dx = sin(angle) * dist;
         dy = cos(angle) * dist;
         x += dx;
         y += dy;
         while (x < 0) x += hmap->w;
         while (x >= hmap->w) x -= hmap->w;
         while (y < 0) y += hmap->h;
         while (y >= hmap->h) y -= hmap->h;
      }
   }
}

void hmap_deposition4(hmap_t *hmap, hmap_collager_t method, int number, int size, double dist, double bsize, int contain)
{
   int i, j;
   int x, y;
   double dx, dy;
   double angle;
   float power;
   float d;

   for (i = 0; i < number; i++) {
      x = (float)rand() / (float)RAND_MAX * hmap->w;
      y = (float)rand() / (float)RAND_MAX * hmap->h;
      power = (rand() % 2 == 0) ? -1 : 1;
      for (j = 0; j < size; j++) {
         method(hmap, x, y, bsize * RAN(1.0), power, contain);
         angle = RAN(2 * M_PI);
         d = dist * RAN(1.0);
         dx = sin(angle) * d;
         dy = cos(angle) * d;
         x += dx;
         y += dy;
         while (x < 0) x += hmap->w;
         while (x >= hmap->w) x -= hmap->w;
         while (y < 0) y += hmap->h;
         while (y >= hmap->h) y -= hmap->h;
      }
   }
}

void hmap_faulting(hmap_t *hmap, int number, float range)
{
   int i;
   float fault_x, fault_y;
   float fault_angle;
   int x, y;
   int temp;
   float change;
   float m, c;
   float d_x, d_y;

   for (i = 0; i < number; i++) {
      fault_x = rand() % hmap->w;
      fault_y = rand() % hmap->h;
//      fault_x = RAN(hmap->w);
//      fault_y = RAN(hmap->h);

      fault_angle = RAN(M_PI * 2);

      d_y = 1.0;
      d_x = d_y / tan(fault_angle);

      m = d_x / d_y;

      c = fault_y - m * fault_x;

//      change = (RAN(range) - (range / 2));
      change = rand() % 2 == 0 ? -1 : 1;

      for (x = 0; x < hmap->w; x++) {
         for (y = 0; y < hmap->h; y++) {
            if (y < (m * x + c)) {
               hmap->squares[x][y] += change;
            }
         }
      }
   }
}

void hmap_fault(hmap_t *hmap, int number, float range)
{
   int i;
   int x, y;
   float change;

   float v, a, b, d, c;

   for (i = 0; i < number; i++) {
      v = rand();
      a = sin(v);
      b = cos(v);
      d = sqrt(hmap->w * hmap->w + hmap->h * hmap->h);
//      c = ((float)rand() / (float)RAND_MAX) * d - d / 2;
      c = ((float)rand() / (float)RAND_MAX) * d;

//      change = 1;
      change = rand() % 2 == 0 ? -1 : 1;

      for (x = 0; x < hmap->w; x++) {
         for (y = 0; y < hmap->h; y++) {
            if (a * x + b * y - c > 0) {
               hmap->squares[x][y] += change;
            }
         }
      }
   }
}

void hmap_fault2(hmap_t *hmap, int number, float range)
{
   int i;
   int x, y;
   float change;
   float x1, y1, x2, y2;

   float a, b, c;

   for (i = 0; i < number; i++) {
      x1 = (float)rand() / (float)RAND_MAX * hmap->w;
      y1 = (float)rand() / (float)RAND_MAX * hmap->h;
      x2 = (float)rand() / (float)RAND_MAX * hmap->w;
      y2 = (float)rand() / (float)RAND_MAX * hmap->h;

      a = (y2 - y1);
      b = -(x2 - x1);
      c = -x1 * (y2 - y1) + y1 * (x2 - x1);

      change = 1;

      for (x = 0; x < hmap->w; x++) {
         for (y = 0; y < hmap->h; y++) {
            if (a * x + b * y - c > 0) {
               hmap->squares[x][y] += change;
            }
         }
      }
   }
}

void hmap_frequency_composition(hmap_t *hmap, int number)
{
   int x, y;
   int i;
   float offset_x, offset_y;
   float frequency_x, frequency_y;
   float amplitude_x, amplitude_y;
   float period_x, period_y;

   period_x = hmap->w / (M_PI * 2);
   period_y = hmap->h / (M_PI * 2);

   for (i = 0; i < number; i++) {
//      frequency_x = 1 / RAN(period_x * 2);
//      frequency_y = 1 / RAN(period_y * 2);
      frequency_x = 1 / RAN(period_x * 2);
      frequency_y = frequency_x;
      offset_x = RAN(period_x);
      offset_y = RAN(period_y);
//      amplitude_x = 1 / frequency_x;
//      amplitude_y = 1 / frequency_y;
      amplitude_x = 1 / sqrt(frequency_x);
      amplitude_y = 1 / sqrt(frequency_y);
      for (x = 0; x < hmap->w; x++) {
         for (y = 0; y < hmap->h; y++) {
           hmap->squares[x][y] += amplitude_x * sin(x * frequency_x + offset_x) +
                                  amplitude_y * sin(y * frequency_y + offset_y);
         }
      }
   }
}

void hmap_perlin(hmap_t *hmap, double persistence, int octaves)
{
   int i, j;
   double pos[2];
   double buf[16];
   double off_x, off_y;
   double val;

   off_x = rand() % 65536;
   off_y = rand() % 65536;

   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         pos[0] = (float)i / (float)hmap->w + off_x;
         pos[1] = (float)j / (float)hmap->h + off_y;
//         val = perlin_noise(linear_interp, 2, 0, pos, persistence, octaves, buf);
//         val = perlin_noise(cosine_interp, 2, 0, pos, persistence, octaves, buf);
         val = perlin_noise(cubic_interp, 4, -1, pos, persistence, octaves, buf);
         hmap->squares[i][j] += val;
      }
   }
}

