#include "tile.h"

#include <string.h>

void tile_init(tile_t *tile)
{
   tile->bitmap = NULL;
}

int tile_create_bitmap(tile_t *tile)
{
   if ((tile->bitmap = create_bitmap(tile->w, tile->h)) == NULL) {
      return 1;
   }

   return 0;
}

void tile_destroy_bitmap(tile_t *tile)
{
   if (tile->bitmap != NULL) {
      destroy_bitmap(tile->bitmap);
      tile->bitmap = NULL;
   }
}

tile_t *tile_create(void)
{
   tile_t *tile = NULL;

   if ((tile = malloc(sizeof(tile_t))) == NULL) {
      return NULL;
   }
   tile_init(tile);

   return tile;
}

void tile_destroy(tile_t *tile)
{
   if (tile != NULL) {
      tile_destroy_bitmap(tile);
      free(tile);
      tile = NULL;
   }
}

int tile_write(const tile_t *tile, FILE *file)
{
   if (fprintf(file, "%4d, %4d, %4d, %4d, %4d, %4d\n", tile->x, tile->y, tile->w, tile->h, tile->focus_x, tile->focus_y) < 6) {
      return 1;
   }

   return 0;
}

int tile_read(tile_t *tile, FILE *file)
{
   if (fscanf(file, "%d, %d, %d, %d, %d, %d\n", &tile->x, &tile->y, &tile->w, &tile->h, &tile->focus_x, &tile->focus_y) < 6) {
      return 1;
   }

   return 0;
}

void tile_set_init(tile_set_t *tile_set)
{
   int i;

   tile_set->size = 0;

   tile_set->tiles = NULL;

   for (i = 0; i < TILE_SET_NAME_SIZE; i++) {
      tile_set->name[i] = ' ';
   }
   tile_set->name[0] = '\0';

   for (i = 0; i < TILE_SET_DESC_SIZE; i++) {
      tile_set->desc[i] = ' ';
   }
   tile_set->desc[0] = '\0';

   for (i = 0; i < TILE_SET_BITMAP_SIZE; i++) {
      tile_set->bitmap[i] = ' ';
   }
   tile_set->bitmap[0] = '\0';
}

int tile_set_write(const tile_set_t *tile_set, FILE *file)
{
   int i;

   /* Name */
   if (fputs(tile_set->name, file) == EOF) {
      return 1;
   }
   if (fputc('\n', file) == EOF) {
      return 1;
   }

   /* Desc */
   if (fputs(tile_set->desc, file) == EOF) {
      return 1;
   }
   if (fputc('\n', file) == EOF) {
      return 1;
   }

   /* Bitmap */
   if (fputs(tile_set->bitmap, file) == EOF) {
      return 1;
   }
   if (fputc('\n', file) == EOF) {
      return 1;
   }

   /* Size */
   if (fprintf(file, "%d\n", tile_set->size) < 1) {
      return 1;
   }

   /* Tiles */
   for (i = 0; i < tile_set->size; i++) {
      if (tile_write(tile_set->tiles[i], file)) {
         return 1;
      }
   }

   return 0;
}

int tile_set_read(tile_set_t *tile_set, FILE *file)
{
   int i;
   BITMAP *bitmap = NULL;

   /* Name */
   if (fgets(tile_set->name, TILE_SET_NAME_SIZE, file) == NULL) {
      return 1;
   }
   tile_set->name[strlen(tile_set->name) - 1] = '\0';

   /* Desc */
   if (fgets(tile_set->desc, TILE_SET_DESC_SIZE, file) == NULL) {
      return 1;
   }
   tile_set->desc[strlen(tile_set->desc) - 1] = '\0';

   /* Bitmap */
   if (fgets(tile_set->bitmap, TILE_SET_BITMAP_SIZE, file) == NULL) {
      return 1;
   }
   tile_set->bitmap[strlen(tile_set->bitmap) - 1] = '\0';

   /* Size */
   if (fscanf(file, "%d\n", &tile_set->size) < 1) {
      return 1;
   }

   /* Load bitmap */
   if ((bitmap = load_bitmap(tile_set->bitmap, NULL)) == NULL) {
      return 1;
   }

   /* Tiles */
   if (tile_set_create_tiles(tile_set)) {
      destroy_bitmap(bitmap);
      return 1;
   }
   for (i = 0; i < tile_set->size; i++) {
      if ((tile_set->tiles[i] = tile_create()) == NULL) {
         destroy_bitmap(bitmap);
         tile_set_destroy_tiles(tile_set);
         return 1;
      }
      if (tile_read(tile_set->tiles[i], file)) {
         destroy_bitmap(bitmap);
         tile_set_destroy_tiles(tile_set);
         return 1;
      }
      if (tile_create_bitmap(tile_set->tiles[i])) {
         destroy_bitmap(bitmap);
         tile_set_destroy_tiles(tile_set);
         return 1;
      }
      blit(bitmap, tile_set->tiles[i]->bitmap, tile_set->tiles[i]->x, tile_set->tiles[i]->y, 0, 0, tile_set->tiles[i]->w, tile_set->tiles[i]->h);
   }

   /* Destroy bitmap */
   destroy_bitmap(bitmap);

   return 0;
}

tile_set_t *tile_set_load(const char *filename)
{
   tile_set_t *tile_set = NULL;
   FILE *file = NULL;

   if ((tile_set = tile_set_create()) == NULL) {
      return NULL;
   }
   if ((file = fopen(filename, "rt")) == NULL) {
      tile_set_destroy(tile_set);
      return NULL;
   }
   if (tile_set_read(tile_set, file)) {
      tile_set_destroy(tile_set);
      fclose(file);
      return NULL;
   }
   fclose(file);

   return tile_set;
}

int tile_set_save(tile_set_t *tile_set, const char *filename)
{
   FILE *file = NULL;

   if ((file = fopen(filename, "wt")) == NULL) {
      return 1;
   }
   if (tile_set_write(tile_set, file)) {
      fclose(file);
      return 1;
   }
   fclose(file);

   return 0;
}

tile_set_t *tile_set_create(void)
{
   tile_set_t *tile_set = NULL;

   if ((tile_set = malloc(sizeof(tile_set_t))) == NULL) {
      return NULL;
   }

   return tile_set;
}

void tile_set_destroy(tile_set_t *tile_set)
{
   if (tile_set != NULL) {
      tile_set_destroy_tiles(tile_set);
      free(tile_set);
      tile_set = NULL;
   }
}

int tile_set_create_tiles(tile_set_t *tile_set)
{
   int i;

   if ((tile_set->tiles = malloc(sizeof(tile_t *) * tile_set->size)) == NULL) {
      return 1;
   }
   for (i = 0; i < tile_set->size; i++) {
      tile_set->tiles[i] = NULL;
   }

   return 0;
}

void tile_set_destroy_tiles(tile_set_t *tile_set)
{
   int i;

   if (tile_set->tiles != NULL) {
      for (i = 0; i < tile_set->size; i++) {
         if (tile_set->tiles[i]->bitmap != NULL) {
            tile_destroy_bitmap(tile_set->tiles[i]);
         }
      }
      free(tile_set->tiles);
      tile_set->tiles = NULL;
   }
}


