/*
 * Lords of War
 * By Carl Olsson
 * February 2001
 */

#include "map.h"

#include <stdlib.h>
#include <allegro.h>

#include "tile.h"
#include "unit.h"

#include <math.h>
#include "math.h"
#include "perlin.h"
#include "interp.h"
#include "hmap\hmap.h"
#include "hmap\generate.h"
#include "hmap\postproc.h"

extern tile_set_t *terrain;

map_t *current_map = NULL;

void put_half_pixel(BITMAP *dest, int x, int y, int colour)
{
   if ((x + y) % 2 == 0) {
      putpixel(dest, x, y, colour);
   }
}

void square_draw(const square_t *square, const view_t *view, float x, float y)
{
   float view_x, view_y;
   int grid_colour;

   unit_to_view(view, x, y, &view_x, &view_y);
   masked_blit(terrain->tiles[square->terrain.type]->bitmap, view->draw, 0, 0, (int)view_x - terrain->tiles[square->terrain.type]->focus_x, (int)view_y - terrain->tiles[square->terrain.type]->focus_y, terrain->tiles[square->terrain.type]->w, terrain->tiles[square->terrain.type]->h);

   grid_colour = 0x000000;
//   do_line(view->draw, (int)view_x, (int)view_y, (int)view_x, (int)view_y + view->unit_h_pixels - 1, grid_colour, put_half_pixel);
//   do_line(view->draw, (int)view_x + view->unit_w_pixels - 1, (int)view_y, (int)view_x + view->unit_w_pixels - 1, (int)view_y + view->unit_h_pixels - 1, grid_colour, put_half_pixel);
//   do_line(view->draw, (int)view_x, (int)view_y, (int)view_x + view->unit_w_pixels - 1, (int)view_y, grid_colour, put_half_pixel);
//   do_line(view->draw, (int)view_x, (int)view_y + view->unit_h_pixels - 1, (int)view_x + view->unit_w_pixels - 1, (int)view_y + view->unit_h_pixels - 1, grid_colour, put_half_pixel);
}

void mini_square_draw(const square_t *square, const view_t *view, float x, float y)
{
   float view_x, view_y;
   int grid_colour;
   RGB *rgb;

   unit_to_view(view, x, y, &view_x, &view_y);

   rgb = &terrain_types[square->terrain.type].colour;

   grid_colour = makecol_depth(24, rgb->r, rgb->g, rgb->b);
   rectfill(view->draw, (int)view_x, (int)view_y, (int)view_x + view->unit_w_pixels, (int)view_y + view->unit_h_pixels, grid_colour);
}

void map_draw(map_t *map, const view_t *view, square_drawer_t *square_drawer, army_drawer_t *army_drawer)
{
   float square_start_x, square_start_y;
   float square_end_x, square_end_y;
   int x, y;
   list_t *current;
   army_t *army;

   int i;
   int start_offset_x, start_offset_y;
   int end_offset_x, end_offset_y;

   start_offset_x = 0;
   start_offset_y = 0;
   end_offset_x = 0;
   end_offset_y = 0;
   for (i = 0; i < terrain->size; i++) {
      if (start_offset_x > terrain->tiles[i]->focus_x + view->unit_w_pixels - terrain->tiles[i]->w) {
         start_offset_x = terrain->tiles[i]->focus_x + view->unit_w_pixels - terrain->tiles[i]->w;
      }
      if (start_offset_y > terrain->tiles[i]->focus_y + view->unit_h_pixels - terrain->tiles[i]->h) {
         start_offset_y = terrain->tiles[i]->focus_y + view->unit_h_pixels - terrain->tiles[i]->h;
      }
      if (end_offset_x < terrain->tiles[i]->focus_x) {
         end_offset_x = terrain->tiles[i]->focus_x;
      }
      if (end_offset_y < terrain->tiles[i]->focus_y) {
         end_offset_y = terrain->tiles[i]->focus_y;
      }
   }

   view_to_unit(view, 0 + start_offset_x, 0 + start_offset_y, &square_start_x, &square_start_y);
   view_to_unit(view, view->w - 1 + end_offset_x, view->h - 1 + end_offset_y, &square_end_x, &square_end_y);

   if (square_start_x < 0) {
      square_start_x = 0;
   }
   if (square_start_y < 0) {
      square_start_y = 0;
   }
   if (square_end_x >= map->w) {
      square_end_x = map->w - 1;
   }
   if (square_end_y >= map->h) {
      square_end_y = map->h - 1;
   }
   current_map = map;

   if (square_drawer) {
      for (x = (int)square_start_x; x <= (int)square_end_x; x++) {
         for (y = (int)square_start_y; y <= (int)square_end_y; y++) {
            square_drawer(&(map->squares[x][y]), view, x, y);
        }
      }
   }

   if (army_drawer) {
      current = map->armies;
      while (current) {
         army = (army_t *)current->data;
         if (army->x >= (int)square_start_x && army->x <= (int)square_end_x &&
             army->y >= (int)square_start_y && army->y <= (int)square_end_y) {
            army_drawer(army, view, army->x, army->y);
         }
         current = current->next;
      }
   }

//   textprintf(view->draw, font,  0, 0, 0xffffff, "%d, %d, %d, %d", (int)square_start_x, (int)square_start_y, (int)square_end_x, (int)square_end_y);
}

void square_init(square_t *square)
{
   terrain_init(&(square->terrain));
}

void map_init(map_t *map)
{
   map->w = map->h = 0;
   map->squares = NULL;
   map->structures = NULL;
   map->units = NULL;
   map->armies = NULL;
}

void map_destroy_squares(map_t *map)
{
   int i;

   if (map->squares != NULL) {
      for (i = 0; i < map->w; i++) {
         free(map->squares[i]);
      }
      free(map->squares);
      map->squares = NULL;
   }
}

int map_create_squares(map_t *map)
{
   int i, j;

   if ((map->squares = malloc(sizeof(square_t *) * map->w)) != NULL) {
      for (i = 0; i < map->w; i++) {
         if ((map->squares[i] = malloc(sizeof(square_t) * map->h)) != NULL) {
            for (j = 0; j < map->h; j++) {
               square_init(&(map->squares[i][j]));
            }
         }
         else {
            free(map->squares);
            map->squares = NULL;
            return 1;
         }
      }
   }
   else {
      return 1;
   }

   return 0;
}

map_t *map_load(const char *filename)
{
   FILE *file = NULL;
   map_t *map = NULL;

   if ((file = fopen(filename, "rb")) == NULL) {
      return NULL;
   }

   if ((map = map_create()) == NULL) {
      fclose(file);
      return NULL;
   }

   if (map_read(map, file)) {
      fclose(file);
      free(map);
      return NULL;
   }

   fclose(file);

   return map;
}

void map_destroy(map_t *map)
{
   if (map != NULL) {
      map_clean(map);
      free(map);
      map = NULL;
   }
}

void map_clean(map_t *map)
{
   map_destroy_squares(map);
   unit_list_clean(&map->units);
   army_list_clean(&map->armies);
}

map_t *map_create(void)
{
   map_t *map = NULL;

   if ((map = malloc(sizeof(map_t))) == NULL) {
      return NULL;
   }
   map_init(map);

   return map;
}

int map_save(map_t *map, const char *filename)
{
   FILE *file = NULL;

   if ((file = fopen(filename, "wb")) == NULL) {
      return 1;
   }

   if (map_write(map, file)) {
      fclose(file);
      return 1;
   }

   fclose(file);

   return 0;
}

int square_write(square_t *square, FILE *file)
{
   if (fwrite((void *)&square->terrain, sizeof(terrain_t), 1, file) < 1) {
      return 1;
   }

   return 0;
}

int square_read(square_t *square, FILE *file)
{
   if (fread((void *)&square->terrain, sizeof(terrain_t), 1, file) < 1) {
      return 1;
   }

//   if (army_read(&(square->army), file)) {
//      return 1;
//   }

   return 0;
}

int map_write(map_t *map, FILE *file)
{
   int i, j;

   current_map = map;

   /* Write squares */
   if (fwrite((void *)map->name, sizeof(char), MAP_NAME_SIZE, file) < 1) {
      return 1;
   }
   if (fwrite((void *)map->desc, sizeof(char), MAP_DESC_SIZE, file) < 1) {
      return 1;
   }
   if (fwrite((void *)&map->w, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fwrite((void *)&map->h, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fwrite((void *)&map->turn, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fwrite((void *)&map->side_count, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fwrite((void *)&map->side_turn, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fwrite((void *)map->side_order, sizeof(int), MAP_SIDE_COUNT, file) < 1) {
      return 1;
   }
   if (map->squares == NULL) {
      return 1;
   }
   for (i = 0; i < map->w; i++) {
      for (j = 0; j < map->h; j++) {
         if (square_write(&(map->squares[i][j]), file)) {
            return 1;
         }
      }
   }

   /* Write units */
   if (unit_list_write(&map->units, file)) {
      return 1;
   }

   /* Write armies */
   if (army_list_write(&map->armies, file)) {
      return 1;
   }

   return 0;
}

int map_read(map_t *map, FILE *file)
{
   int i, j;

   current_map = map;

   /* Read squares */
   if (fread((void *)map->name, sizeof(char), MAP_NAME_SIZE, file) < 1) {
      return 1;
   }
   if (fread((void *)map->desc, sizeof(char), MAP_DESC_SIZE, file) < 1) {
      return 1;
   }
   if (fread((void *)&map->w, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fread((void *)&map->h, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fread((void *)&map->turn, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fread((void *)&map->side_count, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fread((void *)&map->side_turn, sizeof(int), 1, file) < 1) {
      return 1;
   }
   if (fread((void *)map->side_order, sizeof(int), MAP_SIDE_COUNT, file) < 1) {
      return 1;
   }
   if (map_create_squares(map)) {
      return 1;
   }
   for (i = 0; i < map->w; i++) {
      for (j = 0; j < map->h; j++) {
         if (square_read(&(map->squares[i][j]), file)) {
            return 1;
         }
      }
   }

   /* Read units */
   map->units = NULL;
   if (unit_list_read(&map->units, file)) {
      return 1;
   }

   /* Read armies */
   map->armies = NULL;
   if (army_list_read(&map->armies, file)) {
      return 1;
   }

   return 0;
}

#define TER_SEED_COUNT 64

#define SEED_COUNT 64

int map_generate(map_t *map)
{
   int i, j;
   double persistence;
   int octaves;
   float height_min, height_max, height_range;
   hmap_t hmap;
   seed_t seeds[SEED_COUNT];
   int x, y;
   float probability;
   float forest_min, forest_max;
   int range = 256;
   int roughness = 1;

   int end;
   int deepest;
   int this_x, this_y;
   int new_x, new_y;

   hmap_init(&hmap);
   hmap.w = map->w;
   hmap.h = map->h;
   if (hmap_create_squares(&hmap)) {
      return 1;
   }

//   hmap.squares[0][0] = ran(range * roughness) - (range * roughness / 2);
//   hmap.squares[hmap.w - 1][0] = ran(range * roughness) - (range * roughness / 2);
//   hmap.squares[hmap.w - 1][hmap.h - 1] = ran(range * roughness) - (range * roughness / 2);
//   hmap.squares[0][hmap.h - 1] = ran(range * roughness) - (range * roughness / 2);
//   for (i = 0; i < 32; i++) {
//      hmap_subdivide(&hmap, 0, 0, hmap.w - 1, hmap.h - 1, range, roughness);
//   }
//   hmap_collage_rect(&hmap, 10000, 128);
//   hmap_faulting(&hmap, 2000, 256);
//   hmap_fault2(&hmap, 2000, 256);
//   hmap_deposition(&hmap, 256, 16384);
//   hmap_circles(&hmap, 10000, 8);
   persistence = 1.0 / pow(2.0, 1.0 / 2.0);
   octaves = 8;
//   hmap_perlin(&hmap, persistence, octaves);
//   hmap_deposition2(&hmap, peak_collager, 32, 32, 16, 16, CONTAIN_WRAP);
//   hmap_deposition2(&hmap, diamond_collager, 256, 32, 16, 16, CONTAIN_STRICT);
   hmap_collage_reduce(&hmap, fault_collager, 1024, 1, CONTAIN_STRICT);
   hmap_soften(&hmap, NULL, CONTAIN_STRICT);
//   hmap_subdivision(&hmap, range, roughness);
//   hmap_subdivision(&hmap, 256, 2);
//   hmap_frequency_composition(&hmap, 256);

/*   int i, j;
   int x, y;
   seed_t seeds[TER_SEED_COUNT];

   for (i = 0; i < TER_SEED_COUNT; i++) {
      seeds[i].x = rand() % hmap.w;
      seeds[i].y = rand() % hmap.h;
      seeds[i].power = rand() % 256 - 128;
   }

   for (x = 0; x < hmap.w; x++) {
      for (y = 0; y < hmap.h; y++) {
//         hmap->squares[x][y] = 0;
         for (j = 0; j < TER_SEED_COUNT; j++) {
//            hmap.squares[x][y] += (1 / sqrt(pow(x - seeds[j].x, 2) + pow(y - seeds[j].y, 2))) * seeds[j].power;
            hmap.squares[x][y] += MAX(seeds[j].power - sqrt(pow(x - seeds[j].x, 2) + pow(y - seeds[j].y, 2)), 0);
         }
      }
   }*/

/*   for (i = 0; i < hmap.w; i++) {
      for (j = 0; j < hmap.h; j++) {
         putpixel(screen, i, j, hmap.squares[i][j]);
      }
   }
   while (!keypressed() && !mouse_b);*/

   height_max = hmap.squares[0][0];
   height_min = hmap.squares[0][0];
   for (i = 0; i < hmap.w; i++) {
      for (j = 0; j < hmap.h; j++) {
         if (hmap.squares[i][j] > height_max) {
            height_max = hmap.squares[i][j];
         }
         if (hmap.squares[i][j] < height_min) {
            height_min = hmap.squares[i][j];
         }
      }
   }

   height_range = fabs(height_max - height_min);

   for (i = 0; i < map->w; i++) {
      for (j = 0; j < map->h; j++) {
         if (hmap.squares[i][j] < height_min + 20 * (height_range / 32)) {
            map->squares[i][j].terrain.type = 3;
         }
         else if (hmap.squares[i][j] < height_min + 21 * (height_range / 32)) {
            map->squares[i][j].terrain.type = 4;
         }
         else if (hmap.squares[i][j] < height_min + 27 * (height_range / 32)) {
            map->squares[i][j].terrain.type = 0;
         }
         else if (hmap.squares[i][j] < height_min + 30 * (height_range / 32)) {
            map->squares[i][j].terrain.type = 6;
         }
         else {
            map->squares[i][j].terrain.type = 7;
         }
      }
   }

   forest_min = height_min + 11 * (height_range / 16);
   forest_max = height_min + 14 * (height_range / 16);

   for (i = 0; i < SEED_COUNT; i++) {
      seeds[i].x = rand() % map->w;
      seeds[i].y = rand() % map->h;
//      seeds[i].power = rand() % 256 - 128;
      seeds[i].power = rand() % 256;
   }

/*   for (i = 0; i < (map->info.w * map->info.h) / 4; i++) {
      x = rand() % map->info.w;
      y = rand() % map->info.h;
      probability = 0;
      for (j = 0; j < 32; j++) {
         probability += (1 / sqrt(pow(x - seeds[j].x, 2) + pow(y - seeds[j].y, 2))) * seeds[j].power;
      }
      if (probability > 64) {
         if (map->squares[x][y].info.terrain.type == 0) {
            map->squares[x][y].info.terrain.type = 5;
         }
      }
   }*/

   for (x = 0; x < map->w; x++) {
      for (y = 0; y < map->h; y++) {
         probability = 0;
         for (j = 0; j < SEED_COUNT; j++) {
            probability += (1 / sqrt(pow(x - seeds[j].x, 2) + pow(y - seeds[j].y, 2))) * seeds[j].power;
         }
//         if (probability > 256) {
//               map->squares[x][y].info.terrain.type = 2;
//         }
         probability *= (0.5 - fabs((hmap.squares[x][y] - forest_min) / (forest_max - forest_min) - 0.5)) * 2;
//         if (rand() % (int)probability > 128) {
//         if (probability > 128) {
//         if (probability > 256) {
         if (probability > 192) {
//            if (map->squares[x][y].info.terrain.type == 0) {
               map->squares[x][y].terrain.type = 5;
//            }
         }
      }
   }

   for (x = 0; x < 32; x++) {
      end = FALSE;
      this_x = rand() % hmap.w;
      this_y = rand() % hmap.h;
      while (!end) {
         new_x = -1;
         new_y = -1;
         deepest = hmap.squares[this_x][this_y];
         for (i = -1; i < 2; i++) {
            for (j = -1; j < 2; j++) {
               if (this_x + i < 0 || this_x + i >= hmap.w ||
                   this_y + j < 0 || this_y + j >= hmap.h) {
               }
               else {
                  if (hmap.squares[this_x + i][this_y + j] < deepest) {
                     new_x = this_x + i;
                     new_y = this_y + j;
                     deepest = hmap.squares[new_x][new_y];
                  }
               }
            }
         }
         if (map->squares[this_x][this_y].terrain.type == 3) end = TRUE;
         map->squares[this_x][this_y].terrain.type = 3;
         if (new_x == -1) {
            end = TRUE;
         }
         this_x = new_x;
         this_y = new_y;
      }
   }

   hmap_clean(&hmap);

   return 0;
}

