#include "postproc.h"

#include <stddef.h>
#include <math.h>

#include "math.h"

#define MAX(a, b) ((a) > (b) ? (a) : (b))
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#define MID(a, b, c) MIN(MAX(a, b), MAX(b, c))

#define DIST(x1, y1, x2, y2) (sqrt(pow((x1) - (x2), 2) + pow((y1) - (y2), 2)))

void hmap_gaussian_blur(hmap_t *in, hmap_t *out, double dist, int contain)
{
   int i, j, k, l;
   int x, y;
   double total;
   double d;
   double div;
   hmap_t temp;

   dist = fabs(dist);

   if (out == NULL) {
      out = &temp;
      hmap_init(out);
      out->w = in->w;
      out->h = in->h;
      hmap_create_squares(out);
   }
   else {
      if (out->w != in->w || out->h != in->h || out->squares == NULL) {
         hmap_clean(out);
         hmap_init(out);
         out->w = in->w;
         out->h = in->h;
         hmap_create_squares(out);
      }
   }

   for (i = 0; i < in->w; i++) {
      for (j = 0; j < in->h; j++) {
         total = 0.0;
         div = 0.0;
         for (k = -dist; k <= dist; k++) {
            for (l = -dist; l <= dist; l++) {
               x = i + k;
               y = j + l;
               d = DIST(k, l, 0, 0);
               if (d <= dist) {
                  if (contain_rect(&x, &y, in->w, in->h, contain)) {
                     total += in->squares[x][y] * (1.0 - d / dist);
                     div += (1.0 - d / dist);
                  }
               }
            }
         }
         out->squares[i][j] = total / div;
      }
   }

   if (out == &temp) {
      hmap_copy(out, in);
      hmap_clean(out);
   }
}

void hmap_soften(hmap_t *in, hmap_t *out, int contain)
{
   int i, j, k, l;
   int x, y;
   double total;
   double d;
   int pixels;
   hmap_t temp;

   if (out == NULL) {
      out = &temp;
      hmap_init(out);
      out->w = in->w;
      out->h = in->h;
      hmap_create_squares(out);
   }
   else {
      if (out->w != in->w || out->h != in->h || out->squares == NULL) {
         hmap_clean(out);
         hmap_init(out);
         out->w = in->w;
         out->h = in->h;
         hmap_create_squares(out);
      }
   }

   for (i = 0; i < in->w; i++) {
      for (j = 0; j < in->h; j++) {
         total = 0.0;
         pixels = 0;
         for (k = -1; k < 2; k++) {
            for (l = -1; l < 2; l++) {
               x = i + k;
               y = j + l;
               if (contain_rect(&x, &y, in->w, in->h, contain)) {
                  total += in->squares[x][y];
                  pixels++;
               }
            }
         }
         out->squares[i][j] = total / pixels;
      }
   }

   if (out == &temp) {
      hmap_copy(out, in);
      hmap_clean(out);
   }
}

void hmap_get_range(hmap_t *hmap, double *min, double *max)
{
   int i, j;

   *max = hmap->squares[0][0];
   *min = hmap->squares[0][0];
   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         if (*max < hmap->squares[i][j]) *max = hmap->squares[i][j];
         if (*min > hmap->squares[i][j]) *min = hmap->squares[i][j];
      }
   }
}

void hmap_set_range(hmap_t *hmap, double min, double max)
{
   int i, j;
   double old_min, old_max;

   hmap_get_range(hmap, &old_min, &old_max);

   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         hmap->squares[i][j] = (hmap->squares[i][j] - old_min) / (old_max - old_min) * (max - min) + min;
      }
   }
}

void hmap_invert(hmap_t *hmap)
{
   int i, j;

   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         hmap->squares[i][j] = -hmap->squares[i][j];
      }
   }
}

void hmap_high_pass(hmap_t *hmap, double val)
{
   int i, j;

   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         if (hmap->squares[i][j] > val) hmap->squares[i][j] = val;
      }
   }
}

void hmap_low_pass(hmap_t *hmap, double val)
{
   int i, j;

   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         if (hmap->squares[i][j] < val) hmap->squares[i][j] = val;
      }
   }
}

void hmap_scale(hmap_t *hmap, double val)
{
   int i, j;

   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         hmap->squares[i][j] *= val;
      }
   }
}

void hmap_shift(hmap_t *hmap, double val)
{
   int i, j;

   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         hmap->squares[i][j] += val;
      }
   }
}

void hmap_normalize(hmap_t *hmap)
{
   double min, max;
   double scale;

   hmap_get_range(hmap, &min, &max);

   scale = 1.0 / MAX(fabs(min), fabs(max));

   hmap_scale(hmap, scale);
}

void hmap_clear(hmap_t *hmap)
{
   int i, j;

   for (i = 0; i < hmap->w; i++) {
      for (j = 0; j < hmap->h; j++) {
         hmap->squares[i][j] = 0.0;
      }
   }
}

void hmap_copy(hmap_t *in, hmap_t *out)
{
   int i, j;

   if (out->w != in->w || out->h != in->h || out->squares == NULL) {
      hmap_clean(out);
      hmap_init(out);
      out->w = in->w;
      out->h = in->h;
      hmap_create_squares(out);
   }

   for (i = 0; i < in->w; i++) {
      for (j = 0; j < in->h; j++) {
         out->squares[i][j] = in->squares[i][j];
      }
   }
}

void hmap_multiply(hmap_t *a, hmap_t *b, hmap_t *out)
{
   int i, j;

   for (i = 0; i < a->w; i++) {
      for (j = 0; j < a->h; j++) {
         out->squares[i][j] = a->squares[i][j] * b->squares[i][j];
      }
   }
}

void hmap_divide(hmap_t *a, hmap_t *b, hmap_t *out)
{
   int i, j;

   for (i = 0; i < a->w; i++) {
      for (j = 0; j < a->h; j++) {
         out->squares[i][j] = a->squares[i][j] / b->squares[i][j];
      }
   }
}

void hmap_add(hmap_t *a, hmap_t *b, hmap_t *out)
{
   int i, j;

   for (i = 0; i < a->w; i++) {
      for (j = 0; j < a->h; j++) {
         out->squares[i][j] = a->squares[i][j] + b->squares[i][j];
      }
   }
}

void hmap_subtract(hmap_t *a, hmap_t *b, hmap_t *out)
{
   int i, j;

   for (i = 0; i < a->w; i++) {
      for (j = 0; j < a->h; j++) {
         out->squares[i][j] = a->squares[i][j] - b->squares[i][j];
      }
   }
}

double hmap_inverse(double a)
{
   return -a;
}

double hmap_max(double a, double b)
{
   return MAX(a, b);
}


double hmap_min(double a, double b)
{
   return MIN(a, b);
}

double hmap_mid(double a, double b, double c)
{
   return MID(a, b, c);
}


void hmap_unary(hmap_t *a, hmap_t *out, hmap_unary_t *method)
{
   int i, j;

   for (i = 0; i < a->w; i++) {
      for (j = 0; j < a->h; j++) {
         out->squares[i][j] = method(a->squares[i][j]);
      }
   }
}

void hmap_binary(hmap_t *a, hmap_t *b, hmap_t *out, hmap_binary_t *method)
{
   int i, j;

   for (i = 0; i < a->w; i++) {
      for (j = 0; j < a->h; j++) {
         out->squares[i][j] = method(a->squares[i][j], b->squares[i][j]);
      }
   }
}

void hmap_ternary(hmap_t *a, hmap_t *b, hmap_t *c, hmap_t *out, hmap_ternary_t *method)
{
   int i, j;

   for (i = 0; i < a->w; i++) {
      for (j = 0; j < a->h; j++) {
         out->squares[i][j] = method(a->squares[i][j], b->squares[i][j], c->squares[i][j]);
      }
   }
}

