#pragma warning( disable: 4312 )

#include <allegro.h>
#include <string>
#include <vector>
#include <map>

#include <common.h>

class CDLXSolver {

public:

    CDLXSolver();
    virtual ~CDLXSolver();
    void AddColumn(const std::string& name, bool mandatory=true);
    void NewRow();
    void SetColumn(const std::string& name);
    void SetColumn(unsigned int number);
    void DisableRow(unsigned int number);
    void EnableRow(unsigned int number);
    void DeleteRows();
    void Solve();

protected:

    virtual bool Record();
    std::string* GetSolution();
    unsigned int dequeRemovals;
    unsigned int numSolutions;

private:

    // private classes
    class CNode;
    class CHeader;

    // main 2D doubly-linked structure
    CHeader *root;

    // exhaustive list of columns
    std::vector<CHeader*> columns;
    std::map<std::string, int> names;

    // list of non-empty rows
    std::vector<CNode*> rows;

    // solution stack, and pointers to browse it
    std::vector<CNode*> stack;
    std::vector<CNode*>::iterator iterStack;
    CNode *iterRow;

    // Search() temporary variables (not local to reduce recursion load)
    CHeader *column;
    CNode   *row;
    bool     more;

    // search methods
    void Search();
    void Cover(CHeader* col);
    void Uncover(CHeader* col);
    void Choose();
};


/* Solver class implementation ============================================== */

#include <iostream>
#include <sstream>

using namespace std;

class CDLXSolver::CNode {

public:

    CNode *left, *right, *up, *down;
    CHeader *head;
};

class CDLXSolver::CHeader : public CDLXSolver::CNode {

public:

    int size;
    string name;
};

void CDLXSolver::AddColumn(const string& name, bool mandatory) {

    CHeader* col = new CHeader;
    col->name = name;
    col->size = 0;
    col->up = col->down = col->left = col->right = col->head = col;
    if (mandatory) {
        col->right = root;
        col->left = root->left;
        root->left->right = col;
        root->left = col;
    }
    names[name] = (int)columns.size();
    columns.push_back(col);
}

void CDLXSolver::NewRow() {

    row = NULL;
}

void CDLXSolver::DeleteRows() {

    for (vector<CHeader*>::iterator it = columns.begin();
                                    it != columns.end(); it++) {
        (*it)->down = (*it)->up = *it;
    }
    for (vector<CNode*>::iterator it = rows.begin();
                                  it != rows.end(); it++) {
        for (CNode *j = (*it)->right; j != *it; ) {
            CNode* j_right = j->right; delete j; j = j_right;
        }
        delete *it;
    }
    rows.clear();
    row = NULL;
}

void CDLXSolver::DisableRow(unsigned int number) {

    if (number >= rows.size()) return;
    CNode* i = rows[number];
    if (i->up == i) return; // already disabled
    CNode* j = i;
    do {
        j->up->down = j->down;
        j->down->up = j->up;
        j->down = j->up = j;
        j->head->size--;
        j = j->right;
    } while (i != j);
}

void CDLXSolver::EnableRow(unsigned int number) {

    if (number >= rows.size()) return;
    CNode* i = rows[number];
    if (i->up != i) return; // already enabled
    CNode* j = i;
    do {
        j->up = j->head;
        j->down = j->head->down;
        j->up->down = j;
        j->down->up = j;
        j->head->size++;
        j = j->right;
    } while (i != j);
}

void CDLXSolver::SetColumn(const string& name) {

    if (names.find(name) == names.end()) return;
    SetColumn(names[name]);
}

void CDLXSolver::SetColumn(unsigned int number) {

    if (number >= columns.size()) return;
    CHeader* header = columns[number];
    CNode* node = new CNode;
    if (row == NULL) {
        row = node;
        rows.push_back(node);
        node->left  = node;
        node->right = node;
    } else {
        node->left = row;
        node->right = row->right;
        row->right->left = node;
        row->right = node;
    }
    node->head = header;
    node->up = header;
    node->down = header->down;
    header->down->up = node;
    header->down = node;
    header->size++;
}

string* CDLXSolver::GetSolution() {

    if (iterRow != NULL) {
        string* ret = &iterRow->head->name;
        iterRow = iterRow->right;
        if (iterRow == *iterStack) iterRow = NULL;
        return ret;
    }
    if (iterStack != stack.end() && ++iterStack != stack.end()) {
        iterRow = *iterStack;
    }
    return NULL;
}

bool CDLXSolver::Record() {

    cout << "Solution (found after " << dequeRemovals
         << " deque removals):" << endl;
    for (string* s = GetSolution(); s != NULL; s = GetSolution()) {
        for (; s != NULL; s = GetSolution()) {
            cout << *s << " ";
        }
        cout << endl;
    }
    cout << endl;
    return true;
}

void CDLXSolver::Solve() {

    more = true;
    stack.clear();
    dequeRemovals = 0;
    numSolutions = 0;
    Search();
}

void CDLXSolver::Search() {

    if (root->right == root) {
        numSolutions++;
        iterStack = stack.begin();
        iterRow = (iterStack != stack.end() ? *iterStack : NULL);
        more = Record();
        return;
    }
    Choose();
    Cover(column);
    stack.push_back(NULL);
    for (row = column->down; row != column; row = row->down) {
        for (CNode *i = row->right; i != row; i = i->right) {
            Cover(i->head);
        }
        stack.back() = row;
        Search();
        row = stack.back(); column = row->head;
        for (CNode *i = row->left; i != row; i = i->left) {
            Uncover(i->head);
        }
        if (!more) break;
    }
    stack.pop_back();
    Uncover(column);
}

void CDLXSolver::Cover(CHeader* col) {

    col->right->left = col->left;
    col->left->right = col->right;
    dequeRemovals++;
    for (CNode *i = col->down; i != col; i = i->down) {
        for (CNode *j = i->right; j != i; j = j->right) {
            j->down->up = j->up;
            j->up->down = j->down;
            j->head->size--;
            dequeRemovals++;
        }
    }
}

void CDLXSolver::Uncover(CHeader* col) {

    for (CNode *i = col->up; i != col; i = i->up) {
        for (CNode *j = i->left; j != i; j = j->left) {
            j->head->size++;
            j->down->up = j;
            j->up->down = j;
        }
    }
    col->right->left = col;
    col->left->right = col;
}

void CDLXSolver::Choose() {

    int best = INT_MAX;
    for (CHeader* i = (CHeader*)root->right; i != root;
                  i = (CHeader*)i->right) {
        if (i->size < best) {
            column = i;
            best = i->size;
        }
    }
}

CDLXSolver::CDLXSolver() {

    root = new CHeader;
    root->left = root->right = root;
    row = NULL;
}

CDLXSolver::~CDLXSolver() {

    for (vector<CHeader*>::iterator it = columns.begin();
                                         it != columns.end(); it++) {
        for (CNode *j = (*it)->down; j != *it; ) {
            CNode* j_down =  j->down; delete j; j = j_down;
        }
        delete *it;
    }
    delete root;
}


/* Specialized class for Sudoku ============================================= */

class CSudokuSolver : private CDLXSolver {

public:

    CSudokuSolver();
    void SetCell(int R, int C, int d);
    void Solve();
    void Print(bool solved);
    int  solution[9][9];
    int grid[9][9];
    int  count;

private:

    virtual bool Record();
};

CSudokuSolver::CSudokuSolver() {

    stringstream ss;
    memset(grid, 0, sizeof(grid));
    memset(solution, 0, sizeof(solution));
    count = 2;

    // define columns for individual cells
    // ("a cell contains 1 and only 1 digit")
    for (int R = 0; R < 9; R++) {
        for (int C = 0; C < 9; C++) {
            ss.str("");
            ss << "R" << (R+1) << "C" << (C+1);
            AddColumn(ss.str());
            grid[R][C] = 0; // init grid
        }
    }

    // define columns for each group containing each digit
    // ("each digit appears once and only once in each group")
    for (int d = 1; d <= 9; d++) {
        for (int i = 0; i < 3; i++) {
            for (int j = 1; j <= 9; j++) {
                ss.str("");
                ss << d << "in" << "RCB"[i] << j;
                AddColumn(ss.str());
            }
        }
    }

    // create rows ("there could be any digit in any empty cell")
    for (int R = 0; R < 9; R++) {
        for (int C = 0; C < 9; C++) {
            for (int d = 1; d <= 9; d++) {
                NewRow();
                SetColumn(R*9+C);
                SetColumn(54 + d*27 + R);
                SetColumn(63 + d*27 + C);
                SetColumn(72 + d*27 + (R/3)*3+(C/3));
                /*
                // same thing, a bit slower:
                stringstream ss;
                ss.str(""); ss<<"R"<<(R+1)<<"C"<<(C+1);      SetColumn(ss.str());
                ss.str(""); ss<<d<<"inR"<<(R+1);             SetColumn(ss.str());
                ss.str(""); ss<<d<<"inC"<<(C+1);             SetColumn(ss.str());
                ss.str(""); ss<<d<<"inB"<<((R/3)*3+(C/3)+1); SetColumn(ss.str());
                */
            }
        }
    }
}

void CSudokuSolver::SetCell(int R, int C, int d) {

    grid[R][C] = d;
    for (int i = 1; i<= 9; i++) {
        if (d<=0 || i==d) {
            EnableRow(R*81+C*9+i-1);
        } else {
            DisableRow(R*81+C*9+i-1);
        }
    }
}

void CSudokuSolver::Solve() {

    count = 0;
    CDLXSolver::Solve();
}

bool CSudokuSolver::Record() {

    if (++count > 1) return false;
    for (string* s = GetSolution(); s != NULL; s = GetSolution()) {
        int R=-1, C=-1;
        for (; s != NULL; s = GetSolution()) {
            if (s->length()<5) continue;
            if ((*s)[3] == 'R') R = (*s)[4] - '1';
            if ((*s)[3] == 'C') C = (*s)[4] - '1';
            if (R >= 0 && C >= 0) solution[R][C] = (*s)[0] - '0';
        }
    }
    return true;
}

void CSudokuSolver::Print(bool solved)
{
    for (int R=0; R<9; R++)
    {
        for (int C=0; C<9; C++)
        {
            int d = (solved ? solution[R][C] : grid[R][C]);

            cout << ( d <= 0 ? '0' : char( d + '0' ) );

        }
    }
    cout << endl;
}
/*



    for (int R=0; R<9; R++)
    {
        for (int C=0; C<9; C++)
        {
            int d = (solved ? solution[R][C] : grid[R][C]);

            cout << " " << ( d <= 0 ? '.' : char( d + '0' ) );

            if (C == 2 || C == 5) cout << " |";
        }

        cout << " " << endl;

        if (R == 2 || R == 5) cout << " ------+-------+------ " << endl;
    }
    cout << endl;
}
*/

/* Entry point ============================================================== */

#include <ctime>

static struct {
    int grid[81];
    int count;
    int best[81];
    int score;
    CSudokuSolver solver;
} global;

void SetCell(int n, int d) {

    if (global.grid[n] > 0) global.count--; // update fixed cells count
    if (d > 0) global.count++;
    global.grid[n] = d;                     // update cell
    global.solver.SetCell(n/9, n%9, d);
}

void Simplify(int start=0) {

    // save grid if it the best we have for now
    if (global.count < global.score) {
        memcpy(global.best, global.grid, sizeof(global.grid));
        global.score = global.count;
    }

    // move to the next filled cell
    while (start <= 40 && global.grid[start] <= 0) start++;
    if (start>40) return; // (the grid's lower half has been done by symmetry)
    int d1 = global.grid[   start];
    int d2 = global.grid[80-start];

    // this cell being kept, try and remove the next ones
    Simplify(start+1);

    // when clearing this cell, is solution still unique? yes -> clear more cells
    SetCell(   start, 0);
    SetCell(80-start, 0);
    global.solver.Solve();
    if (global.solver.count == 1) Simplify(start+1);
    SetCell(   start, d1);
    SetCell(80-start, d2);
}

GAME_FUNC( int, generate, ( int difficulty, char grid[ 9 ][ 9 ], char solution[ 9 ][ 9 ] ) )
{
    srand( (unsigned int)time( NULL ) );

    memset(global.grid, 0, sizeof(global.grid));
    global.count = 0;
    int sizeMax = 25;

    difficulty = 0;
    
    while ( true )
    {
        // reset grid with fixed central cell
        for ( int n = 0; n < 81; n++ ) 
        {
            if ( global.grid[ n ] !=0 )
            {
                SetCell( n, 0 );
            }
        }

        SetCell( 40, 1 + ( rand() % 9 ) );

        // add cells until the solution is unique
        do 
        {
            if ( global.count >= sizeMax + 4 )
            {
                // abort big grids
                break;   
            }

            int k, n;

            do 
            {
                // choose cell and digits
                k = rand() % ( 40 * 9 * 9 );              
                n = k % 40;
            } 
            while ( global.grid[ n ] > 0 );
            
            k /= 40;

            // set new cells
            SetCell(      n, 1 + ( k % 9 ) );                 
            SetCell( 80 - n, 1 + ( k / 9 ) ) ;

            // solve
            global.solver.Solve();                  

            if ( global.solver.count == 0 ) 
            {
                // cancel new cells
                SetCell(      n, 0 );                   
                SetCell( 80 - n, 0 );
            }
        } 
        while ( global.solver.count != 1 );
        
        // abort big grids
        if ( global.solver.count != 1 ) 
        {
            continue;     
        }

        // remove as many cells as possible while keeping solution unique
        global.score = INT_MAX;
        Simplify();

        // if the grid satisfies maximum size, display it
        if ( global.score <= sizeMax ) 
        {
            for ( int n = 0; n < 81; n++ ) 
            {
                SetCell( n, global.best[ n ] );
            }

            global.solver.Solve();

            for ( int row = 0; row < 9; row++ )
            {
                for ( int col = 0; col < 9; col++ )
                {
                    grid[ row ][ col ] = global.solver.grid[ row ][ col ];
                    solution[ row ][ col ] = global.solver.solution[ row ][ col ];
                }
            }

            return global.score;
        }
    }

    return 0;
}


