#ifdef _MSC_VER
#pragma warning( disable: 4312 )
#else
#define ALLEGRO_STATICLINK
#endif

#include <allegro.h>
#include <allegro/internal/aintern.h>
#include <time.h>
#include <stdio.h>
#include <string>
#include "defines.h"
#include "boardcell.h"
#include "board.h"
#include "module.h"
#include "generator.h"
#include "themefunctions.h"
#include "theme.h"
#include "globals.h"
#include "themeList.h"
#include "misc.h"
#include "game.h"
#include "sudoku.h"


bool            Game::done                  = false;
bool            keyEscPushed                = false;
bool            keyF1Pushed                 = false;
bool            keyNumberDown[ 10 ]         = { false, false, false, false, false, false, false, false, false, false };
bool            keyNumberPushed[ 10 ]       = { false, false, false, false, false, false, false, false, false, false };
bool            keyShiftDown                = false;
bool            keyCtrlLPushed              = false;
bool            keyCtrlSPushed              = false;
bool            keyCtrlQPushed              = false;
bool            keyCtrlNPushed              = false;
bool            mouse1Down                  = false;
bool            mouse2Down                  = false;
bool            mouse1Clicked               = false;
bool            mouse2Clicked               = false;
int             lastButton                  = -1;
int             startNewGame                = -1;
QuestionInfo    question;
ThemeList       themeList;
int             gamestate                   = GS_GAME;
bool            congratulations             = false;

Game game;

volatile int secondCounter = 0;

void Game::secondCounterHandler()
{
    if ( !( game.board.filled() || gamePaused || blankBoard ) )
    {
        secondCounter++;
    }
}

END_OF_FUNCTION( Game::secondCounterHandler )

int main( int argc, char **argv )
{
    return Game::main( argc, argv );
}
END_OF_MAIN()

int Game::main( int argc, char **argv )
{
    int rt = 0;
    
    set_uformat( U_UTF8 );    
    
    if ( allegro_init() < 0 )
    {
        printf( "One or more errors occurred.\nError Codes: %s\n\n", ER_ALLEGRO_INIT );
        return -1;
    }
    else
    {   
        char workingDirectory[ 1024 ] = emptyString;
        
        strcpy( fullPath, emptyString );

#ifndef _DEBUG
        get_executable_name( fullPath,
                            sizeof( fullPath ) );
        replace_filename( workingDirectory,
                        fullPath,
                        emptyString,
                        sizeof( workingDirectory ) );
        chdir( workingDirectory );
#endif      
    
        if ( ( rt = game.init() ) == 0 )
        {
            game.loop();
        }
                  
        if ( errorIndex > 0 )
        {
            char outputString[ 1024 ] = emptyString;
            
            for ( int i = 0; i < errorIndex; i++ )
            {
                char temp[ 8 ] = emptyString;
            
                sprintf( temp, "%d%s", errorList[ i ], ( i + 1 == errorIndex ? "" : ", " ) );
                strcat( outputString, temp );
            }
    
            allegro_message( "This game has aborted due to an error.\nCodes: %s\n\n", outputString );
        }
            
        game.kill();       
    }
    
    return rt;
}

Game::Game()
{
    this->buffer = NULL;
    this->datafile = NULL;
    secondCounter = 0;
}

Game::~Game()
{
}

void Game::loadGame()
{
    PACKFILE *pfile = NULL;

    if ( pfile = pack_fopen( saveGameFile, "rp" ) )
    {
        secondCounter = pack_igetl( pfile );

        this->board.load( pfile );

        pack_fclose( pfile );
    }
}

void Game::saveGame()
{
    PACKFILE *pfile = NULL;

    if ( pfile = pack_fopen( saveGameFile, "wp" ) )
    {
        pack_iputl( secondCounter, pfile );

        this->board.save( pfile );

        pack_fclose( pfile );
    }

    if ( exists( saveGameFile ) )
    {
        saveGameExists = true;
    }
}

int Game::init()
{  
    int rt = 0;
    
    set_window_title( gameWindowTitle );
    set_window_close_hook( Game::closeHook );

    if ( install_keyboard() < 0 )
    {
        return addError( ER_KEYBOARD_INSTALL );
    }
    
    if (  install_timer() < 0 )
    {
        return addError( ER_TIMER_INSTALL );
    }    

    if ( install_mouse() < 0  )
    {
        return addError( ER_MOUSE_INSTALL );
    }
    
    LOCK_VARIABLE( secondCounter );
    LOCK_FUNCTION( secondCounterHandler );    
    
    install_int( secondCounterHandler, 1000 );
   
    set_config_file( configFile );
    Misc::getConfig();
    Misc::getText();

    if ( ( rt = generator.load() ) < 0 )
    {
        return rt;
    }

    if ( ( rt = theme.load( themeName ) ) < 0 )
    {
        if ( ( rt = theme.load( defaultTheme ) ) < 0 )
        {
            return rt;
        }
        strcpy( themeName, defaultTheme );
    }
    strcpy( currentThemeName, themeName );

    if ( ( rt = Misc::getGfxMode() ) < 0 )
    {
        return rt;
    }

    if ( resFullScreen )
    {
        if ( set_display_switch_mode( SWITCH_BACKAMNESIA ) < 0 )
        {
            set_display_switch_mode( SWITCH_NONE );
        }
    }
    else
    {
        if ( set_display_switch_mode( SWITCH_BACKGROUND ) < 0 )
        {
            set_display_switch_mode( SWITCH_NONE );
        }
    }

    set_display_switch_callback( SWITCH_IN,  Misc::callIn );
    set_display_switch_callback( SWITCH_OUT, Misc::callOut );


    if ( theme.initTheme() < 0 )
    {
        return addError( ER_THEME_INIT );
    }

    theme.getGameInfo( gameInfo );
    this->board.init();


    acquire_screen();
    clear_bitmap( screen );
    textout( screen, font, textList[ textLoading ], 10, 10, makecol( 255, 255, 255 ) );
    release_screen();

    if ( !( this->buffer = create_bitmap( screenWidth,
                                    screenHeight ) ) )
    {
        return addError( ER_DOUBLEBUFFER_CREATE );
    }

    if ( !( datafile = load_datafile( dataFile ) ) )
    {
        return addError( ER_DATAFILE_LOAD );
    }

    mousePtrBitmap[ MP_ARROW ] = (BITMAP*)datafile[ BMP_MOUSE1 ].dat;
    mousePtrBitmap[ MP_HAND  ] = (BITMAP*)datafile[ BMP_MOUSE2 ].dat;

    srand( (unsigned int)time( NULL ) );
    text_mode( -1 );

    set_trans_blender( 0, 0, 0, 192 );

    gamestate = GS_GAME;

    return ER_NONE;
}

void Game::kill()
{
    if ( this->datafile )
    {
        unload_datafile( this->datafile );
        this->datafile = NULL;
    }

    if ( this->buffer )
    {
        destroy_bitmap( this->buffer );
        this->buffer = NULL;
    }

    set_gfx_mode( GFX_TEXT, 0, 0, 0, 0 );

    generator.unload();
    theme.uninitTheme();
    theme.unload();

    Misc::setConfig();

    allegro_exit();
}

void Game::loop()
{
    startNewGame = levelEasy;

    while ( !Game::done )
    {
        if ( startNewGame >= 0 )
        {
            gamestate = GS_GAME;

            secondCounter = 0;
            this->pauseGame();

            this->draw( buffer );
            if ( startNewGame > 0 )
            {
                theme.drawGenerating( buffer, textList[ textGeneratingPuzzle ] );
            }
            Misc::flip( buffer );

            newGame( startNewGame );
            startNewGame = -1;
            congratulations = false;

            this->unpauseGame();
            secondCounter = 0;
        }

        mouseX = mouse_x;
        mouseY = mouse_y;
        mouseB = mouse_b;

        this->input();
        this->processInput();

        this->draw( buffer );

        Game::drawMouse( buffer, mouseX, mouseY );
        Misc::flip( buffer );
        Misc::sleep();


        this->logic();
    }
}

void Game::draw( BITMAP *bitmap )
{
    if ( gamestate & GS_GAME )
    {
        theme.drawScreen( bitmap );

        this->board.draw( bitmap );

        if ( this->board.filled() )
        {
            blockIndex = 0;
        }
        else
        {
            if ( selectedCell )
            {
                selectedCell->drawSelection( bitmap );
            }
        }

        theme.drawTime( bitmap, textList[ textTime ], showTime, secondCounter );
    }

    if ( gamestate & GS_MAINMENU )
    {
        theme.drawMainMenu( bitmap, textList[ textMainMenu ] );
    }

    if ( gamestate & GS_THEMEMENU )
    {
        themeList.draw( bitmap );
    }

    if ( gamestate & GS_GAMESETTINGS )
    {
        theme.drawGameSettings( bitmap, textList[ textGameOptions ] );
    }

    if ( gamestate & GS_QUESTION )
    {
        theme.drawQuestion( bitmap, question );
    }
}

void Game::logic()
{
    if ( gamestate == GS_GAME )
    {
        this->gameLogic();
        return;
    }

    if ( gamestate == GS_MAINMENU )
    {
        this->mainMenuLogic();
        return;
    }

    if ( gamestate == GS_GAMESETTINGS )
    {
        this->OptionsMenuLogic();
        return;
    }

    if ( gamestate == GS_THEMEMENU )
    {
        this->themeMenuLogic();
        return;
    }

    if ( gamestate & GS_QUESTION )
    {
        this->questionLogic();
    }
}

void Game::input()
{
    static bool keyEscDown          = false;
    static bool keyF1Down           = false;
    static bool KeyCtrlDown         = false;
    static bool keyCtrlLDown        = false;
    static bool keyCtrlSDown        = false;
    static bool keyCtrlQDown        = false;
    static bool keyCtrlNDown        = false;

    if ( allowCheat )
    {
        if ( key[ KEY_F10 ] )
        {
            this->board.solve();
        }

        if ( key[ KEY_F5 ] )
        {
            this->board.fillTicks();
        }
    }

    for ( int i = 0; i <= 9; i++ )
    {
        Misc::doInput( ( key[ KEY_0 + i ] || key[ KEY_0_PAD + i ] ),
                   keyNumberDown[ i ],
                   keyNumberPushed[ i ] );
    }

    keyNumberDown[ 0 ] = ( keyNumberDown[ 0 ] || key[ KEY_SPACE ] );

    KeyCtrlDown = ( key[ KEY_LCONTROL ] || key[ KEY_RCONTROL ] );


    Misc::doInput( ( KeyCtrlDown || key[ KEY_Q ] ),
                   keyCtrlQDown,
                   keyCtrlQPushed );

    Misc::doInput( ( KeyCtrlDown || key[ KEY_N ] ),
                   keyCtrlNDown,
                   keyCtrlNPushed );

    Misc::doInput( ( KeyCtrlDown || key[ KEY_L ] ),
                   keyCtrlLDown,
                   keyCtrlLPushed );

    Misc::doInput( ( KeyCtrlDown || key[ KEY_S ] ),
                   keyCtrlSDown,
                   keyCtrlSPushed );



    Misc::doInput( ( key[ KEY_ESC ] != 0 ),
                   keyEscDown,
                   keyEscPushed );

    Misc::doInput( ( key[ KEY_F1 ] ),
                   keyF1Down,
                   keyF1Pushed );

    Misc::doInput( ( mouseB & 1 ),
                   mouse1Down,
                   mouse1Clicked );

    Misc::doInput( ( mouseB & 2 ),
                   mouse2Down,
                   mouse2Clicked );

    keyShiftDown = ( key[ KEY_LSHIFT ] || key[ KEY_RSHIFT ] );

}

void Game::processInput()
{
    if ( keyEscPushed )
    {
        keyEscPushed = false;

        switch ( gamestate )
        {
        case GS_GAME:
            {
                Game::closeHook();
                return;
            } break;

        case GS_MAINMENU:
            {
                this->unpauseGame();
                return;
            } break;

        case GS_GAMESETTINGS:
            {
                gamestate = GS_MAINMENU;
                return;
            } break;

        case GS_THEMEMENU:
            {
                gamestate = GS_GAMESETTINGS;
                return;
            } break;
        }

        if ( gamestate & GS_CLOSEHOOK )
        {
            gamestate ^= GS_CLOSEHOOK;

            if ( gamestate == GS_GAME )
            {
                this->unpauseGame();
            }
        }

        if ( gamestate & GS_QUESTION )
        {
            gamestate ^= GS_QUESTION;

            if ( gamestate == GS_GAME )
            {
                this->unpauseGame();
            }
        }
    }

    if ( keyCtrlNPushed )
    {
        keyCtrlNPushed = false;
        this->doGameButton( B_KEY_NEW_GAME );
        return;
    }

    if ( keyCtrlQPushed )
    {
        keyCtrlQPushed = false;
        Game::closeHook();
        return;
    }

    if ( keyCtrlSPushed )
    {
        keyCtrlSPushed = false;
        this->doGameButton( B_KEY_SAVE_GAME );
        return;
    }

    if ( keyCtrlLPushed )
    {
        keyCtrlLPushed = false;
        this->doGameButton( B_KEY_LOAD_GAME );
        return;
    }

    if ( keyF1Pushed )
    {
        keyF1Pushed = false;

        if ( !gamePaused )
        {
            this->doGameButton( B_GAME_SHOWTICK );
        }
    }
}


void Game::gameLogic()
{
    int value = 0;
    bool numberPlaced = false;

    if ( !congratulations && board.filled() )
    {
        gamestate |= GS_QUESTION;
        theme.setQuestionText( question,
                                textList[ textCompletePuzzle ],
                                textList[ textOk ] );

        congratulations = true;
        lastButton = -1;
    }

    if ( this->board.filled() )
    {
        showChoices = false;

        for ( int i = B_GAME_PAUSE; i <= B_GAME_MENU; i++ )
        {
            if ( i != B_GAME_MENU )
            {
                gameInfo.button[ i ].flags = ( BS_NORMAL | BS_DISABLED );
            }
        }
    }
    else
    {
        if ( gamePaused || blankBoard )
        {
            if ( blankBoard )
            {
                secondCounter = 0;
            }
        }
    }

    mousePtr = MP_ARROW;

    value = doButtonLogic( gameInfo.button,
                        B_GAME_PAUSE,
                        B_GAME_MENU );

    if ( value >= 0 )
    {
        this->doGameButton( value );
    }

    if ( gamePaused )
    {
    }
    else
    {
        mouseOver = this->board.logic();

        if ( mouseOver || selectedCell )
        {
            mousePtr = MP_HAND;

            if ( mouseOver && !mouseOver->solid )
            {
                for ( int i = 0; i <= 9; i++ )
                {

                    if ( keyShiftDown )
                    {
                        if ( allowTicks && keyNumberPushed[ i ] )
                        {
                            if ( i == 0 )
                            {
                                mouseOver->addTick( 0 );
                            }
                            else
                            {
                                for ( int c = 0; c < gridSize; c++ )
                                {
                                    if ( mouseOver->choices[ c ] == i )
                                    {
                                        mouseOver->addTick( i );
                                    }
                                }
                            }
                        }
                    }
                    else
                    {
                        if ( keyNumberDown[ i ] )
                        {
                            if ( i == 0 )
                            {
                                this->board.placeNumber( mouseOver, 0 );
                                numberPlaced = true;
                            }
                            else
                            {
                                for ( int c = 0; c < gridSize; c++ )
                                {
                                    if ( mouseOver->choices[ c ] == i )
                                    {
                                        this->board.placeNumber( mouseOver, i );
                                        numberPlaced = true;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        if ( !this->board.filled() )
        {
            if ( mouse1Clicked )
            {
                mouse1Clicked = false;

                if ( !selectedCell )
                {
                    if ( mouseOver )
                    {
                        if ( mouseOver->solid )
                        {
                            if ( blockIndex == mouseOver->value )
                            {
                                blockIndex = 0;
                            }
                            else
                            {
                                blockIndex = mouseOver->value;
                            }
                        }
                        else
                        {
                            selectedCell = mouseOver;
                        }
                    }
                    else
                    {
                        blockIndex = 0;
                    }
                }
                else
                {
                    if ( !selectedCell->solid )
                    {
                        if ( allowTicks && keyShiftDown )
                        {
                            showChoices = true;
                            selectedCell->doAddTick();
                        }
                        else
                        {
                            int newvalue = selectedCell->doSelection();

                            this->board.placeNumber( selectedCell, newvalue );
                            numberPlaced = true;
                        }
                    }
                    selectedCell = NULL;
                }
            }
        }
    }

    if ( numberPlaced )
    {
        if ( board.canUndo() )
        {
            gameInfo.button[ B_GAME_UNDO ].flags = BS_NORMAL;
            gameInfo.button[ B_GAME_RESET ].flags = BS_NORMAL;
        }
    }

    mouse1Clicked = false;
}

int Game::newGame( int difficulty )
{
    int rt = 0;
    blockIndex = 0;

    if ( difficulty == 0 )
    {
        strcpy( gameInfo.button[ B_GAME_PAUSE ].text, textList[ textStart ] );
        blankBoard = true;
    }
    else
    {
        blankBoard = false;
    }

    rt = this->board.newGame( difficulty );

    secondCounter = 0;

    return rt;
}

void Game::drawMouse( BITMAP *bitmap, int x, int y )
{
    if ( _mouse_on )
    {
        draw_sprite( bitmap,
                     mousePtrBitmap[ mousePtr ],
                     x - mousePtrX[ mousePtr ],
                     y - mousePtrY[ mousePtr ] );
    }
}

void Game::doGameButton( int index )
{
    lastButton = index;

    selectedCell = NULL;
    mouseOver = NULL;
    blockIndex = 0;

    switch ( index )
    {
    case B_GAME_PAUSE:
        {
            if ( gamePaused )
            {
                this->unpauseGame();
            }
            else
            {
                if ( blankBoard )
                {
                    int rt = board.activateBlankPuzzle();

                    if ( rt == 0 )
                    {
                        blankBoard = false;
                        this->unpauseGame();
                    }
                    else
                    {
                        int textNumber = textNotValid;

                        if ( rt == -2 )
                        {
                            textNumber = textNotUnique;
                        }

                        gamestate |= GS_QUESTION;
                        theme.setQuestionText( question,
                                                textList[ textNumber ],
                                                textList[ textOk ] );
                    }
                }
                else
                {
                    this->pauseGame();
                }
            }
        } break;

    case B_GAME_UNDO:
        {
            this->board.doUndo();
            if ( !this->board.canUndo() )
            {
                gameInfo.button[ B_GAME_UNDO ].flags = ( BS_NORMAL | BS_DISABLED );
                gameInfo.button[ B_GAME_RESET ].flags = ( BS_NORMAL | BS_DISABLED );
            }
        } break;

    case B_GAME_RESET:
        {
            gamestate |= GS_QUESTION;
            theme.setQuestionText( question,
                                   textList[ textResetBoardQuestion ],
                                   textList[ textYes ],
                                   textList[ textNo ] );
        } break;

    case B_GAME_SHOWTICK:
        {
            showChoices = !showChoices;
            if ( showChoices )
            {
                gameInfo.button[ index ].flags = BS_DOWN;
            }
            else
            {
                gameInfo.button[ index ].flags = BS_NORMAL;
            }
        } break;

    case B_GAME_MENU:
        {
            this->pauseGame();
            gamestate = GS_MAINMENU;

            if ( board.filled() )
            {
                gameInfo.button[ B_MENU_SAVE ].flags = ( BS_NORMAL | BS_DISABLED );
            }
            else
            {
                gameInfo.button[ B_MENU_SAVE ].flags = BS_NORMAL;
            }

            if ( exists( saveGameFile ) )
            {
                gameInfo.button[ B_MENU_LOAD ].flags = BS_NORMAL;
                saveGameExists = true;
            }
            else
            {
                gameInfo.button[ B_MENU_LOAD ].flags = ( BS_NORMAL | BS_DISABLED );
            }
        } break;

    case B_MENU_RESUME:
        {
            this->unpauseGame();
        } break;

    case B_KEY_NEW_GAME:
    case B_MENU_NEW:
        {
            if ( this->board.filled() )
            {
                this->doGameButton( B_CHOOSE_LEVEL );
            }
            else
            {
                gamestate |= GS_QUESTION;
                theme.setQuestionText( question,
                                       textList[ textNewGameQuestion ],
                                       textList[ textYes ],
                                       textList[ textNo ] );
            }
        } break;

    case B_KEY_LOAD_GAME:
    case B_MENU_LOAD:
        {
            if ( this->board.filled() )
            {
                this->doGameButton( B_LOAD_GAME );
            }
            else
            {
                this->pauseGame();
                gamestate |= GS_QUESTION;
                theme.setQuestionText( question,
                                       textList[ textLoadGameQuestion ],
                                       textList[ textYes ],
                                       textList[ textNo ] );
            }
        } break;

    case B_KEY_SAVE_GAME:
    case B_MENU_SAVE:
        {
            if ( exists( saveGameFile ) )
            {
                this->pauseGame();
                gamestate |= GS_QUESTION;
                theme.setQuestionText( question,
                                       textList[ textOverwriteGameQuestion ],
                                       textList[ textYes ],
                                       textList[ textNo ] );
            }
            else
            {
                this->doGameButton( B_SAVE_GAME );
            }
        } break;

    case B_MENU_OPTIONS:
        {
            gameInfo.button[ B_OPTIONS_ALLOWCHEAT ].flags = ( allowCheat ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_ALLOWMARKS ].flags = ( allowTicks ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_SHOWTIME ].flags = ( showTime ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_SHOWCORRECT ].flags = ( showCorrect ? BS_DOWN : BS_NORMAL );

            gameInfo.button[ B_OPTIONS_640X480 ].flags = ( ( SCREEN_W == 640 && SCREEN_H == 480 ) ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_800X600 ].flags = ( ( SCREEN_W == 800 && SCREEN_H == 600 ) ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_1024X768 ].flags = ( ( SCREEN_W == 1024 && SCREEN_H == 768 ) ? BS_DOWN : BS_NORMAL );

            gameInfo.button[ B_OPTIONS_FULLSCREEN ].flags = ( resFullScreen ? BS_DOWN : BS_NORMAL );

            gamestate = GS_GAMESETTINGS;
        } break;

    case B_MENU_CHANGETHEME:
        {
            themeList.init();
            gamestate = GS_THEMEMENU;
        } break;

    case B_MENU_ABOUT:
        {
            char stringBuffer[ 1024 ];

            sprintf( stringBuffer,
                     textList[ textAboutGame ],
                     ( gameVersion & 0xff000000 ) >> 24,
                     ( gameVersion & 0x00ff0000 ) >> 16,
                     ( gameVersion & 0x0000ff00 ) >> 8,
                     ( gameVersion & 0x000000ff ) );

            gamestate |= GS_QUESTION;
            theme.setQuestionText( question,
                                   stringBuffer,
                                   textList[ textOk ] );
        } break;

    case B_MENU_QUIT:
        {
            Game::closeHook();
        } break;


    case B_OPTIONS_ALLOWCHEAT:
        {
            allowCheat = !allowCheat;
            gameInfo.button[ B_OPTIONS_ALLOWCHEAT ].flags = ( allowCheat ? BS_DOWN : BS_NORMAL );
        } break;

    case B_OPTIONS_ALLOWMARKS:
        {
            allowTicks = !allowTicks;
            gameInfo.button[ B_OPTIONS_ALLOWMARKS ].flags = ( allowTicks ? BS_DOWN : BS_NORMAL );
        } break;

    case B_OPTIONS_SHOWTIME:
        {
            showTime = !showTime;
            gameInfo.button[ B_OPTIONS_SHOWTIME ].flags = ( showTime ? BS_DOWN : BS_NORMAL );
        } break;

    case B_OPTIONS_SHOWCORRECT:
        {
            showCorrect = !showCorrect;
            gameInfo.button[ B_OPTIONS_SHOWCORRECT ].flags = ( showCorrect ? BS_DOWN : BS_NORMAL );
        } break;

    case B_OPTIONS_640X480:
        {
            int oldWidth = SCREEN_W;
            int oldHeight = SCREEN_H;

            if ( oldWidth != 640 &&
                 oldHeight != 480 )
            {
                if ( set_gfx_mode( ( resFullScreen  ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED ),
                                   640,
                                   480,
                                   0, 0 ) < 0 )
                {
                    set_gfx_mode( ( resFullScreen  ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED ),
                                  oldWidth,
                                  oldHeight,
                                  0, 0 );
                }
            }

            resWidth = SCREEN_W;
            resHeight = SCREEN_H;

            gameInfo.button[ B_OPTIONS_640X480 ].flags = ( ( SCREEN_W == 640 && SCREEN_H == 480 ) ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_800X600 ].flags = ( ( SCREEN_W == 800 && SCREEN_H == 600 ) ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_1024X768 ].flags = ( ( SCREEN_W == 1024 && SCREEN_H == 768 ) ? BS_DOWN : BS_NORMAL );
        } break;

    case B_OPTIONS_800X600:
        {
            int oldWidth = SCREEN_W;
            int oldHeight = SCREEN_H;

            if ( oldWidth != 800 &&
                 oldHeight != 600 )
            {
                if ( set_gfx_mode( ( resFullScreen  ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED ),
                                   800,
                                   600,
                                   0, 0 ) < 0 )
                {
                    set_gfx_mode( ( resFullScreen  ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED ),
                                  oldWidth,
                                  oldHeight,
                                  0, 0 );
                }
            }

            resWidth = SCREEN_W;
            resHeight = SCREEN_H;

            gameInfo.button[ B_OPTIONS_640X480 ].flags = ( ( SCREEN_W == 640 && SCREEN_H == 480 ) ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_800X600 ].flags = ( ( SCREEN_W == 800 && SCREEN_H == 600 ) ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_1024X768 ].flags = ( ( SCREEN_W == 1024 && SCREEN_H == 768 ) ? BS_DOWN : BS_NORMAL );
        } break;

    case B_OPTIONS_1024X768:
        {
            int oldWidth = SCREEN_W;
            int oldHeight = SCREEN_H;

            if ( oldWidth != 1024 &&
                 oldHeight != 768 )
            {
                if ( set_gfx_mode( ( resFullScreen  ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED ),
                                   1024,
                                   768,
                                   0, 0 ) < 0 )
                {
                    set_gfx_mode( ( resFullScreen  ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED ),
                                  oldWidth,
                                  oldHeight,
                                  0, 0 );
                }
            }

            resWidth = SCREEN_W;
            resHeight = SCREEN_H;

            gameInfo.button[ B_OPTIONS_640X480 ].flags = ( ( SCREEN_W == 640 && SCREEN_H == 480 ) ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_800X600 ].flags = ( ( SCREEN_W == 800 && SCREEN_H == 600 ) ? BS_DOWN : BS_NORMAL );
            gameInfo.button[ B_OPTIONS_1024X768 ].flags = ( ( SCREEN_W == 1024 && SCREEN_H == 768 ) ? BS_DOWN : BS_NORMAL );
        } break;

    case B_OPTIONS_FULLSCREEN:
        {
            resFullScreen = !resFullScreen;

            if ( set_gfx_mode( ( resFullScreen  ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED ),
                                SCREEN_W,
                                SCREEN_H,
                                0, 0 ) < 0 )
            {
                resFullScreen = !resFullScreen;
                set_gfx_mode( ( resFullScreen  ? GFX_AUTODETECT_FULLSCREEN : GFX_AUTODETECT_WINDOWED ),
                              SCREEN_W,
                              SCREEN_H,
                              0, 0 );
            }

            gameInfo.button[ B_OPTIONS_FULLSCREEN ].flags = ( resFullScreen ? BS_DOWN : BS_NORMAL );
        } break;

    case B_OPTIONS_OK:
        {
            gamestate = GS_MAINMENU;
        } break;

    case B_THEME_UP:
        {
            themeList.moveUp();
        } break;

    case B_THEME_DOWN:
        {
            themeList.moveDown();
        } break;

    case B_THEME_OK:
        {
            if ( stricmp( currentThemeName, themeList.getCurrentFilename() ) != 0 )
            {
                if ( this->changeTheme( themeList.getCurrentFilename() ) < 0 )
                {
                    if ( this->changeTheme( currentThemeName ) < 0 )
                    {
                        if ( this->changeTheme( defaultTheme ) < 0 )
                        {
                            Game::done = true;
                            return;
                        }
                    }
                }
            }
            themeList.apply();
            themeList.uninit();
            gamestate = GS_MAINMENU;
        } break;

    case B_THEME_CANCEL:
        {
            themeList.uninit();
            gamestate = GS_MAINMENU;
        } break;

    case B_CHOOSE_LEVEL:
        {
            gamestate |= GS_QUESTION;
            theme.setQuestionText( question,
                                   textList[ textChooseDifficulty ],
                                   textList[ textBlank ],
                                   textList[ textEasy ],
                                   textList[ textMedium ],
                                   textList[ textHard ] );
        } break;

    case B_CLOSE_HOOK:
        {
            gamestate |= GS_QUESTION;
            theme.setQuestionText( question,
                                   textList[ textQuitGameQuestion ],
                                   textList[ textYes ],
                                   textList[ textNo ] );
        } break;

    case B_LOAD_GAME:
        {
            this->loadGame();
            this->unpauseGame();
        } break;

    case B_SAVE_GAME:
        {
            this->saveGame();
            this->unpauseGame();
        } break;

    default:
        {
            return;
        }
    }

    mouse1Clicked = false;
}

void Game::mainMenuLogic()
{
    int value = 0;
    mousePtr = MP_ARROW;

    value = doButtonLogic( gameInfo.button,
                           B_MENU_RESUME,
                           B_MENU_QUIT );

    if ( value >= 0 )
    {
        this->doGameButton( value );
    }

    mouse1Clicked = false;
}

void Game::OptionsMenuLogic()
{
    int value = 0;
    mousePtr = MP_ARROW;

    value = doButtonLogic( gameInfo.button,
                           B_OPTIONS_GAMESETTINGS,
                           B_OPTIONS_OK );

    if ( value >= 0 )
    {
        this->doGameButton( value );
    }

    mouse1Clicked = false;
}

void Game::themeMenuLogic()
{
    int value = 0;
    mousePtr = MP_ARROW;

    value = doButtonLogic( gameInfo.button,
                           B_THEME_UP,
                           B_THEME_OK );

    themeList.logic();

    if ( value >= 0 )
    {
        this->doGameButton( value );
    }

    mouse1Clicked = false;
}

void Game::questionLogic()
{
    int value = 0;
    mousePtr = MP_ARROW;

    value = doButtonLogic( question.button,
                           0,
                           question.count - 1 );

    if ( value >= 0 )
    {
        if ( gamestate & GS_QUESTION )
        {
            gamestate ^= GS_QUESTION;
        }

        if ( lastButton == B_CHOOSE_LEVEL )
        {
            startNewGame = value;
            return;
        }

        if ( value == 0 )
        {
            switch( lastButton )
            {
            case B_GAME_RESET:
                {
                    this->board.reset();
                    gameInfo.button[ B_GAME_UNDO ].flags = ( BS_NORMAL | BS_DISABLED );
                    gameInfo.button[ B_GAME_RESET ].flags = ( BS_NORMAL | BS_DISABLED );
                } break;

            case B_MENU_LOAD:
            case B_KEY_LOAD_GAME:
                {
                    this->doGameButton( B_LOAD_GAME );
                } break;

            case B_MENU_NEW:
            case B_KEY_NEW_GAME:
                {
                    this->doGameButton( B_CHOOSE_LEVEL );
                } break;

            case B_MENU_SAVE:
            case B_KEY_SAVE_GAME:
                {
                    this->doGameButton( B_SAVE_GAME );
                } break;

            case B_CLOSE_HOOK:
                {
                    Game::done = true;
                } break;
            }
        }
        else
        {
            if ( gamestate & GS_CLOSEHOOK )
            {
                gamestate ^= GS_CLOSEHOOK;
                set_window_close_hook( Game::closeHook );
            }
        }

        if ( gamestate == GS_GAME )
        {
            this->unpauseGame();
        }
    }

    mouse1Clicked = false;
}


int Game::doButtonLogic( ButtonInfo *buttonList, int start, int finish )
{
    for ( int i = start; i <= finish; i++ )
    {
        if ( !( buttonList[ i ].flags & BS_DISABLED ) )
        {
            if ( mouseX >= buttonList[ i ].rectInfo.x &&
                 mouseX <  buttonList[ i ].rectInfo.x + buttonList[ i ].rectInfo.width &&
                 mouseY >= buttonList[ i ].rectInfo.y &&
                 mouseY <  buttonList[ i ].rectInfo.y + buttonList[ i ].rectInfo.height )
            {
                mousePtr = MP_HAND;

                if ( !( buttonList[ i ].type == BT_RADIO ||
                        buttonList[ i ].type == BT_CHECK ) )
                {
                    buttonList[ i ].flags = BS_OVER;
                }

                if ( mouse1Down )
                {
                    if ( !( buttonList[ i ].type == BT_RADIO ||
                            buttonList[ i ].type == BT_CHECK ) )
                    {
                        buttonList[ i ].flags = BS_DOWN;
                    }
                }

                if ( mouse1Clicked )
                {
                    mouse1Clicked = false;
                    mousePtr = MP_ARROW;

                    if ( !( buttonList[ i ].type == BT_RADIO ||
                            buttonList[ i ].type == BT_CHECK ) )
                    {
                        buttonList[ i ].flags = BS_NORMAL;
                    }

                    return i;
                }
            }
        }
    }

    return -1;
}

int Game::changeTheme( char *filename )
{
    int rt = 0;

    theme.uninitTheme();
    theme.unload();

    if ( ( rt = theme.load( filename ) ) == 0 )
    {
        if ( ( rt = theme.initTheme() ) < 0 )
        {
            return -1;
        }

        theme.getGameInfo( gameInfo );
        this->board.init();

        strcpy( currentThemeName, filename );

        return 0;
    }

    return -1;
}


void Game::closeHook()
{
    gamestate |= GS_CLOSEHOOK;
    set_window_close_hook( NULL );
    game.pauseGame();
    game.doGameButton( B_CLOSE_HOOK );
}


void Game::pauseGame()
{
    strcpy( gameInfo.button[ B_GAME_PAUSE ].text, textList[ textResume ] );
    gamePaused = true;

    gameInfo.button[ B_GAME_UNDO ].flags = ( BS_NORMAL | BS_DISABLED );
    gameInfo.button[ B_GAME_RESET ].flags = ( BS_NORMAL | BS_DISABLED );

    if ( showChoices && allowTicks )
    {
        gameInfo.button[ B_GAME_SHOWTICK  ].flags = ( BS_DOWN | BS_DISABLED );;
    }
    else
    {
        gameInfo.button[ B_GAME_SHOWTICK  ].flags = ( BS_NORMAL | BS_DISABLED );;
    }

    gameInfo.button[ B_GAME_MENU ].flags = ( BS_NORMAL | BS_DISABLED );
}

void Game::unpauseGame()
{
    if ( blankBoard )
    {
        strcpy( gameInfo.button[ B_GAME_PAUSE ].text, textList[ textStart ] );
    }
    else
    {
        strcpy( gameInfo.button[ B_GAME_PAUSE ].text, textList[ textPause ] );
    }
    gamestate = GS_GAME;
    gamePaused = false;

    gameInfo.button[ B_GAME_PAUSE ].flags = BS_NORMAL;

    if ( board.filled() )
    {
        gameInfo.button[ B_GAME_PAUSE ].flags |= BS_DISABLED;
    }

    if ( board.canUndo() )
    {
        gameInfo.button[ B_GAME_UNDO ].flags = BS_NORMAL;
        gameInfo.button[ B_GAME_RESET ].flags = BS_NORMAL;
    }

    if ( showChoices )
    {
        gameInfo.button[ B_GAME_SHOWTICK  ].flags = BS_DOWN;
    }
    else
    {
        gameInfo.button[ B_GAME_SHOWTICK  ].flags = BS_NORMAL;
    }

    if ( !allowTicks )
    {
        gameInfo.button[ B_GAME_SHOWTICK  ].flags |= BS_DISABLED;
    }

    gameInfo.button[ B_GAME_MENU ].flags = BS_NORMAL;
}

