#pragma warning( disable: 4312 )

#include <allegro.h>
#include <string>
#include <vector>
#include <map>

#include <common.h>

class D_CDLXSolver {

public:

    D_CDLXSolver();
    virtual ~D_CDLXSolver();
    void AddColumn(const std::string& name, bool mandatory=true);
    void NewRow();
    void SetColumn(const std::string& name);
    void SetColumn(unsigned int number);
    void DisableRow(unsigned int number);
    void EnableRow(unsigned int number);
    void DeleteRows();
    void Solve();           

protected:

    virtual bool Record();
    std::string* GetSolution();
    unsigned int dequeRemovals;
    unsigned int numSolutions;

private:

    // private classes
    class CNode;
    class CHeader;
    
    // main 2D doubly-linked structure
    CHeader *root;

    // exhaustive list of columns
    std::vector<CHeader*> columns;
    std::map<std::string, int> names;
    	
    // list of non-empty rows
    std::vector<CNode*> rows;

    // solution stack, and pointers to browse it
    std::vector<CNode*> stack;
    std::vector<CNode*>::iterator iterStack;
    CNode *iterRow;

    // Search() temporary variables (not local to reduce recursion load)
    CHeader *column;
    CNode   *row;
    bool     more;
    
    // search methods
    void Search();          
    void Cover(CHeader* col);
    void Uncover(CHeader* col);
    void Choose();          
};


/* Solver class implementation ============================================== */

#include <iostream>
#include <sstream>

using namespace std;

class D_CDLXSolver::CNode {

public:

    CNode *left, *right, *up, *down;
    CHeader *head;
};

class D_CDLXSolver::CHeader : public D_CDLXSolver::CNode {

public:

    int size;
    string name;
};

void D_CDLXSolver::AddColumn(const string& name, bool mandatory) {

    CHeader* col = new CHeader;
    col->name = name;
    col->size = 0;
    col->up = col->down = col->left = col->right = col->head = col;
    if (mandatory) {
        col->right = root;
        col->left = root->left;
        root->left->right = col;
        root->left = col;
    }
    names[name] = (int)columns.size();
    columns.push_back(col);
}

void D_CDLXSolver::NewRow() {

    row = NULL;
}

void D_CDLXSolver::DeleteRows() {

    for (vector<CHeader*>::iterator it = columns.begin();
                                    it != columns.end(); it++) {
        (*it)->down = (*it)->up = *it;
    }
    for (vector<CNode*>::iterator it = rows.begin();
                                  it != rows.end(); it++) {
        for (CNode *j = (*it)->right; j != *it; ) {
            CNode* j_right = j->right; delete j; j = j_right;
        }
        delete *it;
    }
    rows.clear();
    row = NULL;
}

void D_CDLXSolver::DisableRow(unsigned int number) {

    if (number >= rows.size()) return;
    CNode* i = rows[number];
    if (i->up == i) return; // already disabled
    CNode* j = i;
    do {
    	j->up->down = j->down;
    	j->down->up = j->up;
    	j->down = j->up = j;
    	j->head->size--;
    	j = j->right;
    } while (i != j);
}

void D_CDLXSolver::EnableRow(unsigned int number) {
	
    if (number >= rows.size()) return;
    CNode* i = rows[number];
    if (i->up != i) return; // already enabled
    CNode* j = i;
    do {
    	j->up = j->head;
    	j->down = j->head->down;
    	j->up->down = j;
    	j->down->up = j;
    	j->head->size++;
    	j = j->right;
    } while (i != j);
}

void D_CDLXSolver::SetColumn(const string& name) {

    if (names.find(name) == names.end()) return;
    SetColumn(names[name]);
}

void D_CDLXSolver::SetColumn(unsigned int number) {

    if (number >= columns.size()) return;
    CHeader* header = columns[number];
    CNode* node = new CNode;
    if (row == NULL) {
        row = node;
        rows.push_back(node);
        node->left  = node;
        node->right = node;
    } else {
        node->left = row;
        node->right = row->right;
        row->right->left = node;
        row->right = node;
    }
    node->head = header;
    node->up = header;
    node->down = header->down;
    header->down->up = node;
    header->down = node;
    header->size++;
}

string* D_CDLXSolver::GetSolution() {

    if (iterRow != NULL) {
        string* ret = &iterRow->head->name;
        iterRow = iterRow->right;
        if (iterRow == *iterStack) iterRow = NULL;
        return ret;
    }
    if (iterStack != stack.end() && ++iterStack != stack.end()) {
        iterRow = *iterStack;
    }
    return NULL;
}

bool D_CDLXSolver::Record() {

    cout << "Solution (found after " << dequeRemovals
         << " deque removals):" << endl;
    for (string* s = GetSolution(); s != NULL; s = GetSolution()) {
        for (; s != NULL; s = GetSolution()) {
            cout << *s << " ";
        }
        cout << endl;
    }
    cout << endl;
    return true;
}

void D_CDLXSolver::Solve() {

    more = true;
    stack.clear();
    dequeRemovals = 0;
    numSolutions = 0;
    Search();
}

void D_CDLXSolver::Search() {

    if (root->right == root) {
        numSolutions++;
        iterStack = stack.begin();
        iterRow = (iterStack != stack.end() ? *iterStack : NULL);
        more = Record();
        return;
    }
    Choose();
    Cover(column);
    stack.push_back(NULL);
    for (row = column->down; row != column; row = row->down) {
        for (CNode *i = row->right; i != row; i = i->right) {
            Cover(i->head);
        }
        stack.back() = row;
        Search();
        row = stack.back(); column = row->head;
        for (CNode *i = row->left; i != row; i = i->left) {
            Uncover(i->head);
        }
        if (!more) break;
    }
    stack.pop_back();
    Uncover(column);
}

void D_CDLXSolver::Cover(CHeader* col) {

    col->right->left = col->left;
    col->left->right = col->right;
    dequeRemovals++;
    for (CNode *i = col->down; i != col; i = i->down) {
        for (CNode *j = i->right; j != i; j = j->right) {
            j->down->up = j->up;
            j->up->down = j->down;
            j->head->size--;
            dequeRemovals++;
        }
    }
}

void D_CDLXSolver::Uncover(CHeader* col) {

    for (CNode *i = col->up; i != col; i = i->up) {
        for (CNode *j = i->left; j != i; j = j->left) {
            j->head->size++;
            j->down->up = j;
            j->up->down = j;
        }
    }
    col->right->left = col;
    col->left->right = col;
}

void D_CDLXSolver::Choose() {

    int best = INT_MAX;
    for (CHeader* i = (CHeader*)root->right; i != root;
                  i = (CHeader*)i->right) {
        if (i->size < best) {
            column = i;
            best = i->size;
        }
    }
}

D_CDLXSolver::D_CDLXSolver() {

    root = new CHeader;
    root->left = root->right = root;
    row = NULL;
}

D_CDLXSolver::~D_CDLXSolver() {

    for (vector<CHeader*>::iterator it = columns.begin();
                                         it != columns.end(); it++) {
        for (CNode *j = (*it)->down; j != *it; ) {
            CNode* j_down =  j->down; delete j; j = j_down;
        }
        delete *it;
    }
    delete root;
}


/* Specialized class for Sudoku ============================================= */

class D_CSudokuSolver : private D_CDLXSolver {
	
public:
	
	D_CSudokuSolver();
	int grid[9][9];
	void Solve();
	int solution[9][9];
	int count;
	static void Print(int grid[9][9]);

private:

    virtual bool Record();
};

D_CSudokuSolver::D_CSudokuSolver() {

    stringstream ss;
    // define columns for individual cells
    // ("a cell contains 1 and only 1 digit")
    for (int R = 0; R < 9; R++) {
        for (int C = 0; C < 9; C++) {
            ss.str("");
            ss << "R" << (R+1) << "C" << (C+1);
            AddColumn(ss.str());
        }
    }
    
    // define columns for each group containing each digit
    // ("each digit appears once and only once in each group")
    for (int d = 1; d <= 9; d++) {
        for (int i = 0; i < 3; i++) {
            for (int j = 1; j <= 9; j++) {
                ss.str("");
                ss << d << "in" << "RCB"[i] << j;
                AddColumn(ss.str());
            }
        }
    }

    // create rows ("there could be any digit in any empty cell")
    for (int R = 0; R < 9; R++) {
        for (int C = 0; C < 9; C++) {
            for (int d = 1; d <= 9; d++) {
                NewRow();
                SetColumn(R*9+C);
                SetColumn(54 + d*27 + R);
                SetColumn(63 + d*27 + C);
                SetColumn(72 + d*27 + (R/3)*3+(C/3));
                /*
                // same thing, a bit slower but more extensible:
			    stringstream ss;
                ss.str(""); ss<<"R"<<(R+1)<<"C"<<(C+1);      SetColumn(ss.str());
                ss.str(""); ss<<d<<"inR"<<(R+1); 			 SetColumn(ss.str());
                ss.str(""); ss<<d<<"inC"<<(C+1); 			 SetColumn(ss.str());
                ss.str(""); ss<<d<<"inB"<<((R/3)*3+(C/3)+1); SetColumn(ss.str());
                */
            }
        }
    }
}

void D_CSudokuSolver::Solve() {

    // disable rows not satisfying fixed cells
    for (int R = 0; R < 9; R++) {
        for (int C = 0; C < 9; C++) {
        	if (grid[R][C] < 0) continue;
        	for (int d = 1; d <= 9; d++) {
        		if (grid[R][C] != d) DisableRow(R*81+C*9+d-1);
        	}
        }
    }
    count = 0;
    D_CDLXSolver::Solve();
	// restore rows	
    for (int R = 0; R < 9; R++) {
        for (int C = 0; C < 9; C++) {
        	if (grid[R][C] < 0) continue;
        	for (int d = 1; d <= 9; d++) {
        		if (grid[R][C] != d) EnableRow(R*81+C*9+d-1);
        	}
        }
    }
}
	
bool D_CSudokuSolver::Record() {

	if (++count > 1) return false;
    for (string* s = GetSolution(); s != NULL; s = GetSolution()) {
        int R=-1, C=-1;
        for (; s != NULL; s = GetSolution()) {
            if (s->length()<5) continue;
            if ((*s)[3] == 'R') R = (*s)[4] - '1';
            if ((*s)[3] == 'C') C = (*s)[4] - '1';
            if (R >= 0 && C >= 0) solution[R][C] = (*s)[0] - '0';
        }
    }
    return true;
}

void D_CSudokuSolver::Print(int grid[9][9]) {

    for (int R=0; R<9; R++) {
    	cout << " ";
        for (int C=0; C<9; C++) {
            cout << (grid[R][C]<0?'.':char(grid[R][C]+'0')) << " ";
            if (C == 2 || C == 5) cout << "| ";
        }
        cout << endl;
        if (R == 2 || R == 5) cout << " ------+-------+------ " << endl;
    }
    cout << endl;
}

GAME_FUNC( int, solve, ( const char grid[ 9 ][ 9 ], char solution[ 9 ][ 9 ] ) )
{
    D_CSudokuSolver solver;

    for ( int row = 0; row < 9; row++ )
    {
        for ( int col = 0; col < 9; col++ )
        {
            solver.solution[ row ][ col ] = 0;
            solver.grid[ row ][ col ] = grid[ row ][ col ];
            if ( grid[ row ][ col ] == 0 )
            {
                solver.grid[ row ][ col ] = '.' - '0';
            }
        }
    }

    solver.Solve();

    if ( solver.count == 0 )
    {
        return -1;
    }

    if ( solver.count > 1 )
    {
        return -2;
    }

    for ( int row = 0; row < 9; row++ )
    {
        for ( int col = 0; col < 9; col++ )
        {
            solution[ row ][ col ] = solver.solution[ row ][ col ];
        }
    }

    return 0;
}

