/*

  file containing the game loop functions for dash
  the idea is to have two loops:

  - level_loop : called once for each level. It should (re)initialise
  all game variables, load a level, and then call game_loop until it
  indicates it is done.

  - game_loop: draw a frame.  After that, call logic_loop as many
  times as needed to 'catch up' with the clock. Return value should
  indicate if the player wants to quit: take this from logic_loop

  - logic_loop: handle all objects in the active list, and read
  keyboard / mouse input from the player.

*/

#include <stdio.h>
#include <signal.h>
#include <math.h>
#include "gameloop.h"
#include "globals.h"
#include "utils.h"
#include "draw.h"
#include "timer.h"
#include "placehld.h"
#include "highscor.h"
#include "banner.h"

int level_loop(char const *filename)
{
    int ret = LOOP_CONTINUE;
    FILE *fp;
    
    init_globals();

    reset_graphics();
    
    // load a level and initialise variables
    if (!(fp = fopen(filename, "r")))
    {
	warning("level_loop: could not open '%s'", filename);
	return LOOP_QUIT;
    }
    if (!gr->load(fp))
    {
	warning("level_loop: failed to load level '%s'", filename);
	return LOOP_QUIT;
    }
    fclose(fp);
    
    // wake up all objects, once
    gr->wake_all();

    // run the gameloop
    while ((ret = game_loop()) == LOOP_CONTINUE)
    {
	// do nothing: stops on LOOP_QUIT or LOOP_NEW_LEVEL
    }

    shutdown_globals();
    
    return ret;
}

int game_loop()
{
    int nframes;

    // set the gametimer to the current clock value
    gt = clock;
    
    // start with drawing a frame
    draw_frame();

    // by now, the clock will have increased. do as many logic loops
    // as necessary to catch up.

    nframes = clock - gt;
    if (nframes > 100)
    {
	warning("too slow: cannot manage 1 fps");
	nframes = 100;
    }
    
    for (int n = 0; n < nframes; n++)
    {
	int ret = logic_loop();

	if (ret != LOOP_CONTINUE)
	{
	    warning("game loop ended: logic loop returned %d", ret);
	    return ret;
	}
    }
    
    return LOOP_CONTINUE;
}

int traverse_bullet_list()
{
    dobj *b = bulls;
    dobj *db;
    
    // DO NOT remove elements in this loop!
    while (b)
    {
	b->go();
	b = b->next;
    }

    // remove requested elements from the list
    b = bulls;
    while (b)
    {
	db = b;
	b = b->next;
	
	if (db->please_delete || db->please_destroy)
	{
	    destroy_bullet((bullet *)db);
	}
    }
    
    return 1;
}

int traverse_guns()
{
    for (int i = 0; i < 2; i++)
    {
	guns[i]->ammo++;
	if (guns[i]->ammo > guns[i]->max_ammo)
	    guns[i]->ammo = guns[i]->max_ammo;

	// compute angle
	double dx, dy;
	dx = guns[i]->y - mouse_y;
	dy = mouse_x - guns[i]->x;

	guns[i]->angle = (int) (atan(dy/dx) * 360.0 / (2 * M_PI));
	if (dx < 0)
	    guns[i]->angle += 180;
    }

    
    
    return 1;
}
    
int check_finish()
{
    if (game_over)
	return 0;

    if (!finished)
    {
	// check the whole grid for targets
	dobj *dtmp;

	targets = 0;
	
	for (int x = 0; x < gr->get_w(); x++)
	{
	    for (int y = 0; y < gr->get_h(); y++)
	    {
		if ((dtmp = gr->get(x, y)))
		{
		    if (dtmp->type() == dobj::TARGET)
			targets++;
		}
	    }
	}

	finished = (!targets);
	if (finished)
	{
	    // add countdown to score
	    score += countdown;
	}
    }
    else
    {
	if (++finished == 200)
	{
	    return 1;
	}
    }

    return 0;
}

int logic_loop()
{
    // show the banner (haha, very first thing :-)
    show_banner();

    // if game not started yet, count down
    if (startcount)
    {
	startcount--;
	return LOOP_CONTINUE;
    }
    
    // handle all objects in active list
    traverse_active_list();

    // handle all bullets
    traverse_bullet_list();

    // handle guns
    traverse_guns();

    // decrease countdown
    if (!game_over)
    {
	countdown--;
	if (countdown < 0)
	    countdown = 0;
	if (!countdown)
	    game_over = 1;
    }
    
    // check for game over
    if (game_over)
    {
	if (++game_over == 500)         // 5 seconds
	    return LOOP_GAME_OVER;
    }

    // check if we're done with this level
    if (check_finish())
    {
	return LOOP_NEW_LEVEL;
    }
    
    // read keyboard / mouse input
    return read_input();
}

int read_input()
{
    if (key[KEY_ESC])
    {
	warning("ESC pressed, Buhbye!");
	return LOOP_QUIT;
    }

    if (key[KEY_PRTSCR])
    {
	while(key[KEY_PRTSCR]);
	
	BITMAP *bmp = create_sub_bitmap(screen, 0, 0, SCREEN_W, SCREEN_H);
	save_bmp("screenshot.pcx", bmp, desktop_palette);
	destroy_bitmap(bmp);
    }

    if (mouse_b)
    {
	// compute coordinates
	int lx, ly;

	// correct for half gridbox offset (x + 1/2, y - 1/2)
	// everything is printed that way to center blocks
	// around their coordinates.
	
//	lx = (mouse_x) * COLS / SCREEN_W -
//	     (COLS - gr->get_w()) / 2;

	lx = (mouse_x - SPACING * (COLS - gr->get_w()) / 2) / SPACING;
	lx++;
	
//	ly = y_offset + LINES -
//	    (mouse_y - SPACING / 2) * LINES / SCREEN_H;

	ly = y_offset + (SCREEN_H - SPACING / 2 - mouse_y) / SPACING;

	// we always seem to aim 2 squares too high, 1 too far to the left.
	// don't feel like finding it, so just fix it here.
	// ly -= 2;
	
	if (mouse_b & 1)
	    guns[0]->fire(lx, ly);
	if (mouse_b & 2)
	    guns[1]->fire(lx, ly);
    }
    
    return LOOP_CONTINUE;
}

void check_active_list()
{   
    for (int x = 0; x < gr->get_w(); x++)
    {
	for (int y = 0; y < gr->get_h(); y++)
	{
	    dobj *dt;
	    if ((dt = gr->get(x, y)))
	    {
		if (dt->awake())
		{
		    dobj *dl = active_list;
		    int found = 0;
		    
		    while (dl)
		    {
			if (dl == dt)
			{
			    found = 1;
			    break;
			}

			dl = dl->next;
		    }

		    ASSERT(found);
		}
	    }
	}
    }
}

void check_prev(dobj *list)
{
    int n = 0;
    dobj *dtmp = list;

    while (dtmp)
    {
	n++;
	ASSERT(dtmp->prev || list == dtmp);
	dtmp = dtmp->next;
    }

//    warning("check_prev: list ok, %d elements", n);
}

int traverse_active_list()
{
//    int i = 0;
    dobj
	*dtmp = active_list;

    // check if there are boulders in the grid that are active and not
    // in the list

//    warning("traverse_active_list");

//    check_prev(active_list);
//    check_prev(add_list);

//    warning("starting go's");
    
    // move boulders
    while (dtmp)
    {
	if (dtmp->go() == dobj::SLEEP)
	{
//	    warning("element %d has gone", i++);
//	    check_prev(active_list);
	    
	    dtmp->please_delete = 1;

	    /* SNIP */
	}

	// move dtmp
	dtmp = dtmp->next;
    }

//    check_prev(active_list);
//    check_prev(add_list);

//    warning("starting deletion of elements");
    
    // now remove all elements from the list that requested it
    dtmp = active_list;

    while (dtmp)
    {
	dobj *d = dtmp;
	dtmp = dtmp->next;

//	check_prev(active_list);
	
	if (d->please_delete || d->please_destroy)
	{
//	    warning("deleting 1 element, active_list = %p", active_list);
//	    warning("1 prev: %p, this: %p, next: %p", d->prev, d, d->next);
	    
	    d->please_delete = 0;
	    d->sleep();

//	    warning("2 prev: %p, this: %p, next: %p", d->prev, d, d->next);
	    
	    // now remove it
	    if (d->prev)
	    {
//		warning("3 prev: %p, this: %p, next: %p", d->prev, d, d->next);

		d->prev->next = d->next;
	    }
	    else
	    {
//		warning("4 prev: %p, this: %p, next: %p", d->prev, d, d->next);

		// we were the first one
		ASSERT(active_list == d);
		active_list = dtmp;
	    }

	    if (d->next)
	    {
//		warning("5 prev: %p, this: %p, next: %p", d->prev, d, d->next);

		d->next->prev = d->prev;
	    }

	    d->next = 0;
	    d->prev = 0;

	    if (d->please_destroy)
	    {
		warning("deleting dobj(%d) at <%d, %d>",
			d->type(), d->x, d->y);
		if (d->plch)
		    kill_dobj(d->plch);
		if (d->plch2)
		    kill_dobj(d->plch2);
		kill_dobj(d);
	    }
	    
//	    warning("6 prev: %p, this: %p, next: %p", d->prev, d, d->next);
	}
    }

//    check_prev(active_list);
//    check_prev(add_list);

//    warning("starting addition of elements");
    
    // and add all elements that got activated
    if (add_list)
    {
	dobj *d = add_list;

	// find last element of add_list
	while (d->next)
	{
	    d = d->next;
	}

	// tie it to active_list
	d->next = active_list;
	if (d->next)
	{
	    d->next->prev = d;
	}

	// set active list to beginning of total list
	active_list = add_list;

	// done adding
	add_list = 0;
    }

//    check_prev(active_list);
//    check_prev(add_list);
//    check_active_list();
    
    return 0;
}

