#include <stdlib.h>
#include "bullet.h"
#include "globals.h"
#include "man.h"
#include "utils.h"

bullet::bullet(int _x, int _y, int _tx, int _ty)
    : dobj(BULLET)
{
    x = _x;        // bullet starts at invalid position
    y = _y;
    tx = _tx;      // whereto
    ty = _ty;

    from_x = x;
    from_y = y;
    to_x = tx;
    to_y = ty;
    
    speed = BULLET_SPEED;
    move_state = MOVING;
    fraction = 0;

    dx = from_x;
    dy = from_y;
    
    for (int p = 0; p < MAX_PARTICLES; p++)
    {
	// initial positions
	// maximum deviation of 2 grid boxes
	particles[p][0] = dx + ((rand() % 101) - 50) / 25.0;
	particles[p][1] = dy + ((rand() % 101) - 50) / 25.0;

	// initial velocities 0
	particles[p][2] = 0.0;
	particles[p][3] = 0.0;

//	warning("particle %d initialised: <%lf, %lf> <%lf, %lf>",
//		p, particles[p][0], particles[p][1],
//		particles[p][2], particles[p][3]);
    }

    pmass = 100.0;
    dt    = 1.0;
}

bullet::~bullet()
{
    // do nothing
}

// bullets are not in the grid, so special functions for
// destroying and adding and updating of the bullets list
int add_bullet(bullet *b)
{
    // add to bulls list

    b->prev = 0;
    b->next = bulls;

    if (b->next)
    {
	b->next->prev = b;
    }

    bulls = b;

    return 1;
}

int destroy_bullet(bullet *b)
{
    // remove from bulls list

    if (b->prev)
    {
	b->prev->next = b->next;
    }
    else
    {
	bulls = b->next;
    }
    
    if (b->next)
    {
	b->next->prev = b->prev;
    }

    b->prev = 0;
    b->next = 0;

    delete b;

    return 1;
}

int bullet::go()
{
    if (move_state == STILL)
    {
	// we arrived, or so. don't move
    }
    else if (move_state == MOVING)
    {
	// move
	fraction++;

	// compute new position
	dx = from_x + (double)((to_x - from_x) * fraction) / (double)speed;
	dy = from_y + (double)((to_y - from_y) * fraction) / (double)speed;

	// update particle positions
	for (int p = 0; p < MAX_PARTICLES; p++)
	{
	    move_particle(p);
	}
	
	if (fraction == speed)
	{
	    return arrive();
	}
    }

    return WAKE;
}

void bullet::move_particle(int p)
{
    double fx, fy, ax, ay, vx, vy, px, py;

    px = particles[p][0];
    py = particles[p][1];
    vx = particles[p][2];
    vy = particles[p][3];

    // compute force, accelleration, speed, x
    fx = dx - px;
    fy = dy - py;
    ax = fx / pmass;
    ay = fy / pmass;
    vx += ax * dt;
    vy += ay * dt;
    px += vx * dt;
    py += vy * dt;

    // store in array
    particles[p][0] = px;
    particles[p][1] = py;
    particles[p][2] = vx;
    particles[p][3] = vy;
}

int bullet::explode()
{
    // kill target, and possibly around it.
    int margin = 0;
    
    for (int lx = tx - margin; lx <= tx + margin; lx++)
    {
	for (int ly = ty - margin; ly <= ty + margin; ly++)
	{
	    // check coordinates
	    if (lx < 0 || lx >= gr->get_w() || ly < 0 || ly >= gr->get_h())
	    {
		continue;
	    }

	    // check type of object
	    dobj *dt = gr->get(lx, ly);

	    if (!dt)
		continue;
	    
	    // only explode boulders, humans, targets and sand
	    if (dt->type() == MAN)
	    {
		kill_man((man *)dt);
	    }
	    else if (dt->type() == BOULDER)
	    {
//		warning("shot boulder at <%d,%d>", dt->x, dt->y);

		if (dt->awake())
		    dt->please_destroy = 1;
		else
		    kill_dobj(dt);
	    }
	    else if (dt->type() == SAND || dt->type() == TARGET)
	    {
		// stationary inactive objects (target, sand)
		kill_dobj(dt);
	    }

	    // don't kill other objects
	}
    }

    // kill us as soon as possible
    please_destroy = 1;
    
    return 0;
}

int bullet::arrive()
{
    move_state = STILL;

    explode();

    // now kill us on return (please destroy is now set)
    
    return SLEEP;
}
