#include <math.h>
#include <allegro.h>
#include <stdio.h>
#include "list.h"
#include "block.h"
#include "grid.h"
#include "dirty.h"
#include "sound.h"
#include "timers.h"
#include "globals.h"
#include "init.h"

int min(int x1, int x2, int x3)
{
    int min = x1;

    if (x2 < min)
	min = x2;
    if (x3 < min)
	min = x3;

    return min;
}

int min(int x1, int x2)
{
    return ((x1 < x2) ? x1 : x2);
}

int max(int x1, int x2, int x3)
{
    int max = x1;

    if (x2 > max)
	max = x2;
    if (x3 > max)
	max = x3;

    return max;
}

int max(int x1, int x2)
{
    return ((x1 > x2) ? x1 : x2);
}

void dirty_paddle_round(Paddle *p)
{
    int x1, y1, x2, y2, x3, y3;
    int minx, maxx, miny, maxy;
    int margin;

    double da1 = p->pos - p->width / 2.0;
    double da2 = p->pos + p->width / 2.0;

    x1 = (int)(xo + p->radius * cos(da1));
    y1 = (int)(yo + p->radius * sin(da1));
    x2 = (int)(xo + p->radius * cos(da2));
    y2 = (int)(yo + p->radius * sin(da2));
    x3 = (int)(xo + p->radius * cos(p->pos));
    y3 = (int)(yo + p->radius * sin(p->pos));

    // increase margin for curved pads
    // handwaving formula, won't work for large width
    // upto width 3.0 works ok (half a circle pad)
    margin = (int)(p->width * 10);

    minx = min(x1, x2, x3) - margin;
    maxx = max(x1, x2, x3) + margin;
    miny = min(y1, y2, y3) - margin;
    maxy = max(y1, y2, y3) + margin;

    //    rect(back, minx, miny, maxx, maxy, red);
    add_dirty(minx, maxx, miny, maxy);
}

void draw_paddle_round(Paddle *p)
{
    // take angles to be negative:
    // y axis is inverted
    double da1 = - (p->pos - p->width / 2.0);
    double da2 = - (p->pos + p->width / 2.0);

    while (da1 < 0.0)
	da1 += 2 * M_PI;
    while (da2 < 0.0)
	da2 += 2 * M_PI;

    fixed ang1;
    fixed ang2;

    ang1 = ftofix((da1 * 256.0)/(2.0 * M_PI));
    ang2 = ftofix((da2 * 256.0)/(2.0 * M_PI));

     // since we swapped the angles, we have to start with
     // the largest angle now: allegro draws arcs counter-clockwise
    arc(back, xo, yo, ang2, ang1, p->radius, yellow);
}

// there is no longer a dirty_paddle:
// draw_paddle does that itsself to save some time
void draw_paddle(Paddle *p)
{
    double pad_a = p->pos;
    double sinpad_a = sin(pad_a);
    double cospad_a = cos(pad_a);
    double half_w = p->width / 2.0;

    int margin = 1;
    
    int p1x = xo + (int)(p->radius * cospad_a + half_w * sinpad_a);
    int p1y = yo + (int)(p->radius * sinpad_a - half_w * cospad_a);
    int p2x = xo + (int)(p->radius * cospad_a - half_w * sinpad_a);
    int p2y = yo + (int)(p->radius * sinpad_a + half_w * cospad_a);

    line(back, p1x, p1y, p2x, p2y, p->color);

    // dirty_paddle functionality here, for efficiency
    add_dirty(min(p1x, p2x) - margin, max(p1x, p2x) + margin,
	      min(p1y, p2y) - margin, max(p1y, p2y) + margin);

//    rect(back, min(p1x, p2x) - margin, min(p1y, p2y) - margin,
//	       max(p1x, p2x) + margin, max(p1y, p2y) + margin, red);
    
//    if (abs(p1x - p2x) < 2)
//    {
//	printf("adding dirty vline (%d %d %d %d)\n",
//	       min(p1x, p2x) - margin, max(p1x, p2x) + margin,
//	       min(p1y, p2y) - margin, max(p1y, p2y) + margin);
//    }
}

void dirty_block(Block *b)
{
    // left, right, bottom, top
    add_dirty(b->l + xo, b->r + xo,
	      b->b + yo, b->t + yo);
}

void draw_block(Block *b)
{
    BITMAP *block;

    // first 5 blocks are for positive counters,
    // block nr 6 is for permanent blocks
    if (b->counter < 0)
    {
	block = blocks[MAX_BLOCKS - 1];
    }
    else
    {
	block = blocks[b->counter % (MAX_BLOCKS - 1)];
    }
    
    //  rect(back, b->l + xo, b->t + yo, b->r + xo, b->b + yo, blue);
    blit(block, back, 0, 0, b->l + xo, b->b + yo, block->w, block->h);
}

void dirty_ball(Ball *b)
{
//    int half_size = ball_size / 2 + 2;
    
    // left, right, bottom, top
//    add_dirty((int)b->x + xo - half_size , (int)b->x + xo + half_size,
//	      (int)b->y + yo - half_size, (int)b->y + yo + half_size);

    add_dirty((int)(b->x + xo - b->rad),
	      (int)(b->x + xo - b->rad) + ball->w - 1,
	      (int)(b->y + yo - b->rad),
	      (int)(b->y + yo - b->rad) + ball->h - 1);
}

void draw_ball(Ball *b)
{
    //    circle(back, (int)b->x + xo, (int)b->y + yo, (int)b->rad, yellow);
    masked_blit(ball, back, 0, 0,
		(int)(b->x + xo - b->rad),
		(int)(b->y + yo - b->rad),
		ball->w, ball->h);
}

void next_level()
{
    char buf[80];


//    printf("level was %d\n", level);
    
    level++;
    score += 10 * level;

//    printf("level is %d\n", level);
//    printf("MAX_LEVEL is %d\n", MAX_LEVEL);
    
    int height = SCREEN_H / 2;
    
    sprintf(buf, "Level %d completed, press a key to continue", level);
    textout_centre(screen, font, buf,
		   SCREEN_W / 2, SCREEN_H / 2, yellow);
    height += 2 * text_height(font);
    
    // if we finished all levels, increase difficulty
    if (!(level % MAX_LEVEL))
    {
	difficulty += 2;
	sprintf(buf, "All levels completed, increasing difficulty to %d.",
		difficulty);
	textout_centre(screen, font, buf,
		       SCREEN_W / 2, height, blue);
	height += 2 * text_height(font);
    }
    
    sprintf(buf, "Score: %d", score);
    textout_centre(screen, font, buf,
		   SCREEN_W / 2, height, yellow);

    while (keypressed())
	readkey();

    rest(3000);
    
    readkey();
    
    shutdown_globals();
    init_globals();

    reset_graphics();
    
    // restart game
    gameloops = 0;
    gametime = 0;
    first_time = 1;
}

void die()
{
    // play a sound or something

    char buf[80];
    
    sprintf(buf, "Press a key to continue");
    textout_centre(screen, font, buf,
		   SCREEN_W / 2, SCREEN_H / 2, yellow);

    sprintf(buf, "Level: %d  --  Score: %d", level + 1, score);
    textout_centre(screen, font, buf,
		   SCREEN_W / 2, SCREEN_H / 2 + 2 * text_height(font), yellow);

    while (keypressed())
	readkey();

    rest(3000);
    
    readkey();

    // only reset the moving globals. we want to keep the map
    shutdown_moving_globals();
    init_moving_globals();

    reset_graphics();
    
    // restart game
    gameloops = 0;
    gametime = 0;
    first_time = 1;
}

void game_over()
{
    char buf[80];
    
    sprintf(buf, "GAME OVER");
    textout_centre(screen, font, buf,
		   SCREEN_W / 2, SCREEN_H / 2, yellow);

    sprintf(buf, "Level: %d  --  Score: %d", level + 1, score);
    textout_centre(screen, font, buf,
		   SCREEN_W / 2, SCREEN_H / 2 + 2 * text_height(font), yellow);

    while (keypressed())
	readkey();

    rest(3000);
    
    readkey();
    
    level = startlevel;
    difficulty = start_difficulty;
    score = 0;
    lives = MAX_LIVES;
    
    shutdown_globals();
    init_globals();

    reset_graphics();
    
    // restart game
    gameloops = 0;
    gametime = 0;
    first_time = 1;
}

void check_game_over()
{
    lives--;

    if (lives < 0)
	game_over();
    else
	die();
}

int out_of_range(Ball *b)
{
    int half_w = SCREEN_W / 2;
    int half_h = SCREEN_H / 2;
    int margin = (int)(b->rad * 4);      // make sure they are really gone
    
    return (b->x < -half_w - margin || b->x > half_w + margin
	    ||
	    b->y < -half_h - margin || b->y > half_h + margin);
}

int kill_balls()
{
    Ball *b;

    balls.reset();
    while ((b = (Ball *)balls.get()))
    {
	if (out_of_range(b))
	{
	    balls.pop(b);     // sets current pointer to b->next
	    delete b;
	    continue;
	}

	balls.next();
    }

    return (balls.size());
}

List tmp_balls;

void move_ball(Ball *ba)
{
    Block *bl;
    Ball *other;
    
    // move the ball
    ba->move();

    // collide with all blocks
    for (int x = 0; x < 2 * center; x++)
    {
	for (int y = 0; y < 2 * center; y++)
	{
	    if ((bl = gr.get(x, y)))
	    {
		if (ba->collide(bl))
		{
		    int ret;
		    
		    // play sound
		    play_click();

		    // hit the block
		    ret = bl->hit();

		    // blocks either disappear or change color when hit
		    // except permanent blocks, but we ignore that
		    dirty_block(bl);

		    if (ret >= 0)
		    {
			// add score only for non-permanent blocks
			score++;
		    }

		    if (!ret)
		    {
			// remove the block
			gr.set(0, x, y);
			delete bl;
			bl = 0;
		    }

		    // check if the level is empty
		    if (!gr.count())
		    {
			next_level();
		    }
		}
	    }
	}
    }

    // collide with other balls
    // use a scratch list, because this routine is also called
    // from a loop over all balls in the balls list
    balls.clearcopy(&tmp_balls);
    for (tmp_balls.reset();
	 (other = (Ball *)tmp_balls.get());
	 tmp_balls.next())
    {
	if (ba == other)
	    continue;
	
	if (ba->collide(other))
	{
	    // play a fancy sound
	    play_click_high();
	}
    }
    
    // collide with the paddle
    for (int p = 0; p < players; p++)
    {
	if (ba->collide(pad[p]))
	{
	    play_click_low();
	}
    }
}

void clear_back()
{
    dr *tmp;

//    clear_to_color(back, black);
    
    // clear only parts of the backbuffer that we painted on last frame
    // these rectangles are in the clear list
    while ((tmp = pop_dirty_back()))
    {
	rectfill(back,
		 tmp->left, tmp->bottom,
		 tmp->right, tmp->top,
		 black);

	free_dirty_back(tmp);
    }
}

void blit_dirty()
{
    dr *tmp;

    //    printf(" -- start --\n");

    // clear old parts
    while ((tmp = pop_dirty_clear()))
    {
	/*
	  printf("clean %d %d %d %d\n",
	  tmp->left, tmp->right,
	  tmp->bottom, tmp->top);
	*/

	blit(back, screen,
	     tmp->left, tmp->bottom,
	     tmp->left, tmp->bottom,
	     tmp->right - tmp->left + 1, tmp->top - tmp->bottom + 1);

	free_dirty_clear(tmp);
    }

    // paint new parts
    while ((tmp = pop_dirty_paint()))
    {
	/*
	  printf("paint %d %d %d %d\n",
	  tmp->left, tmp->right,
	  tmp->bottom, tmp->top);
	*/

	blit(back, screen,
	     tmp->left, tmp->bottom,
	     tmp->left, tmp->bottom,
	     tmp->right - tmp->left + 1, tmp->top - tmp->bottom + 1);

	// this will add the rectangle to the clear list
	free_dirty_paint(tmp);
    }

    //    printf(" -- end --\n");
}

void compute_fps()
{
    // compute fps once every 50 frames
    // every frame gives a jumpy readout with
    // almost integer results (1 frame every 3 clock ticks or so)

//    static int frames = 0;
//    frames += gametime - lastframe;
//    lastframe = gametime;
    
    if (++framecount == 100)
    {
	int frames = gametime - lastframe;
	fps = 10000.0 / (double)frames;
	lastframe = gametime;
	framecount = 0;
//	frames = 0;
    }
}

void print_fps()
{
    char buf[20];      // fps: 666.666666

    // print fps
    sprintf(buf, "fps: %d", (int)fps);
    textout(back, font, buf, 10, 10, red);
    add_dirty(10, 10 + text_length(font, buf), 10, 10 + text_height(font));
}

// only print the score bitmap if something has changed
void print_score()
{
    static int last_score = -1;
    static int last_lives = -1;
    
    char buf[30];      // score: 12345678901

    // if nothing has changed, don't draw, just blit the bitmap
    if (last_score != score || last_lives != lives)
    {
	// set last variables
	last_score = score;
	last_lives = lives;
	
	// clear the bitmap
	clear_to_color(score_bmp, black);
	
	// print score
	sprintf(buf, "score: %d", score);
	textout(score_bmp, font, buf,
		0, (score_bmp->h - text_height(font)) / 2, red);

	// draw a ball for every life
	for (int i = 0; i < lives; i++)
	{
	    masked_blit(ball, score_bmp,
			0, 0,
			score_bmp->w - ((i + 2) * (ball->w + 5)),
			(score_bmp->h - ball->h) / 2,
			ball->w, ball->h);
	}
	
//      rect(score_bmp, 0, 0, score_bmp->w - 1, score_bmp->h - 1, red);
    }
	
    // blit bitmap to backbuffer and set dirty rectangle
    blit(score_bmp, back, 0, 0, back->w - score_bmp->w, 0,
	 score_bmp->w, score_bmp->h);
    add_dirty(back->w - score_bmp->w, back->w - 1, 0, score_bmp->h - 1); 
}

void draw_frame()
{
    Block *bl;
    Ball *ba;

    // draw a frame
    clear_back();

    compute_fps();
    print_fps();

    print_score();

    // draw_paddle also does dirty_paddle for efficiency

    for (int p = 0; p < players; p++)
	draw_paddle(pad[p]);
    
    for (int x = 0; x < 2 * center; x++)
    {
	for (int y = 0; y < 2 * center; y++)
	{
	    if ((bl = gr.get(x, y)))
	    {
		draw_block(bl);
		if (first_time)
		    dirty_block(bl);
	    }
	}
    }
	
    for (balls.reset(); (ba = (Ball *)balls.get()); balls.next())
    {
	draw_ball(ba);
	dirty_ball(ba);
    }

    blit_dirty();
//    blit(back, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);

    first_time = 0;
}

int read_input()
{
    // allegro angle is mathematical (increase is counter-clockwise)
    // because y-axis is inverted though, my coordinate system works
    // the other way around. therefore key left (for clockwise movement)
    // needs to increase the angle.

    if (key[KEY_F1])
    {
	// show the backbuffer and pause
	blit(back, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);

	while (keypressed())
	{
	    readkey();
	}

	readkey();
    }
    
    if (key[KEY_LEFT])
    {
	pad[0]->pos += richard * delta_pad_pos;
    }
    if (key[KEY_RIGHT])
    {
	pad[0]->pos -= richard * delta_pad_pos;
    }

    while(pad[0]->pos > 2 * M_PI)
    {
	pad[0]->pos -= 2 * M_PI;
    }
    while(pad[0]->pos < 0.0)
    {
	pad[0]->pos += 2 * M_PI;
    }
    
    if (pad[1])
    {
	if (key[KEY_Z])
	{
	    pad[1]->pos += richard * delta_pad_pos;
	}
	if (key[KEY_X])
	{
	    pad[1]->pos -= richard * delta_pad_pos;
	}
	
	while(pad[1]->pos > 2 * M_PI)
	{
	    pad[1]->pos -= 2 * M_PI;
	}
	while(pad[1]->pos < 0.0)
	{
	    pad[1]->pos += 2 * M_PI;
	}
    }
    
    if (key[KEY_ESC] || key[KEY_Q])
	return 0;
    else
	return 1;
}

int gameloop()
{
    Ball *ba;
    
    // draw a frame
    // limit to 50 / sec: minimum of 2 timer ticks (0.01 sec) per frame
//    if (gametime - lastframe >= 2)
    draw_frame();

    // run gamelogic as much as needed
    while (gameloops < gametime)
    {
	gameloops++;

	// move balls
	for (balls.reset(); (ba = (Ball *)balls.get()); balls.next())
	{
	    move_ball(ba);
	}

	// kill off-screen balls
	// if no balls are left, game over
	if (!kill_balls())
	    check_game_over();
	
	if (read_input() == 0)
	    return 0;
    }
	
    return 1;
}
