#include "ball.h"
#include <math.h>
#include <stdio.h>
#include "grid.h"

extern Grid gr;

Ball::Ball()
{
    x = y = rad = vx = vy = omega = 0.0;
}

Ball::Ball(double _x, double _y, double _rad,
	   double _vx = 0.0, double _vy = 0.0, double _omega = 0.0)
{
    x = _x;
    y = _y;
    rad = _rad;
    vx = _vx;
    vy = _vy;
    omega = _omega;
}

Ball::Ball(Ball const &other)
{
    x = other.x;
    y = other.y;
    rad = other.rad;
    vx = other.vx;
    vy = other.vy;
    omega = other.omega;
}

Ball const &Ball::operator=(Ball const &other)
{
    if (this != &other)
    {
	x = other.x;
	y = other.y;
	rad = other.rad;
	vx = other.vx;
	vy = other.vy;
	omega = other.omega;
    }

    return (*this);
}

int Ball::intersect(int _x, int _y)
{
    return ((x - _x)*(x - _x) + (y - _y)*(y - _y) < rad*rad);
}

int Ball::intersect(double _x, double _y)
{
    return ((x - _x)*(x - _x) + (y - _y)*(y - _y) < rad*rad);
}

int Ball::intersect(Block *b)
{
    // do we intersect cleanly with the block (top and bottom)
    if (x >= b->l && x <= b->r && y >= b->b - rad && y <= b->t + rad)
	return 1;

    // do we intersect cleanly with the block (left and right)
    if (y >= b->b && y <= b->t && x >= b->l - rad && x <= b->r + rad)
	return 1;

    // do we intersect with one of the corners?
    return (intersect(b->l, b->b)
	    ||
	    intersect(b->l, b->t)
	    ||
	    intersect(b->r, b->t)
	    ||
	    intersect(b->r, b->b));
}

int Ball::intersect(Ball *b)
{
    double square_d = (x - b->x) * (x - b->x) + (y - b->y) * (y - b->y);

    return (square_d < (rad + b->rad) * (rad + b->rad));
}

// 0: no intersection
// 1: intersect paddle center
// 2: intersect corner (pos + width / 2)
// 3: intersect corner (pos - width / 2)
int Ball::intersect_round(Paddle *p)
{
    // check if the ball intersects with the paddle

    double pad_r = p->radius;
    double pos_r = sqrt(x * x + y * y);

    // do we intersect the paddle? (in the radial direction)
    if (fabs(pad_r - pos_r) < rad)
    {
        // do we intersect the paddle ? (in the lateral direction)
	double pad_a = p->pos;
	double pos_a = atan(y / x);

//	printf("x: %f    y: %f    atan: %f\n", x, y, pos_a);
	
	if (x < 0.0 && y > 0.0)
	    pos_a = M_PI + pos_a;        // angle is negative so pi + a !!
	if (x < 0.0 && y < 0.0)
	    pos_a += M_PI;               // angle is already positive
	if (x > 0.0 && y < 0.0)
	    pos_a = 2 * M_PI + pos_a;    // angle is negative so 2pi + a !!

//	printf("pos_a : %f     pad_a : %f\n", pos_a, pad_a);

	// check delta_a, but keep in mind they may be across the 2PI border
	if (fabs(pad_a - pos_a) < p->width / 2.0)
	{
//	    printf("normal case\n");
	    return 1;
	}
	if (fabs(pad_a + 2.0 * M_PI - pos_a) < p->width / 2.0)
	{
//	    printf("+2 * PI\n");
	    return 1;
	}

	if (fabs(pad_a - 2.0 * M_PI - pos_a) < p->width / 2.0)
	{
//	    printf("-2 * PI\n");
	    return 1;
	}

	// if it's outside but only half, it's possible we still hit the
	// endpoints

	double pe1x = p->radius * cos(p->pos + p->width / 2);
	double pe1y = p->radius * sin(p->pos + p->width / 2);
	double pe2x = p->radius * cos(p->pos - p->width / 2);
	double pe2y = p->radius * sin(p->pos - p->width / 2);

	if (intersect(pe1x, pe1y))
	{
//	    printf("intersected point 1\n");
	    return 2;
	}
	if (intersect(pe2x, pe2y))
	{
//	    printf("intersected point 2\n");
	    return 3;
	}
    }

    return 0;
}

// intersect function for a straight paddle
// compute lateral and radial distance from paddle center
// by using the angle between pos_r and pad_r

// to prevent unnecessary gonio computations,
// put intersect and collision routines together

int Ball::collide(Paddle *p)
{
    // check if the ball intersects with the paddle

    double pad_r = p->radius;
    double pos_r = sqrt(x * x + y * y);

    double rad_dist = 0.0;    // ball center <-> paddle (shortest dist)
    double lat_dist = 0.0;    // ball center <-> paddle radius (shortest)
    
    double alpha;             // angle between pos_r and pad_r
    
    double pad_a = p->pos;
    double pos_a = atan(y / x);

    double half_w = p->width / 2.0;
    
    // put pos_a in proper 0 - 2PI range (atan is -0.5PI - 0.5PI)
    if (x < 0.0 && y > 0.0)
	pos_a = M_PI + pos_a;        // angle is negative so pi + a !!
    if (x < 0.0 && y < 0.0)
	pos_a += M_PI;               // angle is already positive
    if (x > 0.0 && y < 0.0)
	pos_a = 2 * M_PI + pos_a;    // angle is negative so 2pi + a !!

    alpha = fabs(pos_a - pad_a);     // make sure alpha is positive
    if (alpha > M_PI)
	alpha = 2 * M_PI - alpha;    // always use the smallest angle

    double sina = sin(alpha);
    double cosa = cos(alpha);
    
    lat_dist = pos_r * sina;
    rad_dist = pad_r - (pos_r * cosa);

    // if either distance is too large, we don't intersect
    if (fabs(rad_dist) > rad)
    {
	// radial distance too large: we miss the pad
	return 0;
    }
    else
    {
	double sinpad_a = sin(pad_a);
	double cospad_a = cos(pad_a);

	// four cases for lateral distance: center, corner (l/r), outside
	if (lat_dist < half_w)
	{
	    // clean intersection
	    // determine collision point

	    double int_x, int_y;

	    int_x = x + rad_dist * cospad_a;
	    int_y = y + rad_dist * sinpad_a;

	    return collide(int_x, int_y);
	}
	else
	{
	    // check corner intersections

	    double p1x = p->radius * cospad_a + half_w * sinpad_a;
	    double p1y = p->radius * sinpad_a - half_w * cospad_a;
	    double p2x = p->radius * cospad_a - half_w * sinpad_a;
	    double p2y = p->radius * sinpad_a + half_w * cospad_a;

	    if (intersect(p1x, p1y))
		return collide(p1x, p1y);
	    if (intersect(p2x, p2y))
		return collide(p2x, p2y);
	}
    }

    return 0;
}
	    
    

int Ball::collide(double _x, double _y)
{
    // collide with a point
    double dx = _x - x;
    double dy = _y - y;
    double d = sqrt(dx * dx + dy * dy);
    double sina = dy / d;
    double cosa = dx / d;

    // to figure this out, take both vx and vy apart into a parallel and
    // a cross component. draw it on paper :>
    double vcross = - vx * sina + vy * cosa;
    double vparr  = vx * cosa + vy * sina;

    // check if we're moving toward or away from the point
    // if we're moving away we already collided last frame
    double dparr = dx * cosa + dy * sina;

    if (dparr < 0 && vparr > 0 || dparr > 0 && vparr < 0)
	return 0;

    // flip the parallel speed
    vparr = -vparr;

    // convert back to carthesian components
    vx = vparr * cosa - vcross * sina;
    vy = vparr * sina + vcross * cosa;
    
    return 1;
}

int Ball::collide(Block *b)
{
    Block
	*left = gr.get(b->x - 1, b->y),
	*right = gr.get(b->x + 1, b->y),
	*top = gr.get(b->x, b->y + 1),
	*bot = gr.get(b->x, b->y - 1),
	*lefttop = gr.get(b->x - 1, b->y + 1),
	*leftbot = gr.get(b->x - 1, b->y - 1),
	*righttop = gr.get(b->x + 1, b->y + 1),
	*rightbot = gr.get(b->x + 1, b->y - 1);
    
    // check if we are intersecting the block
    if (!intersect(b))
	return 0;

    // do line collisions first!
    // otherwise we get strange collisions: no corners and
    // invisible sides when blocks are placed next to eachother
    
    // collide depending on where we are
    if (x >= b->l && x <= b->r)
    {
	if (y > b->t)
	{
	    // if there's another block there, don't collide
	    if (top)
		return 0;
	    else
		return collide_top();
	}
	if (y < b->b)
	{
	    // if there's another block there, don't collide
	    if (bot)
		return 0;
	    else
		return collide_bottom();
	}
	else
	    return collide_center(b);
    }

    if (y >= b->b && y <= b->t)
    {
	if (x > b->r)
	{
	    // if there's another block there, don't collide
	    if (right)
		return 0;
	    else
		return collide_right();
	}
	if (x < b->l)
	{
	    // if there's another block there, don't collide
	    if (left)
		return 0;
	    else
		return collide_left();
	}
	else
	    return collide_center(b);
    }

    // collide with a corner
    if (x < b->l)
    {
	if (y < b->b)
	{
	    // left bottom corner
	    if (left || leftbot || bot)
		return 0;
	    else
		return collide(b->l, b->b);
	}
	else
	{
	    // left top corner
	    if (left || lefttop || top)
		return 0;
	    else
		return collide(b->l, b->t);
	}
    }
    if (x > b->r)
    {
	if (y < b->b)
	{
	    // bottom right corner
	    if (right || rightbot || bot)
		return 0;
	    else
		return collide(b->r, b->b);
	}
	else
	{
	    // right top corner
	    if (right || righttop || top)
		return 0;
	    else
		return collide(b->r, b->t);
	}
    }

    return 0;
}

int Ball::collide(Ball *b)
{
    if (!intersect(b))
	return 0;

    // collide with the ball
    double dx = b->x - x;
    double dy = b->y - y;
    double d2 = dx * dx + dy * dy;      // squared distance between balls
    double d  = sqrt(d2);               // real distance

    double m1 = rad * rad * rad;
    double m2 = b->rad * b->rad * b->rad;
    double M  = m1 + m2;
    
    // compute speed of center of mass
    double vcmx = (vx * m1 + b->vx * m2) / M;
    double vcmy = (vy * m1 + b->vy * m2) / M;

    // compute speeds relative to center of mass
    double ux1 = vx - vcmx;
    double uy1 = vy - vcmy;
    double ux2 = b->vx - vcmx;
    double uy2 = b->vy - vcmy;

    // compute cosine and sine alpha (angle from this ball to other ball)
    double cosa = dx / d;
    double sina = dy / d;

    // compute parallel (l) and cross (d) speeds
    double ul1 = ux1 * cosa + uy1 * sina;
    double ud1 = uy1 * cosa - ux1 * sina;

    double ul2 = ux2 * cosa + uy2 * sina;
    double ud2 = uy2 * cosa - ux2 * sina;

    // check that the balls are moving apart
    // compute parallel distance seen from ball 1
    // compare to parallel speed of ball 1
    // center of mass system, so if one ball is moving away,
    // so is the other.
    // this works because dx = positive when moving from ball 1
    // to ball 2, and because the positive parallel direction was
    // also chosen that way
    double dl1 = dx * cosa + dy * sina;

    if (dl1 > 0 && ul1 < 0 || dl1 < 0 && ul1 > 0)
	return 0;

    // ok, we're moving toward eachother. now collide.
    // flip the parallel speeds
    ul1 = -ul1;
    ul2 = -ul2;

    // convert back to carthesian coordinates
    ux1 = ul1 * cosa - ud1 * sina;
    uy1 = ul1 * sina + ud1 * cosa;

    ux2 = ul2 * cosa - ud2 * sina;
    uy2 = ul2 * sina + ud2 * cosa;

    // convert back from center of mass system
    vx = ux1 + vcmx;
    vy = uy1 + vcmy;

    b->vx = ux2 + vcmx;
    b->vy = uy2 + vcmy;
    
    return 1;
}

int Ball::collide_round(Paddle *p)
{
    double cx;
    double cy;
    
    int ret = intersect_round(p);

    if (!ret)
	return 0;

    if (ret == 1)
    {
	// 'normal' intersection: collide as if colliding with a solid point
	// determine collision point by extending ball_pos_radius to the paddle
	double factor =  p->radius / sqrt(x * x + y * y);
	cx = x * factor;
	cy = y * factor;

//	printf("colliding with <%f, %f> , ball at <%f, %f>\n",
//	       cx, cy, x, y);
	
	return collide((int)cx, (int)cy);
    }
    if (ret == 2)
    {
	cx = p->radius * cos(p->pos + p->width / 2);
	cy = p->radius * sin(p->pos + p->width / 2);

	return collide((int)cx, (int)cy);
    }
    if (ret == 3)
    {
	cx = p->radius * cos(p->pos - p->width / 2);
	cy = p->radius * sin(p->pos - p->width / 2);

	return collide((int)cx, (int)cy);
    }
    
    return 0;
}

int Ball::collide_left()
{
    // check if we are moving towards the right
    // we're bouncing against the left side of the block

    if (vx < 0)
	return 0;

    vx = -vx;
    return 1;
}

int Ball::collide_right()
{
    if (vx > 0)
	return 0;

    vx = -vx;
    return 1;
}

int Ball::collide_bottom()
{
    // check that we are moving towards the top
    // we're bouncing against the bottom side of the block

    if (vy < 0)
	return 0;

    vy = -vy;
    return 1;
}

int Ball::collide_top()
{
    if (vy > 0)
	return 0;

    vy = -vy;
    return 1;
}

void Ball::move()
{
    x += vx;
    y += vy;
}

int Ball::collide_center(Block *bl)
{
    double relx;
    double rely;

    relx = (x - bl->l) / (bl->r - bl->l);
    rely = (y - bl->b) / (bl->t - bl->b);

    if (relx < rely)
    {
	// left top triangle

	if (relx + rely < 1.0)
	{
	    // left middle triangle

	    collide_left();
	}
	else
	{
	    // top middle triangle

	    collide_top();
	}
    }
    else
    {
	// right bottom triangle

	if (relx + rely < 1.0)
	{
	    // bottom middle triangle

	    collide_bottom();
	}
	else
	{
	    // right middle triangle

	    collide_right();
	}
    }
       
    return 0;
}
