/* All functions related to Crystal Stacker themes. */

#include "includes.h"
#include "externs.h"

ANIMATION * bitmap2ani(BITMAP * bp)
{
	ANIMATION * ap;

	ap = create_ani(bp->w, bp->h);
	animation_add_frame(ap, bp);

	return ap;
}

/* sets up video and memory areas for use by the game */
int cs_initialize_theme(CS_THEME * tp)
{
    /* set up screen */
    cs_gfx_mode(tp->rx, tp->ry, black_palette);

    /* returns 1 for success */
    return 1;
}

/* saves theme to specified file */
int cs_save_theme(CS_THEME * tp, char * fn)
{
    PACKFILE * file;
    int i, j;

    /* open the file */
    file = pack_fopen(fn, "w");
    if(file == NULL)
    {
        return 0;
    }

    /* write the header */
    tp->header[0] = 'C';
    tp->header[1] = 'T';
    tp->header[2] = 'H';
    tp->header[3] = 22;
    pack_fwrite(tp->header, 4, file);

    /* save video card info */
    pack_putc(tp->card, file);
    pack_iputw(tp->rx, file);
    pack_iputw(tp->ry, file);

    /* save the palette */
    for(i = 0; i < 256; i++)
    {
        pack_putc(tp->palette[i].r, file);
        pack_putc(tp->palette[i].g, file);
        pack_putc(tp->palette[i].b, file);
    }

    /* save the board info */
    pack_putc(tp->board_w, file);
    pack_putc(tp->board_h, file);
    for(i = 0; i < MAX_PLAYERS; i++)
    {
        pack_iputw(tp->board_x[i], file);
        pack_iputw(tp->board_y[i], file);
        pack_iputw(tp->score_x[i], file);
        pack_iputw(tp->score_y[i], file);
        pack_iputw(tp->preview_x[i], file);
        pack_iputw(tp->preview_y[i], file);
        pack_iputw(tp->level_x[i], file);
        pack_iputw(tp->level_y[i], file);
        pack_iputw(tp->block_x[i], file);
        pack_iputw(tp->block_y[i], file);
        pack_iputw(tp->player_message_x[i], file);
        pack_iputw(tp->player_message_y[i], file);
    }

    pack_iputw(tp->time_x, file);
    pack_iputw(tp->time_y, file);
    pack_iputw(tp->high_score_x, file);
    pack_iputw(tp->high_score_y, file);
    pack_iputw(tp->best_time_x, file);
    pack_iputw(tp->best_time_y, file);
    pack_iputw(tp->message_x, file);
    pack_iputw(tp->message_y, file);

    /* save the block info and data */
    for(i = 0; i < NUM_BLOCK_TYPES; i++)
    {
	    save_ani_fp(tp->block_image[i], file, NULL);
	    save_ani_fp(tp->block_flash_image[i], file, NULL);
    }

    /* save the tile info and data */
    if(tp->tile_w > 0 && tp->tile_h > 0)
    {
        for(i = 0; i < NUM_BG_TILES; i++)
        {
	        save_ani_fp(tp->tile_image[i], file, NULL);
        }
    }
    else
    {
        for(i = 0; i < tp->ry; i++)
        {
            for(j = 0; j < tp->rx; j++)
            {
                pack_putc(tp->bg_image->line[i][j], file);
            }
        }
    }

    /* save the font info and data */
    save_ncdfont_fp(&tp->font, file);

    /* save the map info and data */
    pack_putc(tp->map_w, file);
    pack_putc(tp->map_h, file);
    if(tp->map_w > 0 && tp->map_h > 0)
    {
        for(i = 0; i < tp->map_h; i++)
        {
            for(j = 0; j < tp->map_w; j++)
            {
                pack_putc(tp->map[i * tp->map_w + j], file);
            }
        }
    }

    if(tp->sound[SOUND_DROP] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_DROP], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_ROTATEUP] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_ROTATEUP], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_ROTATEDOWN] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_ROTATEDOWN], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_MOVE] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_MOVE], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_HITWALL] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_HITWALL], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_LAND] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_LAND], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_RUN] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_RUN], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_DELETE] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_DELETE], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_LEVELUP] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_LEVELUP], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_PAUSE] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_PAUSE], file);
    }
    else
    {
        pack_putc(0, file);
    }

    if(tp->sound[SOUND_ATTACK] != NULL)
    {
        pack_putc(1, file);
        t3ss_save_wav_fp(tp->sound[SOUND_ATTACK], file);
    }
    else
    {
        pack_putc(0, file);
    }

    /* close the file */
    pack_fclose(file);

    return 1;
}

/* loads theme from specified file and offset */
int cs_load_theme(CS_THEME * tp, char * fn)
{
    PACKFILE * file;
    int i, j, k;
    char TempBit;
    BITMAP * block_image;
    BITMAP * block_flash_image;
    BITMAP * tile_image;

    file = pack_fopen(fn, "r");
    if(file == NULL)
    {
        return 0;
    }

    /* read the header and see if it is a theme file */
    pack_fread(tp->header, 4, file);
    if(tp->header[0] != 'C' || tp->header[1] != 'T' || tp->header[2] != 'H' || (tp->header[3] != 21 && tp->header[3] != 22))
    {
        pack_fclose(file);
        return 0;
    }

    /* video card stuff */
    tp->card = pack_getc(file);
    tp->rx = pack_igetw(file);
    tp->ry = pack_igetw(file);
    tp->screen = create_bitmap(tp->rx, tp->ry);
    clear(tp->screen);
    tp->bg_image = create_bitmap(tp->rx, tp->ry);

    /* load the palette */
    for(i = 0; i < 256; i++)
    {
        tp->palette[i].r = pack_getc(file);
        tp->palette[i].g = pack_getc(file);
        tp->palette[i].b = pack_getc(file);
    }

    /* game board stuff */
    tp->board_w = pack_getc(file);
    tp->board_h = pack_getc(file);
    for(i = 0; i < MAX_PLAYERS; i++)
    {
        tp->board_x[i] = pack_igetw(file);
        tp->board_y[i] = pack_igetw(file);
        tp->score_x[i] = pack_igetw(file);
        tp->score_y[i] = pack_igetw(file);
        tp->preview_x[i] = pack_igetw(file);
        tp->preview_y[i] = pack_igetw(file);
        tp->level_x[i] = pack_igetw(file);
        tp->level_y[i] = pack_igetw(file);
        tp->block_x[i] = pack_igetw(file);
        tp->block_y[i] = pack_igetw(file);
        tp->player_message_x[i] = pack_igetw(file);
        tp->player_message_y[i] = pack_igetw(file);
    }

    tp->time_x = pack_igetw(file);
    tp->time_y = pack_igetw(file);
    tp->high_score_x = pack_igetw(file);
    tp->high_score_y = pack_igetw(file);
    tp->best_time_x = pack_igetw(file);
    tp->best_time_y = pack_igetw(file);
    tp->message_x = pack_igetw(file);
    tp->message_y = pack_igetw(file);

    /* block stuff */
    if(tp->header[3] == 21)
    {
	    tp->block_w = pack_getc(file);
	    tp->block_h = pack_getc(file);
	    block_image = create_bitmap(tp->block_w, tp->block_h);
	    block_flash_image = create_bitmap(tp->block_w, tp->block_h);
	    for(i = 0; i < NUM_BLOCK_TYPES; i++)
	    {
        	for(j = 0; j < tp->block_h; j++)
        	{
	            for(k = 0; k < tp->block_w; k++)
            	{
	                block_image->line[j][k] = pack_getc(file);
                	block_flash_image->line[j][k] = pack_getc(file);
            	}
        	}
   			tp->block_image[i] = bitmap2ani(block_image);
   			tp->block_flash_image[i] = bitmap2ani(block_flash_image);
    	}
    	destroy_bitmap(block_image);
    	destroy_bitmap(block_flash_image);
    }
    else if(tp->header[3] == 22)
    {
	    for(i = 0; i < NUM_BLOCK_TYPES; i++)
	    {
        	tp->block_image[i] = load_ani_fp(file, NULL);
        	tp->block_flash_image[i] = load_ani_fp(file, NULL);
    	}
    }

    /* tile stuff (old) */
    if(tp->header[3] == 21)
    {
	    tp->tile_w = pack_getc(file);
	    tp->tile_h = pack_getc(file);
	    if(tp->tile_w > 0 && tp->tile_h > 0)
	    {
		    tile_image = create_bitmap(tp->tile_w, tp->tile_h);
        	for(i = 0; i < NUM_BG_TILES; i++)
        	{
	            for(j = 0; j < tp->tile_h; j++)
            	{
	                for(k = 0; k < tp->tile_w; k++)
                	{
	                    tile_image->line[j][k] = pack_getc(file);
                	}
            	}
	            tp->tile_image[i] = bitmap2ani(tile_image);
        	}
        	destroy_bitmap(tile_image);
    	}
    	else
    	{
	        for(i = 0; i < tp->ry; i++)
        	{
	            for(j = 0; j < tp->rx; j++)
            	{
	                tp->bg_image->line[i][j] = pack_getc(file);
            	}
        	}
    	}
    }

    /* tile stuff (new) */
    else
    {
	    if(tp->tile_w > 0 && tp->tile_h > 0)
	    {
        	for(i = 0; i < NUM_BG_TILES; i++)
        	{
	            tp->tile_image[i] = load_ani_fp(file, NULL);
        	}
    	}
    	else
    	{
	        for(i = 0; i < tp->ry; i++)
        	{
	            for(j = 0; j < tp->rx; j++)
            	{
	                tp->bg_image->line[i][j] = pack_getc(file);
            	}
        	}
    	}
    }

    /* font stuff */
    load_ncdfont_fp(&tp->font, file);

    /* map stuff */
    tp->map_w = pack_getc(file);
    tp->map_h = pack_getc(file);
    if(tp->map_w > 0 && tp->map_h > 0)
    {
        tp->map = malloc(tp->map_w * tp->map_h);
        for(i = 0; i < tp->map_h; i++)
        {
            for(j = 0; j < tp->map_w; j++)
            {
                tp->map[i * tp->map_w + j] = pack_getc(file);
            }
        }
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_DROP] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_DROP] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_ROTATEUP] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_ROTATEUP] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_ROTATEDOWN] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_ROTATEDOWN] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_MOVE] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_MOVE] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_HITWALL] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_HITWALL] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_LAND] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_LAND] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_RUN] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_RUN] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_DELETE] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_DELETE] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_LEVELUP] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_LEVELUP] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_PAUSE] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_PAUSE] = NULL;
    }

    TempBit = pack_getc(file);
    if(TempBit == 1)
    {
        tp->sound[SOUND_ATTACK] = t3ss_load_wav_fp(file);
    }
    else
    {
        tp->sound[SOUND_ATTACK] = NULL;
    }

    /* close the file */
    pack_fclose(file);

    return 1;
}

/* free theme from memory */
void cs_destroy_theme(CS_THEME * tp)
{
    int i;

    t3ss_free_wav(tp->sound[SOUND_ATTACK]);
    t3ss_free_wav(tp->sound[SOUND_PAUSE]);
    t3ss_free_wav(tp->sound[SOUND_LEVELUP]);
    t3ss_free_wav(tp->sound[SOUND_DELETE]);
    t3ss_free_wav(tp->sound[SOUND_RUN]);
    t3ss_free_wav(tp->sound[SOUND_LAND]);
    t3ss_free_wav(tp->sound[SOUND_HITWALL]);
    t3ss_free_wav(tp->sound[SOUND_MOVE]);
    t3ss_free_wav(tp->sound[SOUND_ROTATEDOWN]);
    t3ss_free_wav(tp->sound[SOUND_ROTATEUP]);
    t3ss_free_wav(tp->sound[SOUND_DROP]);

    destroy_ncdfont(&tp->font);

    for(i = 0; i < NUM_BLOCK_TYPES; i++)
    {
        destroy_ani(tp->block_image[i]);
        destroy_ani(tp->block_flash_image[i]);
    }

    if(tp->map_w > 0 && tp->map_h > 0)
    {
        for(i = 0; i < NUM_BG_TILES; i++)
        {
            destroy_ani(tp->tile_image[i]);
        }
        free(tp->map);
    }
    destroy_bitmap(tp->board_image);
    destroy_bitmap(tp->bg_image);
    destroy_bitmap(tp->screen);
}
