#include <allegro.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include "font.h"

int text_mode_ncd = 0;    /* default to full color drawing */
int text_color_ncd = 255; /* default text color */

void my_itoa(int n, char * buffer)
{
    sprintf(buffer, "%d", n);
}

/* create a blank font, useful for importing bitmaps */
void create_ncdfont(NCDFONT * FontPointer, int w, int h)
{
    int i;
    BITMAP * TempPic;

    FontPointer->w = w;
    FontPointer->h = h;
    for(i = 0; i < 256; i++)
    {
        FontPointer->ch[i] = create_bitmap(w, h);
        clear(FontPointer->ch[i]);
    }

//    text_mode(-1);
    TempPic = create_bitmap(8, 8);
    for(i = 0; i < 256; i++)
    {
        clear_bitmap(TempPic);
        textprintf_ex(TempPic, font, 0, 0, 15, -1, "%c", i);
        stretch_sprite(FontPointer->ch[i], TempPic, 0, 0, FontPointer->w, FontPointer->h);
    }
    destroy_bitmap(TempPic);
}

int load_ncdfont_fp(NCDFONT * FontPointer, PACKFILE * fp)
{
    int i, j, k;
    char Header[4];

    /* read header */
    pack_fread(&Header, 4, fp);

    /* if header is wrong, return 0 */
    if(Header[0] != 'M' || Header[1] != 'F' || Header[2] != 'T' || Header[3] != 21)
    {
        return 0;
    }

    /* read font dimensions */
    FontPointer->w = pack_getc(fp);
    FontPointer->h = pack_getc(fp);

    /* read font data to file */
    for(i = 0; i < 256; i++)
    {
        FontPointer->ch[i] = create_bitmap(FontPointer->w, FontPointer->h);
        for(j = 0; j < FontPointer->h; j++)
        {
            for(k = 0; k < FontPointer->w; k++)
            {
                FontPointer->ch[i]->line[j][k] = pack_getc(fp);
            }
        }
    }

    return 1;
}

/* load a font from specified file */
int load_ncdfont(NCDFONT * FontPointer, char * FileName)
{
    PACKFILE * FontFile;

    /* open file */
    FontFile = pack_fopen(FileName, "r");

    /* return 0 if file couldn't be opened */
    if(FontFile == NULL)
    {
        return 0;
    }

    if(!load_ncdfont_fp(FontPointer, FontFile))
    {
        pack_fclose(FontFile);
        return 0;
    }

    /* close the file and return 1 on success */
    pack_fclose(FontFile);
    return 1;
}

int save_ncdfont_fp(NCDFONT * FontPointer, PACKFILE * fp)
{
    int i, j, k;
    char Header[4] = {'M', 'F', 'T', 21};

    /* write header */
    pack_fwrite(&Header, 4, fp);

    /* write font dimensions */
    pack_putc(FontPointer->w, fp);
    pack_putc(FontPointer->h, fp);

    /* write font data to file */
    for(i = 0; i < 256; i++)
    {
        for(j = 0; j < FontPointer->h; j++)
        {
            for(k = 0; k < FontPointer->w; k++)
            {
                pack_putc(FontPointer->ch[i]->line[j][k], fp);
            }
        }
    }

    return 1;
}

/* create a font file with specified filename */
int save_ncdfont(NCDFONT * FontPointer, char * FileName)
{
    PACKFILE * FontFile;

    /* open file */
    FontFile = pack_fopen(FileName, "w");

    /* return 0 if file couldn't be opened */
    if(FontFile == NULL)
    {
        return 0;
    }

    if(!save_ncdfont_fp(FontPointer, FontFile))
    {
        pack_fclose(FontFile);
        return 0;
    }

    /* close the file and return 1 on success */
    pack_fclose(FontFile);
    return 1;
}

/* free font from memory */
void destroy_ncdfont(NCDFONT * FontPointer)
{
    int i;

    for(i = 0; i < 256; i++)
    {
        if(FontPointer->ch[i] != NULL)
        {
            destroy_bitmap(FontPointer->ch[i]);
        }
    }
}

void ncd_text_mode(int mode)
{
    text_mode_ncd = mode;
}

void ncd_text_color(int color)
{
    text_color_ncd = color;
}

/* prints a string using specified NCDFONT */
void ncd_textout(BITMAP * b, NCDFONT * f, int x, int y, char * s)
{
    int i;

    for(i = 0; s[i] != '\0'; i++)
    {
        if(text_mode_ncd == 0)
        {
            draw_sprite(b, f->ch[(int)s[i]], x + i * f->w, y);
        }
        else if(text_mode_ncd == 1)
        {
            draw_character_ex(b, f->ch[(int)s[i]], x + i * f->w, y, text_color_ncd, -1);
        }
    }
}

/* prints a string using specified NCDFONT */
void ncd_textout_center(BITMAP * b, NCDFONT * f, int ox, int oy, char * s)
{
    int i, x;

    x = (b->w / 2) - ((strlen(s) / 2) * f->w);

    for(i = 0; s[i] != '\0'; i++)
    {
        if(text_mode_ncd == 0)
        {
            draw_sprite(b, f->ch[(int)s[i]], x + i * f->w + ox, oy);
        }
        else if(text_mode_ncd == 1)
        {
            draw_character_ex(b, f->ch[(int)s[i]], x + i * f->w + ox, oy, text_color_ncd, -1);
        }
    }
}

/* formatted output using a NCDFONT */
void ncd_printf(BITMAP *b, NCDFONT *f, int x, int y, char *format, ...)
{
    char buf[1024];

    va_list ap;
    va_start(ap, format);
    vsprintf(buf, format, ap);
    va_end(ap);

    ncd_textout(b, f, x, y, buf);
}

/* formatted output using a NCDFONT */
void ncd_printf_center(BITMAP *b, NCDFONT *f, int ox, int oy, char *format, ...)
{
    char buf[1024];

    va_list ap;
    va_start(ap, format);
    vsprintf(buf, format, ap);
    va_end(ap);

    ncd_textout_center(b, f, ox, oy, buf);
}

/* formatted output using a NCDFONT (limited length) */
void ncd_printf_truncated(BITMAP *b, NCDFONT *f, int x, int y, int length, char *format, ...)
{
    char buf[1024];

    va_list ap;
    va_start(ap, format);
    vsprintf(buf, format, ap);
    va_end(ap);

    buf[length] = '\0';
    ncd_textout(b, f, x, y, buf);
}

/* draws number using specified NCDFONT */
void ncd_draw_number(BITMAP * b, NCDFONT * f, int x, int y, unsigned long n, int d)
{
    char NUMBER[11]; /* array with maximum of 10 digits */
    int i;          /* looping variable */
    int j;           /* horizontal drawing position */

    /* converts <Number> to string <NUMBER> using base 10 */
//    itoa(n, NUMBER, 10);
    my_itoa(n, NUMBER);

    /* find total leading 0s (Digits - i after this loop) */
    for(i = 0; i < d; i++)
    {
        if(NUMBER[i] == '\0')
        {
            break;
        }
    }

    j = x; /* start drawing at x */

    /* draw leading 0s */
    while(i < d)
    {
        if(text_mode_ncd == 0)
        {
            draw_sprite(b, f->ch['0'], j, y);
        }
        else if(text_mode_ncd == 1)
        {
            draw_character_ex(b, f->ch['0'], j, y, text_color_ncd, -1);
        }
        i++;
        j += f->w;
    }

    /* draw the number */
    for(i = 0; NUMBER[i] != '\0'; i++)
    {
        if(text_mode_ncd == 0)
        {
            draw_sprite(b, f->ch[(int)NUMBER[i]], j, y);
        }
        else if(text_mode_ncd == 1)
        {
            draw_character_ex(b, f->ch[(int)NUMBER[i]], j, y, text_color_ncd, -1);
        }
        j += f->w;
    }
}

/* draws number using specified NCDFONT (right aligned) */
void ncd_draw_number_right(BITMAP * b, NCDFONT * f, int x, int y, long n, int d)
{
    char NUMBER[11]; /* array with maximum of 10 digits */
    int i;          /* looping variable */
    int j;           /* horizontal drawing position */

    /* converts <Number> to string <NUMBER> using base 10 */
//    itoa(n, NUMBER, 10);
    my_itoa(n, NUMBER);

    /* find total leading 0s (Digits - i after this loop) */
    for(i = 0; i < d; i++)
    {
        if(NUMBER[i] == '\0')
        {
            break;
        }
    }

    j = x; /* start drawing at x */

    /* draw leading 0s */
    while(i < d)
    {
        i++;
        j += f->w;
    }

    /* draw the number */
    for(i = 0; NUMBER[i] != '\0'; i++)
    {
        if(text_mode_ncd == 0)
        {
            draw_sprite(b, f->ch[(int)NUMBER[i]], j, y);
        }
        else if(text_mode_ncd == 1)
        {
            draw_character_ex(b, f->ch[(int)NUMBER[i]], j, y, text_color_ncd, -1);
        }
        j += f->w;
    }
}

/* left justified */
void ncd_draw_number_left(BITMAP * b, NCDFONT * f, int x, int y, long n, int d)
{
    char NUMBER[11]; /* array with maximum of 10 digits */
    int i;          /* looping variable */
    int j;           /* horizontal drawing position */

    /* converts <Number> to string <NUMBER> using base 10 */
    my_itoa(n, NUMBER);
//    itoa(n, NUMBER, 10);

    /* find total leading 0s (Digits - i after this loop) */
    for(i = 0; i < d; i++)
    {
        if(NUMBER[i] == '\0')
        {
            break;
        }
    }

    j = x; /* start drawing at x */

    /* draw the number */
    for(i = 0; NUMBER[i] != '\0'; i++)
    {
        if(text_mode_ncd == 0)
        {
            draw_sprite(b, f->ch[(int)NUMBER[i]], j, y);
        }
        else if(text_mode_ncd == 1)
        {
            draw_character_ex(b, f->ch[(int)NUMBER[i]], j, y, text_color_ncd, -1);
        }
        j += f->w;
    }
}

void ncd_show_text(BITMAP * bp, NCDFONT * fp, int x, int y, char * tp)
{
    int ox = x;
    int oy = y;
    int i;
    char stp[3] = " ";

    for(i = 0; i < strlen(tp); i++)
    {
        if(tp[i] == '\n')
        {
            ox = x;
            oy += fp->h + 1;
            if(oy >= bp->h) break;
        }
        else
        {
            stp[0] = tp[i];
            if(ox < bp->w && oy > -fp->h && oy < bp->h)
            {
                ncd_textout(bp, fp, ox, oy, stp);
                ox += fp->w;
            }
        }
    }
}
