/* All functions related to Crystal Stacker themes. */

#include "includes.h"

/* sets up video and memory areas for use by the game */
int InitializeTheme(THEME * ThemePointer)
{
        /* set up screen */
        if(set_gfx_mode(ThemePointer->Card, ThemePointer->ResX, ThemePointer->ResY, ThemePointer->ResX, ThemePointer->ResY) < 0)
                return 0; /* return 0 if graphics mode didn't set */

        /* set palette */
        set_palette(black_palette);

        /* returns 1 for success */
        return 1;
}

/* saves theme to specified file */
int SaveTheme(THEME * ThemePointer, char * FileName)
{
        PACKFILE * file;
        int i, j, k;

        /* open the file */
        file = pack_fopen(FileName, "w");
        if(file == NULL)
                return 0;

        /* write the header */
        pack_fwrite(ThemePointer->Header, 4, file);

        /* save video card info */
        pack_putc(ThemePointer->Card, file);
        pack_iputw(ThemePointer->ResX, file);
        pack_iputw(ThemePointer->ResY, file);

        /* save the palette */
        for(i = 0; i < 256; i++){
                pack_putc(ThemePointer->Palette[i].r, file);
                pack_putc(ThemePointer->Palette[i].g, file);
                pack_putc(ThemePointer->Palette[i].b, file);
                }

        /* save the board info */
        pack_putc(ThemePointer->BoardWidth, file);
        pack_putc(ThemePointer->BoardHeight, file);
        for(i = 0; i < MAX_PLAYERS; i++){
                pack_iputw(ThemePointer->BoardX[i], file);
                pack_iputw(ThemePointer->BoardY[i], file);
                pack_iputw(ThemePointer->ScoreX[i], file);
                pack_iputw(ThemePointer->ScoreY[i], file);
                pack_iputw(ThemePointer->NextX[i], file);
                pack_iputw(ThemePointer->NextY[i], file);
                pack_iputw(ThemePointer->LevelX[i], file);
                pack_iputw(ThemePointer->LevelY[i], file);
                pack_iputw(ThemePointer->BlockX[i], file);
                pack_iputw(ThemePointer->BlockY[i], file);
                pack_iputw(ThemePointer->PMessageX[i], file);
                pack_iputw(ThemePointer->PMessageY[i], file);
                }

        pack_iputw(ThemePointer->ClockX, file);
        pack_iputw(ThemePointer->ClockY, file);
        pack_iputw(ThemePointer->HighScoreX, file);
        pack_iputw(ThemePointer->HighScoreY, file);
        pack_iputw(ThemePointer->BestTimeX, file);
        pack_iputw(ThemePointer->BestTimeY, file);
        pack_iputw(ThemePointer->MessageX, file);
        pack_iputw(ThemePointer->MessageY, file);

        /* save the block info and data */
        pack_putc(ThemePointer->BlockWidth, file);
        pack_putc(ThemePointer->BlockHeight, file);
        for(i = 0; i < NUM_BLOCK_TYPES; i++){
                for(j = 0; j < ThemePointer->BlockHeight; j++){
                        for(k = 0; k < ThemePointer->BlockWidth; k++){
                                pack_putc(ThemePointer->Block[i]->line[j][k], file);
                                pack_putc(ThemePointer->BlockErase[i]->line[j][k], file);
                                }
                        }
                }

        /* save the tile info and data */
        pack_putc(ThemePointer->TileWidth, file);
        pack_putc(ThemePointer->TileHeight, file);
        if(ThemePointer->TileWidth > 0 && ThemePointer->TileHeight > 0)
        {
                for(i = 0; i < NUM_BG_TILES; i++)
                {
                        for(j = 0; j < ThemePointer->TileHeight; j++)
                        {
                                for(k = 0; k < ThemePointer->TileWidth; k++)
                                {
                                        pack_putc(ThemePointer->Tile[i]->line[j][k], file);
                                }
                        }
                }
        }
        else
        {
                for(i = 0; i < ThemePointer->ResY; i++)
                {
                        for(j = 0; j < ThemePointer->ResX; j++)
                        {
                                pack_putc(ThemePointer->BGPic->line[i][j], file);
                        }
                }
        }

        /* save the font info and data */
        save_ncdfont_fp(&ThemePointer->Font, file);

        /* save the map info and data */
        pack_putc(ThemePointer->MapWidth, file);
        pack_putc(ThemePointer->MapHeight, file);
        if(ThemePointer->MapWidth > 0 && ThemePointer->MapHeight > 0)
        {
                for(i = 0; i < ThemePointer->MapHeight; i++)
                {
                        for(j = 0; j < ThemePointer->MapWidth; j++)
                        {
                                pack_putc(ThemePointer->BGMapData[i * ThemePointer->MapWidth + j], file);
                        }
                }
        }

        if(ThemePointer->Sound[SOUND_DROP] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_DROP], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_ROTATEUP] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_ROTATEUP], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_ROTATEDOWN] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_ROTATEDOWN], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_MOVE] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_MOVE], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_HITWALL] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_HITWALL], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_LAND] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_LAND], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_RUN] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_RUN], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_DELETE] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_DELETE], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_LEVELUP] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_LEVELUP], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_PAUSE] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_PAUSE], file);
        }
        else
        {
                pack_putc(0, file);
        }

        if(ThemePointer->Sound[SOUND_ATTACK] != NULL)
        {
                pack_putc(1, file);
                ncds_save_wav_fp(ThemePointer->Sound[SOUND_ATTACK], file);
        }
        else
        {
                pack_putc(0, file);
        }

        /* close the file */
        pack_fclose(file);

        return 1;
}

/* loads theme from specified file and offset */
int LoadTheme(THEME * ThemePointer, char * FileName, unsigned long Offset)
{
        PACKFILE * file;
        int i, j, k;
        char TempBit;

        file = pack_fopen(FileName, "r");
        if(file == NULL)
                return 0;

        /* jump to specified offset */
        pack_fseek(file, Offset);

        /* read the header and see if it is a theme file */
        pack_fread(ThemePointer->Header, 4, file);
        if(ThemePointer->Header[0] != 'C' || ThemePointer->Header[1] != 'T' || ThemePointer->Header[2] != 'H' || ThemePointer->Header[3] != 21){
                pack_fclose(file);
                return 0;
                }

        /* video card stuff */
        ThemePointer->Card = pack_getc(file);
        ThemePointer->ResX = pack_igetw(file);
        ThemePointer->ResY = pack_igetw(file);
        ThemePointer->ScreenBuffer = create_bitmap(ThemePointer->ResX, ThemePointer->ResY);
        clear(ThemePointer->ScreenBuffer);
        ThemePointer->BGPic = create_bitmap(ThemePointer->ResX, ThemePointer->ResY);

        /* load the palette */
        for(i = 0; i < 256; i++){
                ThemePointer->Palette[i].r = pack_getc(file);
                ThemePointer->Palette[i].g = pack_getc(file);
                ThemePointer->Palette[i].b = pack_getc(file);
                }

        /* game board stuff */
        ThemePointer->BoardWidth = pack_getc(file);
        ThemePointer->BoardHeight = pack_getc(file);
        for(i = 0; i < MAX_PLAYERS; i++){
                ThemePointer->BoardX[i] = pack_igetw(file);
                ThemePointer->BoardY[i] = pack_igetw(file);
                ThemePointer->ScoreX[i] = pack_igetw(file);
                ThemePointer->ScoreY[i] = pack_igetw(file);
                ThemePointer->NextX[i] = pack_igetw(file);
                ThemePointer->NextY[i] = pack_igetw(file);
                ThemePointer->LevelX[i] = pack_igetw(file);
                ThemePointer->LevelY[i] = pack_igetw(file);
                ThemePointer->BlockX[i] = pack_igetw(file);
                ThemePointer->BlockY[i] = pack_igetw(file);
                ThemePointer->PMessageX[i] = pack_igetw(file);
                ThemePointer->PMessageY[i] = pack_igetw(file);
                }

        ThemePointer->ClockX = pack_igetw(file);
        ThemePointer->ClockY = pack_igetw(file);
        ThemePointer->HighScoreX = pack_igetw(file);
        ThemePointer->HighScoreY = pack_igetw(file);
        ThemePointer->BestTimeX = pack_igetw(file);
        ThemePointer->BestTimeY = pack_igetw(file);
        ThemePointer->MessageX = pack_igetw(file);
        ThemePointer->MessageY = pack_igetw(file);

        /* block stuff */
        ThemePointer->BlockWidth = pack_getc(file);
        ThemePointer->BlockHeight = pack_getc(file);
        for(i = 0; i < NUM_BLOCK_TYPES; i++){
                ThemePointer->Block[i] = create_bitmap(ThemePointer->BlockWidth, ThemePointer->BlockHeight);
                ThemePointer->BlockErase[i] = create_bitmap(ThemePointer->BlockWidth, ThemePointer->BlockHeight);
                for(j = 0; j < ThemePointer->BlockHeight; j++){
                        for(k = 0; k < ThemePointer->BlockWidth; k++){
                                ThemePointer->Block[i]->line[j][k] = pack_getc(file);
                                ThemePointer->BlockErase[i]->line[j][k] = pack_getc(file);
                                }
                        }
                }
        ThemePointer->BoardBGPic = create_bitmap(ThemePointer->BoardWidth * ThemePointer->BlockWidth, ThemePointer->BoardHeight * ThemePointer->BlockHeight);
        for(i = 0; i < ThemePointer->BoardHeight; i++)
        {
                for(j = 0; j < ThemePointer->BoardWidth; j++)
                {
                        blit(ThemePointer->Block[0], ThemePointer->BoardBGPic, 0, 0, j * ThemePointer->BlockWidth, i * ThemePointer->BlockHeight, ThemePointer->BoardWidth * ThemePointer->BlockWidth, ThemePointer->BoardHeight * ThemePointer->BlockHeight);
                }
        }

        /* tile stuff */
        ThemePointer->TileWidth = pack_getc(file);
        ThemePointer->TileHeight = pack_getc(file);
        if(ThemePointer->TileWidth > 0 && ThemePointer->TileHeight > 0)
        {
                for(i = 0; i < NUM_BG_TILES; i++)
                {
                        ThemePointer->Tile[i] = create_bitmap(ThemePointer->TileWidth, ThemePointer->TileHeight);
                        for(j = 0; j < ThemePointer->TileHeight; j++)
                        {
                                for(k = 0; k < ThemePointer->TileWidth; k++)
                                {
                                        ThemePointer->Tile[i]->line[j][k] = pack_getc(file);
                                }
                        }
                }
        }
        else
        {
                for(i = 0; i < ThemePointer->ResY; i++)
                {
                        for(j = 0; j < ThemePointer->ResX; j++)
                        {
                                ThemePointer->BGPic->line[i][j] = pack_getc(file);
                        }
                }
        }

        /* font stuff */
        load_ncdfont_fp(&ThemePointer->Font, file);

        /* map stuff */
        ThemePointer->MapWidth = pack_getc(file);
        ThemePointer->MapHeight = pack_getc(file);
        if(ThemePointer->MapWidth > 0 && ThemePointer->MapHeight > 0)
        {
                ThemePointer->BGMapData = malloc(ThemePointer->MapWidth * ThemePointer->MapHeight);
                for(i = 0; i < ThemePointer->MapHeight; i++)
                {
                        for(j = 0; j < ThemePointer->MapWidth; j++)
                        {
                                ThemePointer->BGMapData[i * ThemePointer->MapWidth + j] = pack_getc(file);
                                blit(ThemePointer->Tile[ThemePointer->BGMapData[i * ThemePointer->MapWidth + j]], ThemePointer->BGPic, 0, 0, j * ThemePointer->TileWidth, i * ThemePointer->TileHeight, ThemePointer->ResX, ThemePointer->ResY);
//                                blit(ThemePointer->Tile[pack_getc(file)], ThemePointer->BGPic, 0, 0, j * ThemePointer->TileWidth, i * ThemePointer->TileHeight, ThemePointer->ResX, ThemePointer->ResY);
                        }
                }
//                for(i = 0; i < NUM_BG_TILES; i++)
//                {
//                        destroy_bitmap(ThemePointer->Tile[i]);
//                }
        }

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_DROP] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_DROP] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_ROTATEUP] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_ROTATEUP] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_ROTATEDOWN] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_ROTATEDOWN] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_MOVE] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_MOVE] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_HITWALL] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_HITWALL] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_LAND] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_LAND] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_RUN] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_RUN] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_DELETE] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_DELETE] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_LEVELUP] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_LEVELUP] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_PAUSE] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_PAUSE] = NULL;

        TempBit = pack_getc(file);

        if(TempBit == 1)
        {
                ThemePointer->Sound[SOUND_ATTACK] = ncds_load_wav_fp(file);
        }
        else

                ThemePointer->Sound[SOUND_ATTACK] = NULL;

        /* close the file */
        pack_fclose(file);

        return 1;
}

/* free theme from memory */
void DestroyTheme(THEME * ThemePointer)
{
        int i;

        ncds_free_sample(ThemePointer->Sound[SOUND_ATTACK]);
        ncds_free_sample(ThemePointer->Sound[SOUND_PAUSE]);
        ncds_free_sample(ThemePointer->Sound[SOUND_LEVELUP]);
        ncds_free_sample(ThemePointer->Sound[SOUND_DELETE]);
        ncds_free_sample(ThemePointer->Sound[SOUND_RUN]);
        ncds_free_sample(ThemePointer->Sound[SOUND_LAND]);
        ncds_free_sample(ThemePointer->Sound[SOUND_HITWALL]);
        ncds_free_sample(ThemePointer->Sound[SOUND_MOVE]);
        ncds_free_sample(ThemePointer->Sound[SOUND_ROTATEDOWN]);
        ncds_free_sample(ThemePointer->Sound[SOUND_ROTATEUP]);
        ncds_free_sample(ThemePointer->Sound[SOUND_DROP]);

        ThemePointer->Sound[SOUND_ATTACK] = NULL;
        ThemePointer->Sound[SOUND_PAUSE] = NULL;
        ThemePointer->Sound[SOUND_LEVELUP] = NULL;
        ThemePointer->Sound[SOUND_DELETE] = NULL;
        ThemePointer->Sound[SOUND_RUN] = NULL;
        ThemePointer->Sound[SOUND_LAND] = NULL;
        ThemePointer->Sound[SOUND_HITWALL] = NULL;
        ThemePointer->Sound[SOUND_MOVE] = NULL;
        ThemePointer->Sound[SOUND_ROTATEDOWN] = NULL;
        ThemePointer->Sound[SOUND_ROTATEUP] = NULL;
        ThemePointer->Sound[SOUND_DROP] = NULL;

        destroy_ncdfont(&ThemePointer->Font);

        for(i = 0; i < NUM_BLOCK_TYPES; i++)
        {
                destroy_bitmap(ThemePointer->Block[i]);
        }

        if(ThemePointer->MapWidth > 0 && ThemePointer->MapHeight > 0)
        {
                for(i = 0; i < NUM_BG_TILES; i++)
                {
                        destroy_bitmap(ThemePointer->Tile[i]);
                }
                free(ThemePointer->BGMapData);
        }

        destroy_bitmap(ThemePointer->BoardBGPic);
        destroy_bitmap(ThemePointer->BGPic);
        destroy_bitmap(ThemePointer->ScreenBuffer);
}
