/* ------------------------------------------------------------------------ */
/* ---------------- T^3 Audio System (c)2001 T^3 Software. ---------------- */
/* ------------------------------------------------------------------------ */


/* include all necessary files */
#include <allegro.h>
#include <string.h>
#include "ncds.h"
#include "wav.h"


/* pointer to the MOD file */
NCDS_MOD * ncds_mod_file = NULL;

int ncds_effects_channels =  4; // how many channels for sound effects
int ncds_effects_channel =   0; // which effect channel to play sound in
int ncds_reserved_channels = 0; // add reserved channels
int ncds_sound_volume =    100; // volume of sound effects
int ncds_initialized =       0; // is T3 Sound System initialized?
int ncds_music_playing =     0; // is there music playing?
int ncds_channel_mode =      0; // mono, stereo, or reverse stereo
int ncds_sound_on =          1;
int ncds_music_on =          1;
int ncds_num_patterns =      0;



/* initialize sound system */
int ncds_install(long freq, int stereo)
{
    if(!ncds_initialized)
    {
        reserve_voices(32, 0);
        if(install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL))
        {
            ncds_initialized = 0;
            return 0;
        }
        ncds_channel_mode = stereo;

        /* start effects-only mode */
        ncds_effects_channel = 0;

        ncds_initialized = 1;
    }
    return 1;
}


/* deinitializes sound system */
void ncds_remove(void)
{
    if(ncds_initialized)
    {
        remove_sound();
        ncds_initialized = 0;
    }
}


/* tell sound system how many effects channels you want */
void ncds_set_effects_channels(int number)
{
    ncds_effects_channel = 0;
    ncds_effects_channels = number;
}


/* tell sound system how many reserved channels you want */
void ncds_set_reserved_channels(int number)
{
    ncds_reserved_channels = number;
}


void ncds_enable_effects(void)
{
    ncds_sound_on = 1;
}

void ncds_enable_music(void)
{
    ncds_music_on = 1;
}

void ncds_disable_effects(void)
{
    ncds_sound_on = 0;
}

void ncds_disable_music(void)
{
    ncds_music_on = 0;
}

NCDS_MOD * ncds_load_mod_fp(PACKFILE * fp)
{
    return NULL;
}

NCDS_MOD * ncds_load_mod(char * fn)
{
    return NULL;
}

void ncds_destroy_mod(NCDS_MOD * mp)
{
}

/* play a MOD file (.XM or .UNI) */
int ncds_play_mod(NCDS_MOD * mp)
{
    return 1;
}


/* stop sound playback */
void ncds_stop_mod(void)
{
}

/* set the music volume */
void ncds_set_music_volume(int volume)
{
}


/* set the sound effects volume */
void ncds_set_sound_volume(int volume)
{
    ncds_sound_volume = volume;
    set_volume(volume, 0);
}


/* jump to specified song pattern */
void ncds_set_music_pattern(int patnum)
{
}

/* loads a wav file from a file pointer */
NCDS_SAMPLE * ncds_load_wav_fp(PACKFILE * fp)
{
    NCDS_SAMPLE *si;
    int TempLength;
    WAVHEADER wh;
    char dID[4];

    /* read the wav header */

    /* identifier */
    pack_fread(wh.rID, 4, fp);

    /* (length of file) - 8 */
    wh.rLen = pack_igetl(fp);

    /* wave identifier */
    pack_fread(wh.wID, 4, fp);

    /* format chunk identifier */
    pack_fread(wh.fID, 4, fp);

    /* length of format chunk */
    wh.fLen = pack_igetl(fp);
    TempLength = wh.fLen;

    /* variable (always 0x01) */
    wh.wFormatTag = pack_igetw(fp);
    TempLength -= 2;

    /* number of channels */
    wh.nChannels = pack_igetw(fp);
    TempLength -= 2;

    /* read sample rate */
    wh.nSamplesPerSec = pack_igetl(fp);
    TempLength -= 4;

    /* read bytes per second */
    wh.nAvgBytesPerSec = pack_igetl(fp);
    TempLength -= 4;

    /* read bytes per sample (should be 1) */
    wh.nBlockAlign = pack_igetw(fp);
    TempLength -= 2;

    /* read bits per sample (should be 8) */
    wh.nFormatSpecific = pack_igetw(fp);
    TempLength -= 2;

    /* skip rest of format chunk */
    if(TempLength > 0)
    {
        pack_fread(NULL, TempLength, fp);
    }

	/* check it */
    if(memcmp(wh.rID, "RIFF" ,4) || memcmp(wh.wID, "WAVE", 4) || memcmp(wh.fID, "fmt ", 4))
    {
		return NULL;
    }

    pack_fread(dID, 4, fp);

    /* make sure we're at a wave data chunk before reading the rest */
    if(memcmp(dID, "data", 4))
    {
		return NULL;
    }

    /* allocate memory for the sample structure */
    wh.sLength = pack_igetl(fp);

    /* if file is 16 bit, double sample length */
    if(wh.nBlockAlign == 2)
    {
        wh.sLength >>= 1;
    }
    si = create_sample(wh.nFormatSpecific, wh.nChannels - 1, wh.nSamplesPerSec, wh.sLength);
    if(!si)
    {
        return NULL;
    }
    if(si->bits == 8)
    {
        pack_fread(si->data, si->len, fp);
    }
    else if(si->bits == 16)
    {
        pack_fread(si->data, si->len * 2, fp);
    }
	return si;
}

/* loads wave from the specified file and offset */
NCDS_SAMPLE * ncds_load_wav(char * FileName)
{
    PACKFILE * file;
    NCDS_SAMPLE * SamPointer;

    /* open the file */
    file = pack_fopen(FileName, "r");
    if(file == NULL)
    {
        return NULL;
    }

    /* load the sample */
    SamPointer = ncds_load_wav_fp(file);

    return SamPointer;
}


/* saves a wave file to file pointer */
int ncds_save_wav_fp(NCDS_SAMPLE * SamplePointer, PACKFILE * f)
{
    /* create the header */
    char RiffHeader[4] = {'R', 'I', 'F', 'F'};
    unsigned long FileLength = 36 + SamplePointer->len;
    char WaveHeader[4] = {'W', 'A', 'V', 'E'};
    char FormatHeader[4] = {'f', 'm', 't', ' '};
    unsigned long FormatLength = 0x10;
    unsigned int ExtraByte = 0x01;
    unsigned int Channels = 0x01;
    unsigned long SampleRate = SamplePointer->freq;
    unsigned long BytesPerSecond = SamplePointer->freq;
    unsigned int BytesPerSample = 0x01;
    unsigned int BitsPerSample = 0x08;
    char DataHeader[4] = {'d', 'a', 't', 'a'};
    unsigned long SampleLength = SamplePointer->len;

    /* write header to file */
    pack_fwrite(RiffHeader, 4, f);
    pack_iputl(FileLength, f);
    pack_fwrite(WaveHeader, 4, f);
    pack_fwrite(FormatHeader, 4, f);
    pack_iputl(FormatLength, f);
    pack_iputw(ExtraByte, f);
    pack_iputw(Channels, f);
    pack_iputl(SampleRate, f);
    pack_iputl(BytesPerSecond, f);
    pack_iputw(BytesPerSample, f);
    pack_iputw(BitsPerSample, f);
    pack_fwrite(DataHeader, 4, f);
    pack_iputl(SampleLength, f);

    /* write sample data to file */
    if(SamplePointer->bits == 8)
    {
        pack_fwrite(SamplePointer->data, SamplePointer->len, f);
    }
    else if(SamplePointer->bits == 16)
    {
        pack_fwrite(SamplePointer->data, SamplePointer->len * 2, f);
    }

    return 1;
}


/* saves wave to the specified file */
int ncds_save_wav(NCDS_SAMPLE * SamplePointer, char * FileName)
{
    PACKFILE * file;

    /* open file */
    file = pack_fopen(FileName, "w");
    if(file == NULL)
    {
        return 0;
    }

    /* save WAV to the file */
    if(!ncds_save_wav_fp(SamplePointer, file))
    {
        return 0;
    }

    /* close the file */
    pack_fclose(file);

    return 1;
}
/* release sound sample */
void ncds_free_sample(NCDS_SAMPLE * sp)
{
    if(sp != NULL)
    {
        destroy_sample(sp);
    }
}        


/* simplified sample playing function */
void ncds_play_sound(NCDS_SAMPLE * sp, int pan)
{
    if(sp != NULL && ncds_initialized && ncds_sound_on && ncds_effects_channels > 0)
    {
        if(ncds_channel_mode == NCDS_CH_MONO || ncds_channel_mode == NCDS_CH_STEREO)
        {
            play_sample(sp, ncds_sound_volume, pan, 1000, 0);
        }
        else if(ncds_channel_mode == NCDS_CH_REVERSE_STEREO)
        {
            play_sample(sp, ncds_sound_volume, 255 - pan, 1000, 0);
        }
    }
}


/* plays sample with the specified parameters */
void ncds_play_sound_ex(NCDS_SAMPLE * sp, int chan, int vol, int pan, int freq)
{
    if(sp != NULL && ncds_initialized && ncds_sound_on)
    {
        if(ncds_channel_mode == NCDS_CH_MONO || ncds_channel_mode == NCDS_CH_STEREO)
        {
            play_sample(sp, ncds_sound_volume, pan, 1000, 0);
        }
        else if(ncds_channel_mode == NCDS_CH_REVERSE_STEREO)
        {
            play_sample(sp, ncds_sound_volume, 255 - pan, 1000, 0);
        }
    }
}


/* play sound through reserved channels */
void ncds_play_sound_reserved(NCDS_SAMPLE * sp, int chan, int pan, int freq)
{
    if(sp != NULL && ncds_initialized && ncds_sound_on  && chan < ncds_reserved_channels)
    {
        if(ncds_channel_mode == NCDS_CH_MONO || ncds_channel_mode == NCDS_CH_STEREO)
        {
            play_sample(sp, ncds_sound_volume, pan, 1000, 0);
        }
        else if(ncds_channel_mode == NCDS_CH_REVERSE_STEREO)
        {
            play_sample(sp, ncds_sound_volume, 255 - pan, 1000, 0);
        }
    }
}


/* figure out sound panning based on X coordinate and width */
int ncds_place_sound(int x, int width)
{
    float wr = width / 256;
    float fx = x * wr;
        
    if(fx < 0)
    {
        return 0;
    }
    else if(fx > 255)
    {
        return 255;
    }
    return fx;
}
