#ifndef TYPES_H
#define TYPES_H


/* limit defines for credits functions */
#define MAX_CREDIT_NAMES    10
#define MAX_CREDITS         32


/* limit defines for theme functions */
#define NUM_BG_TILES        40
#define NUM_BLOCK_TYPES      7
#define MAX_PLAYERS          2
#define MAX_BOARDWIDTH      32

/* sound definitions (for easier reading) */
#define MAX_SOUNDS          11
#define SOUND_DROP           0
#define SOUND_ROTATEUP       1
#define SOUND_ROTATEDOWN     2
#define SOUND_MOVE           3
#define SOUND_HITWALL        4
#define SOUND_LAND           5
#define SOUND_RUN            6
#define SOUND_DELETE         7
#define SOUND_LEVELUP        8
#define SOUND_PAUSE          9
#define SOUND_ATTACK        10


/* maximum number of simultaneous 3D effects blocks */
#define MAX_EFFECTS_BLOCKS 256


/* crystals required for a run */
#define RUN_LENGTH           3


/* height of the falling stack */
#define STACK_HEIGHT         3


/* maximum number of queued messages per player */
#define QUEUE_SIZE          20

/* message types */
#define MESSAGE_FLASH        1
#define MESSAGE_SCROLL       2


/* maximum items in a menu */
#define MAX_MENU_ITEMS      16


/* structure for clocks */
typedef struct
{
    int minutes;
    int seconds;

} CS_CLOCK;


/* structure for player controllers */
typedef struct
{
    /* handles left input */
    int left;
    int was_left;

    /* handles right input */
    int right;
    int was_right;

    /* handles drop input */
    int drop;

    /* handles rotate input */
    int was_drop;
    int rotate_up;
    int was_rotate_up;
    int rotate_down;
    int was_rotate_down;

} CS_CONTROLLER;


/* an item in a credit */
typedef struct
{
    /* the text */
    char text[32];

    /* current state of the item */
    int  x, y, active;

} CREDIT_ITEM;

/* one credit (title and names) */
typedef struct
{
    /* credit title (example: Programmer) */
    CREDIT_ITEM title;

    /* credit names (people credited) */
    CREDIT_ITEM name[MAX_CREDIT_NAMES];

    /* number of names currently in credits list */
    int       name_count;

} CREDIT;

/* list of credits */
typedef struct
{
    /* the credits */
    CREDIT    credit[MAX_CREDITS];

    /* the number of credits currently in list */
    int       credit_count;

    /* pointer to title font (used to print title) */
    NCDFONT * title_font;

    /* pointer to name font (used to print names) */
    NCDFONT * name_font;

} CREDIT_LIST;


/* structure holding all game options */
typedef struct
{
    /* wild card option (0 = off, 1 = on) */
    int  wild_card;

    /* bomb option (0 = off, 1 = on) */
    int  bomb;

    /* effects option (0 = off, 1 = on) */
    int  effects;

    /* transitions option (0 = off, 1 = on) */
    int  transitions;

    /* joystick option (0 = off, 1 = on) */
    int  joystick;

    /* volume settings (0-100) */
    int  sound_volume;
    int  music_volume;

    /* sound quality setting (0 = Bad, ..., 4 = Best) */
    int  sound_quality;

    /* stereo setting (0 = Mono, 1 = Stereo, 2 = Reverse Stereo) */
    int  stereo;

    /* vsync option (0 = off, 1 = on) */
    int  vsync;

    /* name of currently selected theme (full path name) */
    char theme_name[256];

    /* used by the menu system to select starting level */
    int  level;

    /* used by the menu system to select starting blocks */
    int  block_index;

    /* used by the menu system to select starting time */
    int  time_index;

} CS_OPTIONS;


typedef struct
{
    int left;
    int right;
    int drop;
    int rup;
    int rdown;
} CKEYS;


/* used for menu selector */
typedef struct
{
    /* animation pointer */
    ANIMATION * ap;

    /* coordinates */
    int         x, y;

    /* current frame */
    int frame;

} CS_SPRITE;


/* a scaling block (used for "3D" effect) */
typedef struct
{
    /* use these internally */
    float fx, fy, fvx, fvy, fsize, ds, dy, dx, dvs, dvx, dvy;

    /* these are used to display the block */
    int x, y, size, color;

    /* indicates if block is active */
    char active;

} EFFECT_BLOCK;

/* collection of EFFECT_BLOCKs */
typedef struct
{
    /* the blocks */
    EFFECT_BLOCK block[MAX_EFFECTS_BLOCKS];

    /* tells generator which block to make next */
    int          current_block;

} EFFECT_BLOCKS;


/* structure for falling stack of crystals */
typedef struct
{
    /* the interrupt handler will use these */
    int  ix;
    int  iy;
    char itype[STACK_HEIGHT];
    char iactive;

    /* use these instead of the above for drawing */
    int  x;
    int  y;
    char type[STACK_HEIGHT];
    char active;

    int  dx;
    int  dy;
    char dtype[STACK_HEIGHT];
    char dactive;
} CS_STACK;


typedef struct
{
    /* the messages */
    char message[QUEUE_SIZE][128];

    /* keep up with the mode of each message (flashing or scrolling) */
    char mode[QUEUE_SIZE];

    /* can't remember */
    char cw;

    /* how many messages are in the queue */
    int  size;

    /* is the current message visible */
    int  visible;

    /* counter to do timing of scroll and flash */
    int  counter;

} CS_MESSAGE_LIST;


typedef struct
{
    /* current board data */
    unsigned char * board;

    /* current run data (used to keep up with and erase runs) */
    unsigned char * runs;

    BITMAP *        board_screen;       /* board1 bitmap */

    /* message list */
    CS_MESSAGE_LIST messages;

    /* player stats */
    unsigned long   score;
    int             blocks;

    /* which player this is */
    unsigned char   player;

    /* the currently falling stack */
    CS_STACK        stack;

    /* the next stack, once the current one lands */
    CS_STACK        next_stack;

    /* keep up with controller state */
    CS_CONTROLLER   control;

    /* current drop time (adjust each level) */
    int             drop_time;

    /* are there runs on the board (wait until done before playing on) */
    char            runs_left;

    /* used to help draw the wild cards and bombs */
    unsigned char   wild_count;

    /* game state */
    unsigned char   state;

    /* game mode */
    unsigned char   mode;

    /* counter used for something I can't remember yet */
    unsigned int    counter;


    /* store number of crystals destroyed by a run */
    int             removed;

    /* combo level counter */
    int             power;

    /* total of crystals destroyed by a combo */
    int             total_removed;

    /* current level */
    int             level;

    /* blocks remaining (used in Destroyer mode) */
    int             blocks_left;

    /* don't know */
    unsigned char   waitForAttack;

} CS_GAME;


typedef struct
{
    /* classic names and scores */
    char *        classic_name[10];
    unsigned long classic_score[10];

    /* frenzy names and scores */
    char *        frenzy_name[10][10];
    unsigned long frenzy_score[10][10];

    /* destroyer names and times */
    char *        destroyer_name[10][10];
    unsigned char destroyer_minutes[10][10];
    unsigned char destroyer_seconds[10][10];
} CS_TOP10;


typedef struct
{
    /* where to place the selector when this item is selected */
    int x;
    int y;

    /* the menu item text */
    char text[128];

    /* text that appears at the bottom when this item is selected */
    char sub_text[41];

    /* procedure for left keypress */
    void(*left_proc)();

    /* procedure for right keypress */
    void(*right_proc)();

    /* procedure for enter keypress */
    void(*enter_proc)();

} CS_MENU_ITEM;

typedef struct
{
    /* menu items */
    CS_MENU_ITEM item[MAX_MENU_ITEMS];

    /* don't know if this is used */
    NCDFONT *    font;

    /* number of items in menu */
    int          item_count;

    /* which item is selected */
    int          selected;

    /* which menu to go to when ESC pressed */
    int          parent;

} CS_MENU;


typedef struct
{
    /* store the header when loading */
    char      header[4];

    /* graphics settings (rx = width, ry = height) */
    unsigned  char  card;
    unsigned  short rx;
    unsigned  short ry;

    /* theme palette */
    PALETTE   palette;

    /* screen buffer to draw to */
    BITMAP *  screen;

    /* the theme background image */
    BITMAP *  bg_image;

    /* the theme board background image */
    BITMAP *  board_image;

    /* block dimensions (should be the same) */
    int       block_w;
    int       block_h;

    /* block images */
    ANIMATION * block_image[NUM_BLOCK_TYPES];
    ANIMATION * block_flash_image[NUM_BLOCK_TYPES];

    /* board dimensions (should be 6x13) */
    int       board_w;
    int       board_h;

    /* board locations */
    short     board_x[MAX_PLAYERS];
    short     board_y[MAX_PLAYERS];

    /* tile dimensions (unless fullscreen backdrop is used) */
    int       tile_w;
    int       tile_h;

    /* tilemap dimensions (dependant on tile dimensions above) */
    int       map_w;
    int       map_h;

    /* tilemap data */
    char *    map;

    /* background tile images */
    ANIMATION * tile_image[NUM_BG_TILES];

    /* size of the theme font */
    int       font_size;

    /* the theme font */
    NCDFONT   font;

    /* score locations */
    short     score_x[MAX_PLAYERS];
    short     score_y[MAX_PLAYERS];

    /* preview locations */
    short     preview_x[MAX_PLAYERS];
    short     preview_y[MAX_PLAYERS];

    /* level display locations */
    short     level_x[MAX_PLAYERS];
    short     level_y[MAX_PLAYERS];

    /* block counter locations */
    short     block_x[MAX_PLAYERS];
    short     block_y[MAX_PLAYERS];

    /* player message locations */
    short     player_message_x[MAX_PLAYERS];
    short     player_message_y[MAX_PLAYERS];

    /* main clock location */
    short     time_x;
    short     time_y;

    /* high score location */
    short     high_score_x;
    short     high_score_y;

    /* best time location */
    short     best_time_x;
    short     best_time_y;

    /* main message location */
    short     message_x;
    short     message_y;

    /* theme sounds */
    MSAMPLE * sound[MAX_SOUNDS];

} CS_THEME;


#endif
