/* ------------------------------------------------------------------------ */
/* ---------------- T^3 Audio System (c)2001 T^3 Software. ---------------- */
/* ------------------------------------------------------------------------ */


/* include all necessary files */
#define __USE_GNU
#include <allegro.h>
#include <aldumb.h> 
#include "t3ssal.h"
#include "mywav.h"
#ifdef T3SS_PTHREADS
#include <unistd.h>
#include <pthread.h>
#endif


int t3ss_effects_channels =  4; // how many channels for sound effects
int t3ss_effects_channel =   0; // which effect channel to play sound in
int t3ss_reserved_channels = 0; // add reserved channels
int t3ss_sound_volume =    100; // volume of sound effects
int t3ss_music_volume =    100; // volume of music
int t3ss_initialized =       0; // is T3 Sound System initialized?
int t3ss_music_playing =     0; // is there music playing?
int t3ss_channel_mode =      0; // mono, stereo, or reverse stereo
long t3ss_freq =         44100;
int t3ss_sound_on =          1;
int t3ss_music_on =          1;
int t3ss_num_patterns =      0;
int t3ss_music_pattern =     0;

/* dumb / mod(music) playing related vars */
static DUH *t3ss_duh = NULL;
static AL_DUH_PLAYER *t3ss_duh_player = NULL;

#ifdef T3SS_PTHREADS
static pthread_t t3ss_duh_player_thread;
static pthread_mutex_t t3ss_duh_player_mutex;
#endif


/* initialize sound system */
int t3ss_init(long freq, int stereo)
{
    if(!t3ss_initialized)
    {
        /* try to install sound */
        reserve_voices(32, 0);
        if(install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL))
        {
            t3ss_initialized = 0;
            return 0;
        }
        set_volume((float)t3ss_sound_volume * 2.5, 0);

        /* setup sound channels (mono/stereo) */
        t3ss_channel_mode = stereo;
        t3ss_freq = freq;
        t3ss_effects_channel = 0;

        /* init dumb */
        dumb_register_packfiles();
#ifdef T3SS_PTHREADS
        pthread_mutex_init(&t3ss_duh_player_mutex, NULL);
#endif
        
        t3ss_initialized = 1;
    }
    return 1;
}


/* deinitializes sound system */
void t3ss_exit(void)
{
    if(t3ss_initialized)
    {
        t3ss_stop_mod();
        dumb_exit();
        remove_sound();
        t3ss_initialized = 0;
    }
}


/* tell sound system how many effects channels you want */
void t3ss_set_effects_channels(int number)
{
    t3ss_effects_channel = 0;
    t3ss_effects_channels = number;
}


/* tell sound system how many reserved channels you want */
void t3ss_set_reserved_channels(int number)
{
    t3ss_reserved_channels = number;
}


void t3ss_enable_effects(void)
{
    t3ss_sound_on = 1;
}

void t3ss_enable_music(void)
{
    t3ss_music_on = 1;
}

void t3ss_disable_effects(void)
{
    t3ss_sound_on = 0;
}

void t3ss_disable_music(void)
{
    if (t3ss_duh_player)
        t3ss_stop_mod();
    t3ss_music_on = 0;
}


#ifdef T3SS_PTHREADS
static void *t3ss_duh_player_thread_function(void *arg)
{
    while(1)
    {
        pthread_mutex_lock(&t3ss_duh_player_mutex);
        /* this may be nuked if things fail in t3ss_set_music_pattern */
        if (t3ss_duh_player)
            al_poll_duh(t3ss_duh_player);
        pthread_mutex_unlock(&t3ss_duh_player_mutex);
        pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
        usleep(10000);
        pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, NULL);
    }
}
#endif

/* play a MOD file (.XM or .UNI) */
int t3ss_play_mod(char * fn)
{
    DUMB_IT_SIGDATA *sigdata;
    DUH_SIGRENDERER *renderer;
    
    if (!t3ss_initialized || !t3ss_music_on)
        return 0;
        
    if (t3ss_duh_player)
        t3ss_stop_mod();
    
    t3ss_duh = dumb_load_xm_quick(fn);
    if (!t3ss_duh)
        return 0;
        
    sigdata = duh_get_it_sigdata(t3ss_duh);
    if (sigdata)
        t3ss_num_patterns = dumb_it_sd_get_n_orders(sigdata);
    else
        t3ss_num_patterns = 0;
        
    if (t3ss_music_pattern >= t3ss_num_patterns)
        t3ss_music_pattern = 0;
        
    renderer = dumb_it_start_at_order(t3ss_duh, t3ss_channel_mode? 2:1,
        t3ss_music_pattern);
    if (!renderer)
    {
        unload_duh(t3ss_duh);
        t3ss_duh = NULL;
        return 0;
    }

    t3ss_duh_player = al_duh_encapsulate_sigrenderer(renderer,
        t3ss_music_volume/100.0f, 4096, t3ss_freq);
    if (!t3ss_duh_player)
    {
        duh_end_sigrenderer(renderer);
        unload_duh(t3ss_duh);
        t3ss_duh = NULL;
        return 0;
    }
        
#ifdef T3SS_PTHREADS
    if (pthread_create(&t3ss_duh_player_thread, NULL,
            t3ss_duh_player_thread_function, NULL))
    {
        al_stop_duh(t3ss_duh_player);
        t3ss_duh_player = NULL;
        unload_duh(t3ss_duh);
        t3ss_duh = NULL;
        return 0;
    }
#endif

    t3ss_music_playing = 1;
    return 1;
}


/* stop sound playback */
void t3ss_stop_mod(void)
{
    if (t3ss_duh_player)
    {
        /* stop the player thread */
#ifdef T3SS_PTHREADS
        pthread_cancel(t3ss_duh_player_thread);
        pthread_join(t3ss_duh_player_thread, NULL);
#endif
        /* and clean up everything */
        al_stop_duh(t3ss_duh_player);
        t3ss_duh_player = NULL;
        unload_duh(t3ss_duh);
        t3ss_duh = NULL;
        t3ss_music_playing = 0;
    }
}


/* set the music volume */
void t3ss_set_music_volume(int volume)
{
    t3ss_music_volume = volume;
    if (t3ss_duh_player)
    {
#ifdef T3SS_PTHREADS
        pthread_mutex_lock(&t3ss_duh_player_mutex);
#endif
        al_duh_set_volume(t3ss_duh_player, t3ss_music_volume/100.0f);
#ifdef T3SS_PTHREADS
        pthread_mutex_unlock(&t3ss_duh_player_mutex);
#endif
    }
}


/* set the sound effects volume */
void t3ss_set_sound_volume(int volume)
{
    t3ss_sound_volume = volume;
}


/* jump to specified song pattern */
void t3ss_set_music_pattern(int patnum)
{
    t3ss_music_pattern = patnum;
    if (t3ss_duh_player)
    {
        DUH_SIGRENDERER *renderer;
        if (t3ss_music_pattern >= t3ss_num_patterns)
            t3ss_music_pattern = 0;
#ifdef T3SS_PTHREADS
        pthread_mutex_lock(&t3ss_duh_player_mutex);
#endif
        al_stop_duh(t3ss_duh_player);
        t3ss_duh_player = NULL;

        renderer = dumb_it_start_at_order(t3ss_duh, 2, t3ss_music_pattern);
        if (renderer)
            t3ss_duh_player = al_duh_encapsulate_sigrenderer(renderer,
                t3ss_music_volume/100.0f, 4096, t3ss_freq);

        if (!t3ss_duh_player)
        {
            if (renderer)
                duh_end_sigrenderer(renderer);
            unload_duh(t3ss_duh);
            t3ss_duh = NULL;
        }
#ifdef T3SS_PTHREADS
        pthread_mutex_unlock(&t3ss_duh_player_mutex);
        /* stop the player thread? */
        if (!t3ss_duh_player)
        {
            pthread_cancel(t3ss_duh_player_thread);
            pthread_join(t3ss_duh_player_thread, NULL);
        }
#endif
    }
}


/* load a WAV sample */
MSAMPLE * t3ss_load_wav_fp(PACKFILE * fp)
{
    MSAMPLE * sp = NULL;

    if(t3ss_initialized)
    {
        sp = tc_load_wav_fp(fp);
        return sp;
    }
    return NULL;
}


/* load a WAV sample */
MSAMPLE * t3ss_load_wav_fn(char * fn)
{
    MSAMPLE * TempSample;

    if(t3ss_initialized)
    {
        TempSample = tc_load_wav(fn);
        return TempSample;
//        TempSample=load_wav(fn);
//        return TempSample;
    }
    return NULL;
}


/* save a WAV sample into specified file pointer */
int t3ss_save_wav_fp(MSAMPLE * sp, PACKFILE * fp)
{
    if(sp != NULL)
    {
        tc_save_wav_fp(fp, sp);
    }

    return 0;
}


/* save a WAV sample into specified file */
int t3ss_save_wav_fn(MSAMPLE * sp, char * fn)
{
    if(sp != NULL)
    {
        tc_save_wav(fn, sp);
    }

    return 0;
}


/* release sound sample */
void t3ss_free_wav(MSAMPLE * sp)
{
    if(sp != NULL)
    {
        destroy_sample(sp);
    }
}        


/* simplified sample playing function */
void t3ss_play_sound(MSAMPLE * sp, int pan, int freq)
{
    t3ss_effects_channel++;
    if(t3ss_effects_channel >= t3ss_effects_channels)
    {
        t3ss_effects_channel = 0;
    }
    t3ss_play_sound_ex(sp, t3ss_effects_channel, t3ss_sound_volume, pan, freq);
}


/* plays sample with the specified parameters */
void t3ss_play_sound_ex(MSAMPLE * sp, int chan, int vol, int pan, int freq)
{
	int voice;
	
    if(sp != NULL && t3ss_initialized && t3ss_sound_on &&
            t3ss_effects_channels > 0)
    {
	    voice = allocate_voice(sp);
	    voice_set_volume(voice, (float)vol * 2.5);
	    voice_set_pan(voice, pan);
	    voice_set_frequency(voice, freq >= 0 ? freq : sp->freq);
	    voice_start(voice);
	    release_voice(voice);
//        play_sample(sp, (float)vol * 2.5, pan, freq, 0);
        t3ss_effects_channel = t3ss_effects_channel;
    }
}


/* play sound through reserved channels */
void t3ss_play_sound_reserved(MSAMPLE * sp, int chan, int pan, int freq)
{
	int voice;
	
    if(sp != NULL && t3ss_initialized && t3ss_sound_on &&
            chan < t3ss_reserved_channels)
    {
	    voice = allocate_voice(sp);
	    voice_set_volume(voice, (float)t3ss_sound_volume * 2.5);
	    voice_set_pan(voice, pan);
	    voice_set_frequency(voice, freq >= 0 ? freq : sp->freq);
	    voice_start(voice);
	    release_voice(voice);
//        play_sample(sp, (float)t3ss_sound_volume * 2.5, pan, freq, 0);
    }
}


/* figure out sound panning based on X coordinate and width */
int t3ss_place_sound(int x, int width)
{
    float wr = (float)width / 256.0;
    float fx = x * wr;
        
    if(fx < 0)
    {
        return 0;
    }
    else if(fx > 255)
    {
        return 255;
    }
    return fx;
}

void t3ss_pause_music(void)
{
    if(t3ss_duh_player)
    {
#ifdef T3SS_PTHREADS
        pthread_mutex_lock(&t3ss_duh_player_mutex);
#endif
        al_pause_duh(t3ss_duh_player);
#ifdef T3SS_PTHREADS
        pthread_mutex_unlock(&t3ss_duh_player_mutex);
#endif
        t3ss_music_playing = 0;
    }
}

void t3ss_resume_music(void)
{
    if (t3ss_duh_player)
    {
#ifdef T3SS_PTHREADS
        pthread_mutex_lock(&t3ss_duh_player_mutex);
#endif
        al_resume_duh(t3ss_duh_player);
#ifdef T3SS_PTHREADS
        pthread_mutex_unlock(&t3ss_duh_player_mutex);
#endif
        t3ss_music_playing = 1;
    }
}

#ifndef T3SS_PTHREADS
void t3ss_poll(void)
{
    if(t3ss_duh_player)
    {
        al_poll_duh(t3ss_duh_player);
    }
}
#endif
