#include "includes.h"

/* stuff used by the game engine */
CS_THEME      cs_theme;
CS_GAME       cs_game[MAX_PLAYERS];
CS_CLOCK      cs_clock;
EFFECT_BLOCKS cs_effects_blocks;
int           cs_panning[3][MAX_PLAYERS];

/* data used by the game */
CS_OPTIONS    cs_options;
CS_TOP10      cs_scores;
CKEYS         cs_controls[2];

/* data used in the menu */
CS_MENU       cs_menu[TOTAL_MENUS];
NCDFONT       cs_menu_font;
NCDFONT       cs_big_menu_font;
NCDFONT       cs_big_menu_font_un;
NCDFONT       cs_big_menu_font_green;
NCDFONT       cs_big_menu_font_green_un;
NCDFONT       cs_big_menu_font_red;
NCDFONT       cs_big_menu_font_red_un;

int           current_menu = MENU_MAIN;
BITMAP *      cs_menu_bg_image;
BITMAP *      cs_logo_image;
BITMAP *      cs_high_score_image;
CS_SPRITE     cs_select_sprite;
ANIMATION *   cs_select_animation;
char *        sound_quality_text[5] = {"Bad", "AM Quality", "FM Quality", "Excellent", "CD Quality"};
char *        toggle_text[2] = {"Off", "On"};
char *        stereo_mode_text[3] = {"Mono", "Stereo", "Reverse Stereo"};
char *        screen_mode_text[3] = {"Full Screen", "Windowed", "Windowed (2x)"};
char          cs_game_mode = CS_1NS;
MSAMPLE *     cs_sound[MAX_CS_SOUNDS];
char          menu_quit = 0;

/* store filenames used by the program */
char          music_name[256];

/* charts used to set things up */
const int     block_chart[10] = {50, 100, 150, 200, 250, 300, 350, 400, 450, 500};
const int     time_minutes_chart[10] = {0, 1, 1, 2, 2, 3, 3, 4, 4, 5};
const int     time_seconds_chart[10] = {30, 0, 30, 0, 30, 0, 30, 0, 30, 0};
const int     sound_quality_chart[5] = {8000, 11025, 22050, 33075, 44100};

/* variables use by game modules */
char          game_quit = 0;
char          beat_score = 0;

/* stuff used by the high score routines */
int           cs_place = 0;

/* lookup chart used to find drop times */
int           level_chart[20] = {670, 600, 540, 480, 420, 370, 320, 280, 240, 200, 170, 140, 120, 100, 80, 70, 60, 55, 50, 45};

/* screen buffer and palette */
BITMAP *      screen_buffer;
PALETTE       screen_palette;
BITMAP *      mouse_image;

/* miscellaneous variables */
char          nosound = 0;
char          cs_screen_mode = CS_FULL;
int           drop_amount[MAX_PLAYERS];
unsigned long cs_frame = 0;

/* loads all the data */
void cs_load_data(void)
{
    /* load sounds */
    cs_sound[SOUND_MENU_MOVE] = t3ss_load_wav_fn(DATADIR "cs.dat#menu_sound");
    cs_sound[SOUND_MENU_ENTER] = t3ss_load_wav_fn(DATADIR "cs.dat#menu_change_sound");
    cs_sound[SOUND_MENU_CHANGE] = t3ss_load_wav_fn(DATADIR "cs.dat#menu_change_sound");
    cs_sound[SOUND_MENU_SWOOSH] = t3ss_load_wav_fn(DATADIR "cs.dat#menu_swoosh_sound");
    cs_sound[SOUND_MENU_BACK] = t3ss_load_wav_fn(DATADIR "cs.dat#menu_back_sound");
    cs_sound[SOUND_MENU_TYPE] = t3ss_load_wav_fn(DATADIR "cs.dat#menu_type_sound");
    cs_sound[SOUND_GAME_STOP] = t3ss_load_wav_fn(DATADIR "cs.dat#game_stop_sound");
    cs_sound[SOUND_GAME_GO] = t3ss_load_wav_fn(DATADIR "cs.dat#game_go_sound");
    cs_sound[SOUND_FADE_OUT] = t3ss_load_wav_fn(DATADIR "cs.dat#game_fade_out_sound");
    cs_sound[SOUND_FADE_IN] = t3ss_load_wav_fn(DATADIR "cs.dat#game_fade_in_sound");
    cs_sound[SOUND_ENTER_NAME] = t3ss_load_wav_fn(DATADIR "cs.dat#menu_enter_sound");

    /* load images */
    cs_menu_bg_image = load_pcx(DATADIR "cs.dat#menu", screen_palette);
    cs_logo_image = load_pcx(DATADIR "cs.dat#cslogo", NULL);
    cs_high_score_image = load_pcx(DATADIR "cs.dat#highs", NULL);
    cs_select_animation = load_ani(DATADIR "cs.dat#menu_selector", NULL);
    cs_select_sprite.ap = cs_select_animation;
    load_ncdfont(&cs_menu_font, DATADIR "cs.dat#menu_font");
    load_ncdfont(&cs_big_menu_font, DATADIR "cs.dat#menu_font_big");
    load_ncdfont(&cs_big_menu_font_un, DATADIR "cs.dat#menu_ufont_big");
    load_ncdfont(&cs_big_menu_font_red, DATADIR "cs.dat#menu_rfont_big");
    load_ncdfont(&cs_big_menu_font_green, DATADIR "cs.dat#menu_gfont_big");
    load_ncdfont(&cs_big_menu_font_red_un, DATADIR "cs.dat#menu_urfont_big");
    load_ncdfont(&cs_big_menu_font_green_un, DATADIR "cs.dat#menu_ugfont_big");
    mouse_image = load_pcx(DATADIR "cs.dat#cursor", NULL);
}

/* frees all the data */
void cs_free_data(void)
{
    int i;

    /* free sounds */
    for(i = 0; i < MAX_CS_SOUNDS; i++)
    {
        t3ss_free_wav(cs_sound[i]);
    }

    /* free images */
    destroy_ncdfont(&cs_menu_font);
    destroy_ncdfont(&cs_big_menu_font);
    destroy_ncdfont(&cs_big_menu_font_un);
    destroy_ncdfont(&cs_big_menu_font_red);
    destroy_ncdfont(&cs_big_menu_font_green);
    destroy_ncdfont(&cs_big_menu_font_red_un);
    destroy_ncdfont(&cs_big_menu_font_green_un);
    destroy_ani(cs_select_animation);
    destroy_bitmap(cs_menu_bg_image);
    destroy_bitmap(cs_logo_image);
    destroy_bitmap(cs_high_score_image);
    destroy_bitmap(mouse_image);
}

/* handle graphics mode switching */
void cs_gfx_mode(int w, int h, PALETTE mpal)
{
    int dw, dh;

    /* don't think this works */
    request_refresh_rate(60);

    /* only switch when necessary */
    if(w != SCREEN_W || h != SCREEN_H)
    {
            /* set full screen mode */
            if(cs_screen_mode == CS_FULL)
            {
                if(set_gfx_mode(GFX_AUTODETECT, w, h, 0, 0))
            	{
	                if(set_gfx_mode(GFX_AUTODETECT, w * 2, h * 2, 0, 0))
            		{
		                if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w * 2, h * 2, 0, 0))
		                {
			                if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0))
		                	{
			                	allegro_message("Unable to set graphics mode!");
			                	exit(0);
		                	}
		                }
            		}
            	}
            }

            /* or normal window */
            else if(cs_screen_mode == CS_WIN)
            {
                if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0))
                {
					allegro_message("Unable to set graphics mode!");
					exit(0);
                }
            }

            /* or 2x window */
            else if(cs_screen_mode == CS_WIN2)
            {
                /* set double size window if possible */
                get_desktop_resolution(&dw, &dh);
                if(dw > 640 && dh > 480)
                {
                    if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w * 2, h * 2, 0, 0))
                    {
	                    if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0))
                    	{
	                    	allegro_message("Unable to set graphics mode!");
	                    	exit(0);
                    	}
                    }
                }

                /* if can't do double size, attempt normal window */
                else
                {
                    cs_screen_mode = CS_WIN;
                    if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0))
                    {
                    	allegro_message("Unable to set graphics mode!");
                    	exit(0);
                    }
                }
            }
    }

    /* set the mouse sprite */
    set_mouse_sprite(mouse_image);

    /* fix the colors */
    set_palette(mpal);
    gui_fg_color = 234;
    gui_bg_color = 238;
}

/* same as above, but sets the mode every time instead of by necessity */
void cs_gfx_mode_force(int w, int h, PALETTE mpal)
{
    int dw, dh;

    /* don't think this works */
    request_refresh_rate(60);

            /* set full screen mode */
            if(cs_screen_mode == CS_FULL)
            {
                if(set_gfx_mode(GFX_AUTODETECT, w, h, 0, 0))
            	{
	                if(set_gfx_mode(GFX_AUTODETECT, w * 2, h * 2, 0, 0))
            		{
		                if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w * 2, h * 2, 0, 0))
		                {
			                if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0))
		                	{
			                	allegro_message("Unable to set graphics mode!");
			                	exit(0);
		                	}
		                }
            		}
            	}
            }

            /* or normal window */
            else if(cs_screen_mode == CS_WIN)
            {
                if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0))
                {
					allegro_message("Unable to set graphics mode!");
					exit(0);
                }
            }

            /* or 2x window */
            else if(cs_screen_mode == CS_WIN2)
            {
                /* set double size window if possible */
                get_desktop_resolution(&dw, &dh);
                if(dw > 640 && dh > 480)
                {
                    if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w * 2, h * 2, 0, 0))
                    {
	                    if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0))
                    	{
	                    	allegro_message("Unable to set graphics mode!");
	                    	exit(0);
                    	}
                    }
                }

                /* if can't do double size, attempt normal window */
                else
                {
                    cs_screen_mode = CS_WIN;
                    if(set_gfx_mode(GFX_AUTODETECT_WINDOWED, w, h, 0, 0))
                    {
                    	allegro_message("Unable to set graphics mode!");
                    	exit(0);
                    }
                }
            }

    /* fix the colors */
    set_palette(mpal);
    gui_fg_color = 234;
    gui_bg_color = 238;
}

/* to rid the code of numerous if statements */
void cs_vsync(void)
{
    if(cs_options.vsync)
    {
        vsync();
    }
    t3ss_poll();
}

/* handles blitting a full-screen image to the screen */
void cs_screen_blit(BITMAP * bp, int x, int y)
{
    int screen_x, screen_y;

    /* center the image */
    if(cs_screen_mode == CS_FULL || cs_screen_mode == CS_WIN)
    {
        screen_x = (bp->w == SCREEN_W) ? (0) : ((SCREEN_W - bp->w) >> 1);
        screen_y = (bp->h == SCREEN_H) ? (0) : ((SCREEN_H - bp->h) >> 1);
    }
    else
    {
        screen_x = ((bp->w << 1) == SCREEN_W) ? (0) : ((SCREEN_W - (bp->w << 1)) >> 1);
        screen_y = ((bp->h << 1) == SCREEN_H) ? (0) : ((SCREEN_H - (bp->h << 1)) >> 1);
    }

	if(bp->w != SCREEN_W || bp->h != SCREEN_H)
	{
        stretch_blit(bp, screen, 0, 0, bp->w, bp->h, 0, 0, bp->w * 2, bp->h * 2);
	}
	else
	{
        blit(bp, screen, 0, 0, screen_x, screen_y, SCREEN_W, SCREEN_H);
	}
}

char *cs_get_config_file_name(char *suffix)
{
    static char config_file_name[512];

#ifdef __unix__   
    extern char *get_home_dir(void);
    char *home_dir = get_home_dir();
     
    snprintf(config_file_name, sizeof(config_file_name), "%s/.cs.%s",
        home_dir? home_dir:".", suffix);
#else
    snprintf(config_file_name, sizeof(config_file_name), "cs.%s", suffix);
#endif
    return config_file_name;
}

/* load config file */
void cs_load_config(void)
{
    override_config_file(cs_get_config_file_name("cfg"));

    /* read options */
    cs_options.wild_card   = get_config_int("CS_Options", "Wildcard", 1);
    cs_options.bomb        = get_config_int("CS_Options", "Bomb", 1);
    cs_options.effects     = get_config_int("CS_Options", "Effects", 1);
    cs_options.transitions = get_config_int("CS_Options", "Transitions", 1);
    cs_options.joystick    = get_config_int("CS_Options", "Joystick", 1);

    cs_options.sound_volume  = get_config_int("CS_Sound", "Effects", 100);
    cs_options.music_volume  = get_config_int("CS_Sound", "Music",   100);
    cs_options.sound_quality = get_config_int("CS_Sound", "Quality", 4);
    cs_options.stereo        = get_config_int("CS_Sound", "Stereo",  1);

    cs_options.vsync = get_config_int("CS_Video", "VSync", 1);
    cs_screen_mode   = get_config_int("CS_Video", "Mode", CS_FULL);

    /* read custom controls */
    cs_controls[0].left  = get_config_int("CS_Player1Keys", "Left",  KEY_A);
    cs_controls[0].right = get_config_int("CS_Player1Keys", "Right", KEY_D);
    cs_controls[0].drop =  get_config_int("CS_Player1Keys", "Drop",  KEY_S);
    cs_controls[0].rup =   get_config_int("CS_Player1Keys", "RUp",   KEY_W);
    cs_controls[0].rdown = get_config_int("CS_Player1Keys", "RDown", KEY_X);
    cs_controls[1].left  = get_config_int("CS_Player2Keys", "Left",  KEY_J);
    cs_controls[1].right = get_config_int("CS_Player2Keys", "Right", KEY_L);
    cs_controls[1].drop =  get_config_int("CS_Player2Keys", "Drop",  KEY_K);
    cs_controls[1].rup =   get_config_int("CS_Player2Keys", "RUp",   KEY_I);
    cs_controls[1].rdown = get_config_int("CS_Player2Keys", "RDown",
        KEY_COMMA);

    /* read current theme name */
    snprintf(cs_options.theme_name, 256, "%s",
        get_config_string("CS_Options", "Theme", "cs.cth"));
}

/* save config file */
void cs_save_config(void)
{
    /* write options */
    set_config_int("CS_Options", "Wildcard",    cs_options.wild_card);
    set_config_int("CS_Options", "Bomb",        cs_options.bomb);
    set_config_int("CS_Options", "Effects",     cs_options.effects);
    set_config_int("CS_Options", "Transitions", cs_options.transitions);
    set_config_int("CS_Options", "Joystick",    cs_options.joystick);

    set_config_int("CS_Sound", "Effects", cs_options.sound_volume);
    set_config_int("CS_Sound", "Music",   cs_options.music_volume);
    set_config_int("CS_Sound", "Quality", cs_options.sound_quality);
    set_config_int("CS_Sound", "Stereo",  cs_options.stereo);

    set_config_int("CS_Video", "VSync", cs_options.vsync);
    set_config_int("CS_Video", "Mode",  cs_screen_mode);

    /* write custom controls */
    set_config_int("CS_Player1Keys", "Left",  cs_controls[0].left );
    set_config_int("CS_Player1Keys", "Right", cs_controls[0].right);
    set_config_int("CS_Player1Keys", "Drop",  cs_controls[0].drop);
    set_config_int("CS_Player1Keys", "RUp",   cs_controls[0].rup);
    set_config_int("CS_Player1Keys", "RDown", cs_controls[0].rdown);
    set_config_int("CS_Player2Keys", "Left",  cs_controls[1].left );
    set_config_int("CS_Player2Keys", "Right", cs_controls[1].right);
    set_config_int("CS_Player2Keys", "Drop",  cs_controls[1].drop);
    set_config_int("CS_Player2Keys", "RUp",   cs_controls[1].rup);
    set_config_int("CS_Player2Keys", "RDown", cs_controls[1].rdown);

    /* write current theme name */
    set_config_string("CS_Options", "Theme", cs_options.theme_name);
}

/* install this on an interrupt to make the stack drop */
void core_drop_stack0(void)
{
    /* only drop when these conditions are met */
//    if(cs_game[0].state != STATE_GAMEOVER && cs_game[0].state != STATE_WIN && !cs_game[0].runs_left)
//    {
//        core_drop_stack(&cs_game[0], &cs_theme);
//    }
    if(cs_game[0].state != STATE_GAMEOVER && cs_game[0].state != STATE_WIN && !cs_game[0].runs_left)
    {
        drop_amount[0]++;
    }
}
END_OF_FUNCTION(core_drop_stack0);

/* install this on an interrupt to make the stack drop */
void core_drop_stack1(void)
{
    /* only drop when these conditions are met */
//    if(cs_game[1].state != STATE_GAMEOVER && cs_game[1].state != STATE_WIN && !cs_game[1].runs_left)
//    {
//        core_drop_stack(&cs_game[1], &cs_theme);
//    }
    if(cs_game[1].state != STATE_GAMEOVER && cs_game[1].state != STATE_WIN && !cs_game[1].runs_left)
    {
        drop_amount[1]++;
    }
}
END_OF_FUNCTION(core_drop_stack1);

/* get system ready to play game */
void cs_init(void)
{
    /* initialize Allegro stuff */
    allegro_init();
    set_window_title("Crystal Stacker");
    set_close_button_callback(NULL);

    /* make sure game is installed properly */
    if(!exists(DATADIR "cs.dat"))
    {
        allegro_message("Game not installed correctly!");
        exit(0);
    }

    /* install the rest of the handlers */
    install_timer();
    install_keyboard();
    install_mouse();

    /* fix some things */
    three_finger_flag = FALSE;
//    text_mode(-1);

    /* load the config file */
    cs_load_config();

    /* install the joystick if the option is set to on */
    if(cs_options.joystick)
    {
        ncd_joy_install();
    }

    /* initialize sound system */
    if(!nosound)
    {
        t3ss_init(sound_quality_chart[(int)cs_options.sound_quality], cs_options.stereo);
        t3ss_set_effects_channels(4);
        t3ss_set_reserved_channels(2);
        t3ss_set_music_volume(cs_options.music_volume);
        t3ss_set_sound_volume(cs_options.sound_volume);

        /* in Windows, the panning is always right AFAIK */
        #ifdef ALLEGRO_WINDOWS
        cs_panning[0][0] = 90;
        cs_panning[0][1] = 166;
        cs_panning[1][0] = 90;
        cs_panning[1][1] = 166;
        cs_panning[2][0] = 90;
        cs_panning[2][1] = 166;

        /* in DOS, we have normal and reverse stereo */
        #else
        cs_panning[0][0] = 128;
        cs_panning[0][1] = 128;
        cs_panning[1][0] = 90;
        cs_panning[1][1] = 166;
        cs_panning[2][0] = 166;
        cs_panning[2][1] = 90;
        #endif
    }

    /* load the data */
    cs_load_data();

    /* load the high scores */
    if(!cs_load_scores(&cs_scores, cs_get_config_file_name("hs")))
    {
        cs_reset_scores();
    }

    /* initialize effects system */
    gef_init(320, 240);

    /* randomize the game */
    ncd_randomize();

    /* lock some stuff */
    LOCK_VARIABLE(drop_amount);
    LOCK_FUNCTION(core_drop_stack0);
    LOCK_FUNCTION(core_drop_stack1);
}

/* get ready to go back to OS */
void cs_exit(void)
{
    /* free the data */
    cs_free_data();

    /* save config file */
    cs_save_config();

    /* unload scores */
    cs_destroy_scores(&cs_scores);

    /* remove sound system */
    t3ss_exit();

    /* destroy the screen buffer */
    destroy_bitmap(screen_buffer);

    /* deinit allegro stuff */
    gef_exit();
    ncd_joy_remove();
    remove_mouse();
    remove_keyboard();
    remove_timer();
    set_gfx_mode(GFX_TEXT, 80, 25, 80, 25);
    allegro_exit();
}

/* play the game in the specified mode */
void cs_play(int mode)
{
    /* load the theme (exit and show error message if problem occurs */
    if(!cs_load_theme(&cs_theme, cs_options.theme_name))
    {
        allegro_message("Error loading theme!");
        exit(0);
    }

    /* see which game mode to play and play */
    if(mode == CS_1NS)
    {
        play_1p_classic();
    }
    else if(mode == CS_1TD)
    {
        play_1p_destroyer();
    }
    else if(mode == CS_1TF)
    {
        play_1p_frenzy();
    }
    else if(mode == CS_2NS)
    {
        play_2pn_top_score();
    }
    else if(mode == CS_2NL)
    {
        play_2pn_last_out();
    }
    else if(mode == CS_2ND)
    {
        play_2pn_destroyer();
    }
    else if(mode == CS_2BS)
    {
        play_2pb_top_score();
    }
    else if(mode == CS_2BL)
    {
        play_2pb_last_out();
    }
    else if(mode == CS_2BD)
    {
        play_2pb_destroyer();
    }

    /* unload the theme */
    cs_destroy_theme(&cs_theme);

    /* if player was playing a 1 player mode, go to high score screen */
    if(cs_game[0].mode < CS_2NS)
    {
        high_scores(screen_buffer, &cs_game[0], &cs_menu_font, &cs_scores);
    }

    /* otherwise go back to main menu */
    else
    {
        cs_place = 0;
        reset_menus();
    }
}

/* check command line arguments */
void cs_check_args(int argc, char * argv[])
{
    if(argc > 1)
    {
        /* see if user is requesting a version number */
        if(!stricmp(argv[1], "-v") || !stricmp(argv[1], "-V"))
        {
            allegro_init();
            set_window_title("Crystal Stacker");
            allegro_message(VERSION);
            allegro_exit();

            exit(1);
        }

        /* check for -nosound argument */
        if(!stricmp(argv[1], "-nosound") || !stricmp(argv[1], "-NOSOUND") || !stricmp(argv[1], "-Nosound") || !stricmp(argv[1], "-NoSound"))
        {
            nosound = 1;
        }
    }
}

/* the main function */
int main(int argc, char * argv[])
{
    /* check command line arguments */
    cs_check_args(argc, argv);

    /* initialize */
    cs_init();

    /* get the theme filenames */
    update_theme_filenames(cs_options.theme_name);

    /* show the T^3 logo */
    t3_logo();

    /* go to the menu */
    cs_main_menu();

    /* shut everything down */
    cs_exit();

    return 0;
}
END_OF_MAIN()
