/* the core game engine stuff */

#include "includes.h"
#include "externs.h"

#define WILD_PROB   80
#define TRIPLE_PROB 60
#define DOUBLE_PROB 75
#define BOMB_PROB   60

/* chart used to generate a new stack */
char type_chart[5] = {1, 2, 3, 4, 5};
char chart_size = 5;

/* deletes block from specified array, moving all blocks downward */
void core_delete_block(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy, int boardwidth, int boardheight)
{
    int i;

    /* generate effect block */
    if(gp->board[boardy * boardwidth + boardx] < NUM_BLOCK_TYPES)
    {
        generate_effect_block(tp, &cs_effects_blocks, tp->board_x[gp->player] + boardx * tp->block_w, tp->board_y[gp->player] + boardy * tp->block_h - 3 * tp->block_w, boardx, boardy, gp->board[boardy * boardwidth + boardx]);
    }
    else
    {
        generate_effect_block(tp, &cs_effects_blocks, tp->board_x[gp->player] + boardx * tp->block_w, tp->board_y[gp->player] + boardy * tp->block_h - 3 * tp->block_w, boardx, boardy, (gp->wild_count >> WILD_SHIFT) + 1);
    }

    /* delete crystal and move above crystals down */
    for(i = boardy; i > 0; i--)
    {
        gp->board[i * boardwidth + boardx] = gp->board[(i - 1) * boardwidth + boardx];
    }
    gp->board[boardx] = 0;
}

//void core_replace_block(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy, int Block)
//{
//    gp->board[boardy * tp->board_w + boardx] = Block;
//}

/* insert a row of solid blocks on board */
void core_insert_solid_row(CS_GAME * gp)
{
    int i, j;

    /* move blocks up to make room for solid blocks */
    for(i = 0; i < cs_theme.board_h + STACK_HEIGHT - 1; i++)
    {
        for(j = 0; j < cs_theme.board_w; j++)
        {
            gp->board[i * cs_theme.board_w + j] = gp->board[(i + 1) * cs_theme.board_w + j];
            gp->runs[i * cs_theme.board_w + j] = gp->runs[(i + 1) * cs_theme.board_w + j];
        }
    }

    /* make bottom row solid */
    for(i = 0; i < cs_theme.board_w; i++)
    {
        gp->board[(cs_theme.board_h + 2) * cs_theme.board_w + i] = 6;
        gp->runs[(cs_theme.board_h + 2) * cs_theme.board_w + i] = 0;
    }

    /* move dropping blocks up */
    if(gp->stack.dy > 0)
    {
        gp->stack.dy--;
        gp->stack.y = gp->stack.dy;
    }
}

//int core_test_block(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy)
//{
//    return gp->board[boardy * tp->board_w + boardx];
//}

/* marks run to the right */
void core_mark_runs_right(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy, int runlength)
{
    int i;

    for(i = boardx; i < boardx + runlength; i++)
    {
        gp->runs[boardy * tp->board_w + i] = 1;
    }

    /* when this is called, we know there are runs to erase */
    gp->runs_left = 1;
}

/* finds runs that go right */
int core_find_runs_right(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy)
{
    int i;
    char RunLength = 0;
    int startx = boardx;
    int starty = boardy;

    /* while we are on wild cards, increase the run length */
    while(gp->board[starty * tp->board_w + startx] == NUM_BLOCK_TYPES && startx < tp->board_w)
    {
        RunLength++;
        startx++;
    }

    /* now we take the first non-wild color and use that as the match color */
    for(i = startx; i < tp->board_w && ((gp->board[starty * tp->board_w + i] == gp->board[starty * tp->board_w + startx]) || (gp->board[starty * tp->board_w + i] == NUM_BLOCK_TYPES)) && gp->board[starty * tp->board_w + i] != 0 && gp->board[starty * tp->board_w + i] != NUM_BLOCK_TYPES - 1; i++)
    {
        RunLength++;
    }

    /* if our run is long enough, mark the runs for destruction */
    if(RunLength >= RUN_LENGTH)
    {
        core_mark_runs_right(gp, tp, boardx, boardy, RunLength);
    }

    return RunLength;
}

/* marks run that goes down */
void core_mark_runs_down(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy, int runlength)
{
    int i;

    for(i = boardy; i < boardy + runlength; i++)
    {
        gp->runs[i * tp->board_w + boardx] = 1;
    }

    /* when this is called, we know there are runs to erase */
    gp->runs_left = 1;
}

/* finds run that goes down */
int core_find_runs_down(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy)
{
    int i;
    char RunLength = 0;
    int startx = boardx;
    int starty = boardy;

    /* while we are on wild cards, increase the run length */
    while(gp->board[starty * tp->board_w + startx] == NUM_BLOCK_TYPES && starty < tp->board_h + STACK_HEIGHT)
    {
        RunLength++;
        starty++;
    }

    /* now we take the first non-wild color and use that as the match color */
    for(i = starty; i < tp->board_h + STACK_HEIGHT && ((gp->board[i * tp->board_w + startx] == gp->board[starty * tp->board_w + startx]) || (gp->board[i * tp->board_w + startx] == NUM_BLOCK_TYPES)) && gp->board[i * tp->board_w + startx] != 0 && gp->board[i * tp->board_w + startx] != NUM_BLOCK_TYPES - 1; i++)
    {
        RunLength++;
    }

    /* if our run is long enough, mark the runs for destruction */
    if(RunLength >= RUN_LENGTH)
    {
        core_mark_runs_down(gp, tp, boardx, boardy, RunLength);
    }

    return RunLength;
}

/* marks run that goes up-right */
void core_mark_runs_up_right(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy, int runlength)
{
    int i;

    for(i = 0; i < runlength; i++)
    {
        gp->runs[(boardy - i) * tp->board_w + (boardx + i)] = 1;
    }

    /* when this is called, we know there are runs to erase */
    gp->runs_left = 1;
}

/* finds run that goes up-right */
int core_find_runs_up_right(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy)
{
    int i;
    char RunLength = 0;
    int startx = boardx;
    int starty = boardy;

    /* while we are on wild cards, increase the run length */
    while(gp->board[starty * tp->board_w + startx] == NUM_BLOCK_TYPES && starty >= 0 && startx < tp->board_w)
    {
        RunLength++;
        startx++;
        starty--;
    }

    /* now we take the first non-wild color and use that as the match color */
    for(i = 0; i + startx < tp->board_w && starty - i >= 0 && ((gp->board[(starty - i) * tp->board_w + startx + i] == gp->board[starty * tp->board_w + startx]) || (gp->board[(starty - i) * tp->board_w + startx + i] == NUM_BLOCK_TYPES)) && gp->board[(starty - i) * tp->board_w + startx + i] != 0 && gp->board[(starty - i) * tp->board_w + startx + i] != NUM_BLOCK_TYPES - 1; i++)
    {
        RunLength++;
    }

    /* if our run is long enough, mark the runs for destruction */
    if(RunLength >= RUN_LENGTH)
    {
        core_mark_runs_up_right(gp, tp, boardx, boardy, RunLength);
    }

    return RunLength;
}

/* marks run that goes down-right */
void core_mark_runs_down_right(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy, int runlength)
{
    int i;

    for(i = 0; i < runlength; i++)
    {
        gp->runs[(boardy + i) * tp->board_w + (boardx + i)] = 1;
    }

    /* when this is called, we know there are runs to erase */
    gp->runs_left = 1;
}

/* finds run that goes down-right */
int core_find_runs_down_right(CS_GAME * gp, CS_THEME * tp, int boardx, int boardy)
{
    int i;
    char RunLength = 0;
    int startx = boardx;
    int starty = boardy;

    /* while we are on wild cards, increase the run length */
    while(gp->board[starty * tp->board_w + startx] == NUM_BLOCK_TYPES && startx < tp->board_w && starty < tp->board_h + STACK_HEIGHT)
    {
        RunLength++;
        startx++;
        starty++;
    }

    /* now we take the first non-wild color and use that as the match color */
    for(i = 0; i + startx < tp->board_w && i + starty < tp->board_h + STACK_HEIGHT && ((gp->board[(starty + i) * tp->board_w + (startx + i)] == gp->board[starty * tp->board_w + startx]) || (gp->board[(starty + i) * tp->board_w + (startx + i)] == NUM_BLOCK_TYPES)) && gp->board[(starty + i) * tp->board_w + (startx + i)] != 0 && gp->board[(starty + i) * tp->board_w + (startx + i)] != NUM_BLOCK_TYPES - 1; i++)
    {
        RunLength++;
    }

    /* if our run is long enough, mark the runs for destruction */
    if(RunLength >= RUN_LENGTH)
    {
        core_mark_runs_down_right(gp, tp, boardx, boardy, RunLength);
    }

    return RunLength;
}

/* marks bombed blocks */
void core_mark_runs_bomb(CS_GAME * gp, CS_THEME * tp, int x, int y)
{
    int bombed;
    int i;

    /* see which color we are bombing */
    bombed = gp->board[(y + 1) * tp->board_w + x];

    /* make bomb change to color of bombed block (apparently) */
    for(i = 0; i < STACK_HEIGHT; i++)
    {
        gp->board[(gp->stack.y + i) * tp->board_w + gp->stack.x] = bombed;
    }

    /* do this when bombing a wild! */
    if(bombed == NUM_BLOCK_TYPES)
    {
        queue_message(&gp->messages, 2, "            Mass Destruction!");
        /* scan board for blocks to bomb */
        for(i = 0; i < tp->board_w * (tp->board_h + STACK_HEIGHT); i++)
        {
            if(gp->board[i])
            {
                gp->runs[i] = 1;
                gp->runs_left = 1;
            }
        }
    }

    /* do this when bombing a solid */
    else if(bombed == 6)
    {
        queue_message(&gp->messages, 2, "            Can't destroy those!");
        for(i = 0; i < STACK_HEIGHT; i++)
        {
            gp->board[(gp->stack.y + i) * tp->board_w + gp->stack.x] = ncd_random() % 5 + 1;
        }
    }

    /* otherwise, do this */
    else
    {
        queue_message(&gp->messages, 2, "            Destruction!");
        /* scan board for blocks to bomb */
        for(i = 0; i < tp->board_w * (tp->board_h + STACK_HEIGHT); i++)
        {
            if(gp->board[i] == bombed)
            {
                gp->runs[i] = 1;
                gp->runs_left = 1;
            }
        }
    }
}

/* drops specified stack 1 unit down */
void core_drop_stack(CS_GAME * gp, CS_THEME * tp)
{
    int i;

    /* see if stack is landing */
    if(gp->stack.dactive && (gp->board[(gp->stack.y + STACK_HEIGHT) * tp->board_w + gp->stack.x] || gp->stack.y >= tp->board_h))
    {
        t3ss_play_sound(tp->sound[SOUND_LAND], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);

        /* if stack is a bomb do this */
        if(gp->stack.type[2] == NUM_BLOCK_TYPES + 1)
        {
            if(gp->stack.y < tp->board_h)
            {
                core_mark_runs_bomb(gp, tp, gp->stack.x, gp->stack.y + 2);
            }
            else
            {
                queue_message(&gp->messages, 2, "            You really hit bottom.");
                for(i = 0; i < STACK_HEIGHT; i++)
                {
                    gp->board[(gp->stack.y + i) * tp->board_w + gp->stack.x] = ncd_random() % 5 + 1;
                }
            }
        }

        /* otherwise do this */
        else
        {
            for(i = 0; i < STACK_HEIGHT; i++)
            {
                gp->board[(gp->stack.y + i) * tp->board_w + gp->stack.x] = gp->stack.type[i];
            }
        }

        /* check for runs */
        core_find_runs(gp, tp);

        /* disable drop until user releases drop button */
        gp->control.was_drop = 1;

        /* reset drop amount */
        drop_amount[gp->player] = 0;

        /* generate a new stack for next time */
        core_copy_stack(gp);
        core_generate_stack(gp, tp);
    }
    else
    {
        t3ss_play_sound(tp->sound[SOUND_DROP], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);
        gp->stack.y++;
    }
}

/* copies the next stack to the current one */
void core_copy_stack(CS_GAME * gp)
{
    int i;

    gp->stack.x = gp->next_stack.x;
    gp->stack.y = gp->next_stack.y;

    for(i = 0; i < STACK_HEIGHT; i++)
    {
        gp->stack.type[i] = gp->next_stack.type[i];
    }
}

/* removes a number from the chart (helps make non-repeating stacks) */
void core_remove_from_chart(char Number)
{
    int i;

    for(i = 0; i < 5; i++)
    {
        if(type_chart[i] == Number)
        {
            chart_size--;
            break;
        }
    }
    for(i = i; i < 4; i++)
    {
        type_chart[i] = type_chart[i + 1];
    }
}

/* resets the char used to generate stacks */
void core_reset_chart(void)
{
    type_chart[0] = 1;
    type_chart[1] = 2;
    type_chart[2] = 3;
    type_chart[3] = 4;
    type_chart[4] = 5;
    chart_size = 5;
}

/* generates stack to drop */
void core_generate_stack(CS_GAME * gp, CS_THEME * tp)
{
    int i;
    char CurrentType;

    core_reset_chart();

    gp->next_stack.x = tp->board_w / 2;
    gp->next_stack.y = 0;

    /* generate a triple every once in a while */
    if(ncd_random() % TRIPLE_PROB == 0)
    {
        /* triple wild rarely */
        if(cs_options.wild_card)
        {
            if(ncd_random() % WILD_PROB == 0)
            {                                
                gp->next_stack.type[0] = NUM_BLOCK_TYPES;
            }
            else
            {
                gp->next_stack.type[0] = ncd_random() % (NUM_BLOCK_TYPES - 2) + 1;
            }
            gp->next_stack.type[1] = gp->next_stack.type[0];
            gp->next_stack.type[2] = gp->next_stack.type[0];
        }

        /* normally chooses a color */
        else
        {
            gp->next_stack.type[0] = ncd_random() % (NUM_BLOCK_TYPES - 2) + 1;
            gp->next_stack.type[1] = gp->next_stack.type[0];
            gp->next_stack.type[2] = gp->next_stack.type[0];
        }
    }

    /* generate a bomb every once in a while */
    else if((ncd_random() % BOMB_PROB == 0 && cs_options.bomb))
    {
        queue_message(&gp->messages, 2, "            Here comes a BOMB!");
        gp->next_stack.type[0] = NUM_BLOCK_TYPES + 1;
        gp->next_stack.type[1] = NUM_BLOCK_TYPES + 1;
        gp->next_stack.type[2] = NUM_BLOCK_TYPES + 1;
    }

    /* otherwise generate a normal stack */
    else
    {
        for(i = 0; i < STACK_HEIGHT; i++)
        {
            /* wild cards every once in a while */
            if(ncd_random() % WILD_PROB == 0 && cs_options.wild_card)
            {
                gp->next_stack.type[i] = NUM_BLOCK_TYPES;
            }

            /* regular crystals most of the time */
            else
            {
                CurrentType = type_chart[ncd_random() % chart_size];

                /* allow more than one of the same color sometimes */
                if(ncd_random() % DOUBLE_PROB != 0)
                {
                    core_remove_from_chart(CurrentType);
                }
                gp->next_stack.type[i] = CurrentType;
            }
        }
    }
}

/* deletes all marked blocks */
int core_delete_runs(CS_GAME * gp, CS_THEME * tp)
{
    int i, j;
    int TotalRemoved = 0;

    /* search entire board */
    for(i = 0; i < tp->board_h + STACK_HEIGHT; i++)
    {
        for(j = 0; j < tp->board_w; j++)
        {
            if(gp->runs[i * tp->board_w + j])
            {
                core_delete_block(gp, tp, j, i, tp->board_w, tp->board_h);
                TotalRemoved++;
                gp->runs[i * tp->board_w + j] = 0;
            }
        }
    }

    return TotalRemoved;
}

/* search for runs */
void core_find_runs(CS_GAME * gp, CS_THEME * tp)
{
    int i, j;

    /* set runs_left off (the procedures called below will set it properly */
    gp->runs_left = 0;

    /* search the board */
    for(i = 0; i < tp->board_h + STACK_HEIGHT; i++)
    {
        for(j = 0; j < tp->board_w; j++)
        {
            /* look in every possible direction */
            core_find_runs_right(gp, tp, j, i);
            core_find_runs_down(gp, tp, j, i);
            core_find_runs_up_right(gp, tp, j, i);
            core_find_runs_down_right(gp, tp, j, i);
        }
    }
}

/* handles runs in normal mode */
void core_handle_runs_normal(CS_GAME * gp, CS_THEME * tp)
{
    /* play the sound on the first frame */
    if(gp->runs_left && gp->counter == 0)
    {
        t3ss_play_sound(tp->sound[SOUND_RUN], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);
    }

    /* update board graphics (alternate between normal and flash mode) */
    gp->counter++;
    if((gp->counter / 3) % 2)
    {
        draw_board_runs(gp, tp);
        draw_preview_current(gp);
    }
    else
    {
        draw_board(gp);
        draw_preview_current(gp);
    }

    /* do this after flashing is done */
    if(gp->counter >= 18)
    {
        t3ss_play_sound(tp->sound[SOUND_DELETE], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);

        /* delete runs */
        gp->removed += core_delete_runs(gp, tp);

        /* tally blocks removed */
        gp->total_removed += gp->removed;

        /* check for level increase */
        core_update_level(gp, tp);

        /* decrease blocks left */
        gp->blocks_left -= gp->removed;

        /* increase block counter */
        gp->blocks += gp->removed;

        /* don't let blocks left go negative */
        if(gp->blocks_left < 0)
        {
            gp->blocks_left = 0;
        }

        /* increase the combo size */
        gp->power++;

        /* reset the counter */
        gp->counter = 0;

        /* look for more runs */
        core_find_runs(gp, tp);

        /* if there are none, do this */
        if(!gp->runs_left)
        {
            /* calculate new score */
            gp->score += gp->power * (((gp->removed * 5) * ((gp->level * 100) / 4 + 100)) / 100);

            /* don't let score go over 99999 (don't think this can happen) */
            if(gp->score > 99999)
            {
                gp->score = 99999;
            }

            /* do combo messages */
            if(gp->power > 3)
            {
                queue_message(&gp->messages, 2, "            Monster %dX combo", gp->power);
            }
            else if(gp->power > 1)
            {
                queue_message(&gp->messages, 2, "            %dX combo", gp->power);
            }

            /* set if player wins */
            core_handle_win(gp, tp);

            /* reactivate the stack */
            gp->stack.active = 1;
            if(gp->player == 0)
            {
                install_int(core_drop_stack0, gp->drop_time);
            }
            if(gp->player == 1)
            {
                install_int(core_drop_stack1, gp->drop_time);
            }
        }
    }
}

/* handles runs in battle mode */
void core_handle_runs_battle(CS_GAME * gp1, CS_GAME * gp2, CS_THEME * tp)
{
    /* play the sound on the first frame */
    if(gp1->runs_left && gp1->counter == 0)
    {
        t3ss_play_sound(tp->sound[SOUND_RUN], cs_panning[(int)cs_options.stereo][gp1->player], I_NULL);
    }

    /* update board graphics (alternate between normal and flash mode) */
    gp1->counter++;
    if((gp1->counter / 3) % 2)
    {
        draw_board_runs(gp1, tp);
        draw_preview_current(gp1);
    }
    else
    {
        draw_board(gp1);
        draw_preview_current(gp1);
    }

    /* do this after flashing is done */
    if(gp1->counter >= 18)
    {
        t3ss_play_sound(tp->sound[SOUND_DELETE], cs_panning[(int)cs_options.stereo][gp1->player], I_NULL);

        /* delete the runs */
        gp1->removed += core_delete_runs(gp1, tp);

        /* tally up blocks removed */
        gp1->total_removed += gp1->removed;

        /* check for level increase */
        core_update_level(gp1, tp);

        /* decrease the blocks left */
        gp1->blocks_left -= gp1->removed;

        /* increase the block counter */
        gp1->blocks += gp1->removed;

        /* make sure we don't go negative in blocks left */
        if(gp1->blocks_left < 0)
        {
            gp1->blocks_left = 0;
        }

        /* increase the size of the combo */
        gp1->power++;

        /* restart the counter */
        gp1->counter = 0;

        /* search for runs */
        core_find_runs(gp1, tp);

        /* if no runs left, do this */
        if(!gp1->runs_left)
        {
            /* figure up new score */
            gp1->score += ((5 * (gp1->removed / 5 + 1) * gp1->removed + gp1->level * 3) ^ gp1->power);

            /* don't let score go over 99999 (don't think this can happen) */
            if(gp1->score > 99999)
            {
                gp1->score = 99999;
            }

            /* handle attacks */
            if(gp1->removed >= COMBO_LARGE && gp2->state != STATE_GAMEOVER)
            {
                level_3_attack(gp2);
                queue_message(&gp2->messages, 1, " Take that!");
            }
            else if(gp1->removed >= COMBO_MEDIUM && gp2->state != STATE_GAMEOVER)
            {
                level_2_attack(gp2);
                queue_message(&gp2->messages, 1, " Take that!");
            }
            else if(gp1->removed >= COMBO_SMALL && gp2->state != STATE_GAMEOVER)
            {
                level_1_attack(gp2);
                queue_message(&gp2->messages, 1, " Take that!");
            }

            /* display combo messages */
            if(gp1->power > 3)
            {
                queue_message(&gp1->messages, 2, "            Monster %dX combo", gp1->power);
            }
            else if(gp1->power > 1)
            {
                queue_message(&gp1->messages, 2, "            %dX combo", gp1->power);
            }

            /* see if player wins */
            core_handle_win(gp1, tp);

            /* reactivate the player stack */
            gp1->stack.active = 1;
            if(gp1->player == 0)
            {
                install_int(core_drop_stack0, gp1->drop_time);
            }
            if(gp1->player == 1)
            {
                install_int(core_drop_stack1, gp2->drop_time);
            }
        }
    }
}

/* does wild card animation */
void core_handle_wild_card(CS_GAME * gp)
{
    gp->wild_count++;
    if(gp->wild_count >= ((NUM_BLOCK_TYPES - 2) << WILD_SHIFT))
    {
        gp->wild_count = 0;
    }
}

/* does a step of the game in normal mode */
void core_game_step_normal(CS_GAME * gp, CS_THEME * tp)
{
    /* handle the player message list */
    handle_message_list(&gp->messages);
    if((gp->state == STATE_NORMAL) && !gp->runs_left)
    {
        core_movement(gp, tp);
        core_handle_game_over(gp, tp);
    }

    /* handle runs */
    core_find_runs(gp, tp);
    if(gp->runs_left)
    {
        if(gp->counter == 0)
        {
            gp->stack.active = 0;
        }
    }
    else
    {
        gp->power = 0;
        gp->removed = 0;
    }

    /* animate wild cards */
    core_handle_wild_card(gp);

    /* update some variables */
    gp->stack.dx = gp->stack.x;
    gp->stack.dy = gp->stack.y;
    gp->stack.dtype[0] = gp->stack.type[0];
    gp->stack.dtype[1] = gp->stack.type[1];
    gp->stack.dtype[2] = gp->stack.type[2];
    gp->stack.dactive = gp->stack.active;
    gp->next_stack.dtype[0] = gp->next_stack.type[0];
    gp->next_stack.dtype[1] = gp->next_stack.type[1];
    gp->next_stack.dtype[2] = gp->next_stack.type[2];
}

/* does a step of the game in battle mode */
void core_game_step_battle(CS_GAME * gp1, CS_GAME * gp2, CS_THEME * tp)
{
    /* handle the player message list */
    handle_message_list(&gp1->messages);

    /* handle movement */
    if((gp1->state == STATE_NORMAL) && !gp1->runs_left)
    {
        core_movement(gp1, tp);
        core_handle_game_over(gp1, tp);
    }

    /* handle runs */
    core_find_runs(gp1, tp);
    if(gp1->runs_left)
    {
        if(gp1->counter == 0)
        {
            gp1->stack.active = 0;
        }
    }
    else
    {
        gp1->power = 0;
        gp1->removed = 0;
    }

    /* animate the wild cards */
    core_handle_wild_card(gp1);

    /* update some variables */
    gp1->stack.dx = gp1->stack.x;
    gp1->stack.dy = gp1->stack.y;
    gp1->stack.dtype[0] = gp1->stack.type[0];
    gp1->stack.dtype[1] = gp1->stack.type[1];
    gp1->stack.dtype[2] = gp1->stack.type[2];
    gp1->stack.dactive = gp1->stack.active;
    gp1->next_stack.dtype[0] = gp1->next_stack.type[0];
    gp1->next_stack.dtype[1] = gp1->next_stack.type[1];
    gp1->next_stack.dtype[2] = gp1->next_stack.type[2];
}

/* updates the level based on blocks destroyed */
void core_update_level(CS_GAME * gp, CS_THEME * tp)
{
    if(gp->total_removed >= gp->level * BLOCKS_PER_LEVEL)
    {
        t3ss_play_sound(tp->sound[SOUND_LEVELUP], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);
        queue_message(&gp->messages, 1, "  Level Up  ");
        gp->level++;

        /* can't go past 20 */
        if(gp->level >= 20)
        {
            gp->level = 20;
        }

        /* increase the speed of the stack */
        gp->drop_time = level_chart[gp->level - 1];
    }
}

/* checks for game over situation */
void core_handle_game_over(CS_GAME * gp, CS_THEME * tp)
{
    int i, j;

    /* see if player topped out (look at top 3 rows for crystals) */
    for(i = 0; i < 3; i++)
    {
        for(j = 0; j < tp->board_w; j++)
        {
            /* if a space is occupied and there are no runs left... */
            if(gp->board[i * tp->board_w + j] && !gp->runs_left)
            {
                gp->state = STATE_GAMEOVER;
                erase_board(gp);

                /* do this in last out mode (2p) */
                if(gp->mode == CS_2NL || gp->mode == CS_2BL)
                {
                    if(gp->player == 0 && cs_game[1].state != STATE_GAMEOVER)
                    {
                        erase_board(&cs_game[1]);
                    }
                    else if(cs_game[0].state != STATE_GAMEOVER)
                    {
                        erase_board(&cs_game[0]);
                    }
                }
                queue_message(&gp->messages, 2, "            You topped out!");
                break;
            }
        }
    }
    if(gp->mode == CS_1TF && !time_left())
    {
        gp->state = STATE_GAMEOVER;
        erase_board(gp);
        gp->stack.active = 0;
        gp->stack.dactive = 0;
        queue_message(&cs_game[0].messages, 2, "            Time's Up!");
    }
}

/* checks a game for a win situation */
void core_handle_win(CS_GAME * gp, CS_THEME * tp)
{
    /* only these modes have a possible "win" situation */
    if(gp->mode == CS_1TD || gp->mode == CS_2ND || gp->mode == CS_2BD)
    {
        /* win when you've destroyed enough blocks */
        if(gp->blocks_left <= 0)
        {
            gp->state = STATE_WIN;
            erase_board(gp); // just a test

            /* in 2 players, make the other player game over */
            if(gp->mode == CS_2ND || gp->mode == CS_2BD)
            {
                if(gp->player == 0 && cs_game[1].state != STATE_GAMEOVER)
                {
                    erase_board(&cs_game[1]);
                }
                else if(cs_game[0].state != STATE_GAMEOVER)
                {
                    erase_board(&cs_game[0]);
                }
            }
        }
    }
}

/* moves the player's stack left */
void core_move_left(CS_GAME * gp, CS_THEME * tp)
{
    /* see if it can move */
    if(!gp->board[((gp->stack.y + STACK_HEIGHT - 1) * tp->board_w) + (gp->stack.x - 1)] && !gp->board[((gp->stack.y + 1) * tp->board_w) + (gp->stack.x - 1)] && !gp->board[((gp->stack.y + 2) * tp->board_w) + (gp->stack.x - 1)] && gp->stack.x > 0)
    {
        t3ss_play_sound(tp->sound[SOUND_MOVE], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);
        gp->stack.x--;
    }

    /* if not play the bumping wall sound */
    else
    {
        t3ss_play_sound(tp->sound[SOUND_HITWALL], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);
    }
}

/* moves the player's stack right */
void core_move_right(CS_GAME * gp, CS_THEME * tp)
{
    /* see if it can move */
    if(!gp->board[((gp->stack.y + STACK_HEIGHT - 2) * tp->board_w) + (gp->stack.x + 1)] && !gp->board[((gp->stack.y + 1) * tp->board_w) + (gp->stack.x + 1)] && !gp->board[((gp->stack.y + 2) * tp->board_w) + (gp->stack.x + 1)] && gp->stack.x < tp->board_w - 1)
    {
        t3ss_play_sound(tp->sound[SOUND_MOVE], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);
        gp->stack.x++;
    }

    /* if not play the bumping wall sound */
    else
    {
        t3ss_play_sound(tp->sound[SOUND_HITWALL], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);
    }
}

/* performs a rotate up on the player's stack */
void core_rotate_up(CS_GAME * gp)
{
    char temp;
    int i;

    /* remember the top crystal */
    temp = gp->stack.type[0];

    /* move the others up */
    for(i = 0; i < STACK_HEIGHT; i++)
    {
        gp->stack.type[i] = gp->stack.type[i + 1];
    }

    /* place remembered crystal at the bottom */
    gp->stack.type[STACK_HEIGHT - 1] = temp;
}

/* performs a rotate down on the player's stack */
void core_rotate_down(CS_GAME * gp)
{
    char temp;
    int i;

    /* remember the bottom crystal */
    temp = gp->stack.type[STACK_HEIGHT - 1];

    /* move the others down */
    for(i = STACK_HEIGHT - 1; i > 0; i--)
    {
        gp->stack.type[i] = gp->stack.type[i - 1];
    }

    /* place the remembered one at the top */
    gp->stack.type[0] = temp;
}

/* handles movement for a player */
void core_movement(CS_GAME * gp, CS_THEME * tp)
{
    int i;
    int td = drop_amount[gp->player];

    /* handle automatic dropping */
    for(i = 0; i < td; i++)
    {
        core_drop_stack(gp, tp);
    }
    drop_amount[gp->player] = 0;

    /* check for drop */
    if(gp->control.drop)
    {
        /* remove the interrupt handler */
        if(gp->player == 0)
        {
            remove_int(core_drop_stack0);
        }
        else if(gp->player == 1)
        {
            remove_int(core_drop_stack1);
        }

        /* drop the stack one position downward */
        core_drop_stack(gp, tp);

        /* restart the interrupt handler */
        if(gp->player == 0)
        {
            install_int(core_drop_stack0, gp->drop_time);
        }
        else if(gp->player == 1)
        {
            install_int(core_drop_stack1, gp->drop_time);
        }
    }

    /* check for left */
    if(gp->control.left && gp->stack.x > 0)
    {
        core_move_left(gp, tp);
    }

    /* check for right */
    if(gp->control.right && gp->stack.x < tp->board_w - 1)
    {
        core_move_right(gp, tp);
    }

    /* check for rotate up */
    if(gp->control.rotate_up)
    {
        t3ss_play_sound(tp->sound[SOUND_ROTATEUP], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);
        core_rotate_up(gp);
    }

    /* check for rotate down */
    if(gp->control.rotate_down)
    {
        t3ss_play_sound(tp->sound[SOUND_ROTATEDOWN], cs_panning[(int)cs_options.stereo][gp->player], I_NULL);
        core_rotate_down(gp);
    }
}
