#ifndef GUI_H

#include <allegro.h>

#define MENU_SPACE     2
#define FONT_WIDTH     8
#define FONT_HEIGHT    8
#define MAX_ITEMS     20
#define MAX_CHARS     30

#define TYPE_MENU      0
#define TYPE_IMAGE     1
#define TYPE_TEXT      2


/* structure for item on a menu */
typedef struct
{

        char name[MAX_CHARS]; /* this is what user sees on menu */
        int child;            /* indicates child window (-1 if has none) */
        void(*procedure)();   /* procedure that this item calls */

} MENU_ITEM;


/* structure that holds a menu window */
typedef struct
{

        int item_count; /* how many items are in the menu */
        int item;       /* which item is currently selected */
        int parent;     /* which item on the parent windows menu made this menu */

        MENU_ITEM items[MAX_ITEMS]; /* item array */

} MENU_WINDOW;


/* structure that holds an image window */
typedef struct
{

        char loaded;    /* set if image is loaded */
        BITMAP * image; /* pointer to image */

} IMAGE_WINDOW;


/* structure that holds a text window */
typedef struct
{

        int lines;
        char * string[MAX_ITEMS];

} TEXT_WINDOW;


/* structure that holds a window */
typedef struct
{
        unsigned char type;     /* what type of window this is */
        char title[MAX_CHARS];  /* window title */
        int x, y;               /* window coordinates */
        int w, h;               /* window dimensions */
        unsigned char parent;   /* indicates which window is its parent */
        unsigned char fg;       /* colors used to draw window */
        unsigned char bg;
        unsigned char hfg;
        unsigned char hbg;
        unsigned char mhfg;
        unsigned char mhbg;
        char selected;          /* set if this window is the current window */
        unsigned char active;   /* set if window is to be displayed */

        /* unite all the window types */
        union
        {
                MENU_WINDOW mw;
                IMAGE_WINDOW iw;
                TEXT_WINDOW tw;
        } dat;

} WINDOW;


/* functions to initialize a window */
void create_window(WINDOW * wp, unsigned char type, char * title);
void set_window_position(WINDOW * wp, int x, int y);
void set_window_color(WINDOW * wp, int bg, int fg, int hbg, int hfg, int mhbg, int mhfg);
void set_window_parent(WINDOW * wp, int parent);
void select_window(WINDOW * wp);
void deselect_window(WINDOW * wp);
void activate_window(WINDOW * wp);
void deactivate_window(WINDOW * wp);
void destroy_window(WINDOW * wp);


/* menu window functions */
void add_menu_item(WINDOW * wp, char * name, int child, void(*procedure)());
void delete_menu_item(WINDOW * wp, int item);
void clear_menu(WINDOW * wp);


/* text window functions */
void add_text_item(WINDOW * wp, char * text);
void replace_text_item(WINDOW * wp, int line, char * text);
void clear_text_window(WINDOW * wp);


/* image window functions */
void import_image(WINDOW * wp, BITMAP * image);
void delete_image(WINDOW * wp);


/* miscellaneous window functions */
void drop_down(WINDOW * sw, WINDOW * dw);
void move_window(WINDOW * wp, void(*uproc)());
char title_click(WINDOW * wp);
char window_click(WINDOW * wp);
void mouse_select_item(WINDOW * wp);
void do_select_item(WINDOW * wp, void(*uproc)());


void draw_window(WINDOW * wp, BITMAP * bp);


/* zbuffer routines */
void create_zbuff(char * bp, unsigned char size);
void make_closest(char * bp, char window, unsigned char size);
unsigned char find_parent(char * bp, char parent, unsigned char size);
void destroy_zbuff(char * bp);

#define GUI_H
#endif
