#include <string.h>
#include "gui.h"


/* creates a new window */
void create_window(WINDOW * wp, unsigned char type, char * title)
{
    wp->type = type;
    strcpy(wp->title, title);
    wp->w = strlen(wp->title) * FONT_WIDTH + 2;
    wp->h = 10;
    wp->selected = 0;
    wp->active = 0;

    /* menu window only */
    if(type == TYPE_MENU)
    {
        wp->dat.mw.item_count = 0;
    }

    /* image window only */
    if(type == TYPE_IMAGE)
    {
        wp->dat.iw.loaded = 0;
    }
}


/* set window location */
void set_window_position(WINDOW * wp, int x, int y)
{
    wp->x = x;
    wp->y = y;
}


/* set window colors */
void set_window_color(WINDOW * wp, int bg, int fg, int hbg, int hfg, int mhbg, int mhfg)
{
    wp->bg = bg;
    wp->fg = fg;
    wp->hbg = hbg;
    wp->hfg = hfg;
    wp->mhbg = mhbg;
    wp->mhfg = mhfg;
}


/* set window parent */
void set_window_parent(WINDOW * wp, int parent)
{
    wp->parent = parent;
}


/* sets selection bit of wp */
void select_window(WINDOW * wp)
{
    wp->selected = 1;
}


/* unsets selection bit of wp */
void deselect_window(WINDOW * wp)
{
    wp->selected = 0;
}


/* activates window */
void activate_window(WINDOW * wp)
{
    wp->active = 1;
}


/* deactivates window */
void deactivate_window(WINDOW * wp)
{
    wp->active = 0;
}


/* frees a window from memory */
void destroy_window(WINDOW * wp)
{
    int i;

    /* destroy menu window */
    if(wp->type == TYPE_MENU)
    {
        wp->dat.mw.item_count = 0;
    }

    /* destroy image window */
    else if(wp->type == TYPE_IMAGE)
    {
        if(wp->dat.iw.loaded)
        {
            destroy_bitmap(wp->dat.iw.image);
            wp->dat.iw.loaded = 0;
        }
        deactivate_window(wp);
    }

    /* destroy text window */
    else if(wp->type == TYPE_TEXT)
    {
        for(i = 0; i < wp->dat.tw.lines; i++)
        {
            free(wp->dat.tw.string[i]);
        }
        wp->dat.tw.lines = 0;
        deactivate_window(wp);
    }
}


/* adds an item to a menu window */
void add_menu_item(WINDOW * wp, char * name, int child, void(*procedure)())
{
        int size;

        /* make sure window is a menu */
        if(wp->type == TYPE_MENU)
        {
                /* copy the name */
                strcpy(wp->dat.mw.items[wp->dat.mw.item_count].name, name);

                /* change width of window if needs be */
                size = strlen(wp->dat.mw.items[wp->dat.mw.item_count].name) * FONT_WIDTH + 2;
                if(size > wp->w)

                        wp->w = size;

                /* point to the specified procedure */
                wp->dat.mw.items[wp->dat.mw.item_count].procedure = procedure;

                wp->dat.mw.items[wp->dat.mw.item_count].child = child;
                wp->dat.mw.parent = -1;

                /* update the item count and increase window height */
                wp->dat.mw.item_count++;
                wp->h = wp->dat.mw.item_count * 10 + 11;
        }
}


/* removes an item from a menu */
void delete_menu_item(WINDOW * wp, int item)
{
        int i;

        /* make sure window is a menu and has an item */
        if(wp->type == TYPE_MENU && wp->dat.mw.item_count > 0)
        {

                /* if removing item other than last, move all item data as required */
                if(item < wp->dat.mw.item_count)
                {
                        for(i = item; i < wp->dat.mw.item_count; i++)
                        {
                                strcpy(wp->dat.mw.items[i].name, wp->dat.mw.items[i + 1].name);
                                wp->dat.mw.items[i].child = wp->dat.mw.items[i + 1].child;
                                wp->dat.mw.items[i].procedure = wp->dat.mw.items[i + 1].procedure;
                        }
                        wp->dat.mw.item_count--;
                }

                /* otherwise just remove the item */
                else

                        wp->dat.mw.item_count--;

                /* find new window height */
                wp->h = wp->dat.mw.item_count * 10 + 11;
        }
}


/* removes all items from a menu */
void clear_menu(WINDOW * wp)
{
        if(wp->type == TYPE_MENU)
        {
                wp->dat.mw.item_count = 0;
                wp->h = wp->dat.mw.item_count * 10 + 11;
        }
}


/* add a line of text to a text window */
void add_text_item(WINDOW * wp, char * text)
{
        int size;

        /* see if window is a text window */
        if(wp->type == TYPE_TEXT)
        {

                /* make sure we have room for more items */
                if(wp->dat.tw.lines < MAX_ITEMS)
                {

                        /* allocate and copy string */
                        wp->dat.tw.string[wp->dat.tw.lines] = malloc(strlen(text) + 1);
                        strcpy(wp->dat.tw.string[wp->dat.tw.lines], text);

                        /* get new window width */
                        size = strlen(wp->dat.tw.string[wp->dat.tw.lines]) * FONT_WIDTH + 2;
                        if(size > wp->w)

                                wp->w = size;

                        /* update line counter and get new window height */
                        wp->dat.tw.lines++;
                        wp->h = wp->dat.tw.lines * 10 + 11;
                }
        }
}


/* replaces a text line with a new one */
void replace_text_item(WINDOW * wp, int line, char * text)
{
        int size;

        if(wp->type == TYPE_TEXT && line <= wp->dat.tw.lines)
        {
                /* free old string and allocate and copy new one */
                free(wp->dat.tw.string[line]);
                wp->dat.tw.string[line] = malloc(strlen(text));
                strcpy(wp->dat.tw.string[line], text);

                /* get new window width */
                size = strlen(wp->dat.tw.string[wp->dat.tw.lines]) * FONT_WIDTH + 2;
                if(size > wp->w)

                        wp->w = size;
        }
}


/* clears a text window */
void clear_text_window(WINDOW * wp)
{
        int i;

        if(wp->type == TYPE_TEXT)
        {
                for(i = 0; i < wp->dat.tw.lines; i++)

                        free(wp->dat.tw.string[i]);

                wp->dat.tw.lines = 0;
        }
}


/* imports an image into an image window */
void import_image(WINDOW * wp, BITMAP * image)
{
        if(wp->type == TYPE_IMAGE)
        {
                if(wp->dat.iw.loaded == 1)

                        destroy_bitmap(wp->dat.iw.image);

                wp->w = image->w + 1;
                wp->h = image->h + 11;
                wp->dat.iw.image = image;
                wp->dat.iw.loaded = 1;
        }
}


/* frees an image window image */
void delete_image(WINDOW * wp)
{
        if(wp->dat.iw.loaded)
        {
                destroy_bitmap(wp->dat.iw.image);
                wp->dat.iw.loaded = 0;
        }
}


/* drops dw down from sw */
void drop_down(WINDOW * sw, WINDOW * dw)
{
        if(sw->type == TYPE_MENU)
        {
                dw->x = sw->x + sw->w + 1;
                dw->y = sw->y + sw->dat.mw.item * (MENU_SPACE + FONT_HEIGHT);
        }
}


/* moves window and calls uproc (should be used to update screen) */
void move_window(WINDOW * wp, void(*uproc)())
{
        int ax, ay;

        ax = mouse_x - wp->x;
        ay = mouse_y - wp->y;

        while(mouse_b & 1)
        {
                if(wp->type == TYPE_MENU)
                {
                        if(wp->dat.mw.parent > -1)

                                return;
                }
                set_window_position(wp, mouse_x - ax, mouse_y - ay);
                uproc();
        }
}


/* see if user is clicking the title bar */
char title_click(WINDOW * wp)
{
        if(mouse_b & 1 && mouse_x > wp->x && mouse_x < wp->x + wp->w && mouse_y > wp->y && mouse_y < wp->y + FONT_HEIGHT + MENU_SPACE)
        {
                return 1;
        }

        return 0;
}


/* see if user is clicking the window */
char window_click(WINDOW * wp)
{
        if(wp->active && (mouse_b & 1 || mouse_b & 2) && mouse_x > wp->x && mouse_x < wp->x + wp->w && mouse_y > wp->y && mouse_y < wp->y + wp->h)
        {
                return 1;
        }

        return 0;
}


/* selects an item on a menu with the mouse */
void mouse_select_item(WINDOW * wp)
{
        if(wp->active && wp->type == TYPE_MENU)

                if(mouse_b & 1 && mouse_x > wp->x && mouse_x < wp->x + wp->w && mouse_y > wp->y && mouse_y < wp->y + wp->h)

                        wp->dat.mw.item = (mouse_y - (wp->y + 11)) / 10;
}


/* initiates the selected item in the window */
void do_select_item(WINDOW * wp, void(*uproc)())
{
        if(wp->dat.mw.items[wp->dat.mw.item].procedure != NULL)
        {
                if(uproc != NULL)

                        uproc();

                wp->dat.mw.items[wp->dat.mw.item].procedure();
        }
}


/* draws a window onto the specified bitmap */
void draw_window(WINDOW * wp, BITMAP * bp)
{
    int i;

    /* draw window if it's active */
    if(wp->active)
    {

        /* draw menu */
        if(wp->type == TYPE_MENU)
        {
            if(wp->selected)
            {
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->hbg);
                rect(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->hfg);
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + MENU_SPACE + FONT_HEIGHT, wp->hfg);
                textout_ex(bp, font, wp->title, wp->x + MENU_SPACE, wp->y + MENU_SPACE, wp->hbg, -1);
            }
            else
            {
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->bg);
                rect(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->fg);
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + MENU_SPACE + FONT_HEIGHT, wp->fg);
                textout_ex(bp, font, wp->title, wp->x + MENU_SPACE, wp->y + MENU_SPACE, wp->bg, -1);
            }
            for(i = 0; i < wp->dat.mw.item_count; i++)
            {
                if(wp->dat.mw.item == i)
                {
                    rectfill(bp, wp->x + 1, wp->y + (i + 1) * (FONT_HEIGHT + MENU_SPACE) + 1, wp->x + wp->w - 1,  wp->y + (i + 1) * (FONT_HEIGHT + MENU_SPACE) + MENU_SPACE + FONT_HEIGHT, wp->mhbg);
                    textout_ex(bp, font, wp->dat.mw.items[i].name, wp->x + MENU_SPACE, wp->y + MENU_SPACE + (i + 1) * (FONT_HEIGHT + MENU_SPACE), wp->mhfg, -1);
                }
                else
                {
                    textout_ex(bp, font, wp->dat.mw.items[i].name, wp->x + MENU_SPACE, wp->y + MENU_SPACE + (i + 1) * (FONT_HEIGHT + MENU_SPACE), wp->fg, -1);
                }
            }
        }

        /* or image window */
        else if(wp->type == TYPE_IMAGE && wp->dat.iw.loaded)
        {
            if(wp->selected)
            {
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->hbg);
                rect(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->hfg);
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + MENU_SPACE + FONT_HEIGHT, wp->hfg);
                textout_ex(bp, font, wp->title, wp->x + MENU_SPACE, wp->y + MENU_SPACE, wp->hbg, -1);
            }
            else
            {
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->bg);
                rect(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->fg);
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + MENU_SPACE + FONT_HEIGHT, wp->fg);
                textout_ex(bp, font, wp->title, wp->x + MENU_SPACE, wp->y + MENU_SPACE, wp->bg, -1);
            }
            blit(wp->dat.iw.image, bp, 0, 0, wp->x + 1, wp->y + 11, bp->w, bp->h);
        }

        /* or text window */
        else if(wp->type == TYPE_TEXT)
        {
            if(wp->selected)
            {
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->hbg);
                rect(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->hfg);
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + MENU_SPACE + FONT_HEIGHT, wp->hfg);
                textout_ex(bp, font, wp->title, wp->x + MENU_SPACE, wp->y + MENU_SPACE, wp->hbg, -1);
            }
            else
            {
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->bg);
                rect(bp, wp->x, wp->y, wp->x + wp->w, wp->y + wp->h, wp->fg);
                rectfill(bp, wp->x, wp->y, wp->x + wp->w, wp->y + MENU_SPACE + FONT_HEIGHT, wp->fg);
                textout_ex(bp, font, wp->title, wp->x + MENU_SPACE, wp->y + MENU_SPACE, wp->bg, -1);
            }
            for(i = 0; i < wp->dat.tw.lines; i++)
            {
                textout_ex(bp, font, wp->dat.tw.string[i], wp->x + MENU_SPACE, wp->y + MENU_SPACE + (i + 1) * (FONT_HEIGHT + MENU_SPACE), wp->fg, -1);
            }
        }
    }
}


/* sets up a zbuffer */
void create_zbuff(char * bp, unsigned char size)
{
    unsigned char i;

    for(i = 0; i < size; i++)
    {
        bp[i] = i;
    }
}


/* make specified window the top window in the zbuffer */
void make_closest(char * bp, char window, unsigned char size)
{
    unsigned char i;
    char * TempBuffer;

    TempBuffer = malloc(sizeof(char) * size);
    for(i = 0; bp[i] != window; i++)
    {
        TempBuffer[i] = bp[i];
    }

    for(i = i + 1; i < size; i++)
    {
        TempBuffer[i - 1] = bp[i];
    }
    TempBuffer[size - 1] = window;

    for(i = 0; i < size; i++)
    {
        bp[i] = TempBuffer[i];
    }
    free(TempBuffer);
}


/* finds parent using zbuffer */
unsigned char find_parent(char * bp, char parent, unsigned char size)
{
    unsigned char i;

    for(i = 0; i < size; i++)
    {
        if(bp[i] == parent)
        {
            return i;
        }
    }

    return 0;
}


/* frees zbuffer from memory */
void destroy_zbuff(char * bp)
{
}
