/*   ______   ___    ___
    /\  _  \ /\_ \  /\_ \                             
    \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
     \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
      \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
       \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
        \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
                                       /\____/
    Allegro Low-Level Game Routines    \_/__/   Version 4.2.0
         
         Arena Football using Dev-Cpp 4.9.9.2  (c) 2006 by Gia T. Nguyen */

// *******************************************************************************
//
// This 2-D football game is based on the classic hand-held electronic football
// games back in the late 70's from Mattel and Coleco.
// 
// 1 or 2 Players under 3 player modes:
//     
//      1-Player Mode:   Player 1 plays offense versus computerized defense.
//      2-Player Modes:  (Turn-Based) Players 1 and 2 take turns playing offense against computerized defense.
//                       (Versus) Player 1 and 2 in head-to-head mode, Player 2 controls movements of Free Safety.
// 
// 2 levels of difficulty:
//     
//      Beginner:  Defensive backs move at same speed as blockers.
//      Pro:       Defensive backs move at faster speed.
//
// Player 1 - Home Team (Blue)   - 4 players
// Player 2 - Visitor Team (Red) - 6 players
//
// Run, pass, kick field goal, safety
//
// Move Ball Carrier:  Up, Down, Left, Right arrow keys
// Move Free Safety:   A (left), S (down), D (right), W (up)
// Pass                P
// Kick Field Goal:    K
// Quit:               Esc
// Toggle Play         F1
// Music On/Off:       F2
//
// Points:
//   Touch Down - 7
//   Field Goal - 3
//   Safety     - 2
//
// This C++ program compiles under Windows XP on a Pentium IV CPU
// using the Dev-Cpp IDE with MinGW32, and the Allegro game programming library.
// It utilizes 16-bit color, 640x480 screen resolution, and MIDI sample.
//
// Dependencies:
//     alleg42.dll, 
//     allegro.h
//     string.h
//     football.h,
//     fbdata.h,
//     football.dat
//     football.ico (optional)
//
// Last Updated:  April 12, 2006
//
// *******************************************************************************

#include <string>
#include "football.h"
#include "fbdata.h"

int main(int argc, char *argv[]) {
    
    int graphics = GFX_AUTODETECT_FULLSCREEN;
    
    if (argc == 2)
       if (strcmp(argv[1],"-w") >= 0 || strcmp(argv[1],"-W") >= 0)
          graphics = GFX_AUTODETECT_WINDOWED;
    
	init(graphics);
    while(play());
	deinit();
    return 0;
}
END_OF_MAIN()

void init(int graphics) {                                
	
    int res = allegro_init();
	set_color_depth(16);
	res = set_gfx_mode(graphics, 640, 480, 0, 0);
	res = install_timer();
	res = install_keyboard();
	res = install_mouse();
    res = install_sound(DIGI_AUTODETECT,MIDI_AUTODETECT,"");
    
    if (res != 0) {
       allegro_message("Gamelibinit errors occurred. Program aborted.");
       exit(-1);
    }

    if (!exists(DATA)) {
       allegro_message("Missing data file. Program aborted.");
       exit(-1);
    }                       
    
    dat = load_datafile(DATA);
    LOCK_FUNCTION(auto_play);            
    rndgen(0);
    init_game();

}

bool play() {                            

    int res = FREE;
       
    if (readkey() == KEY_PAUSE) goto new_down;

    if (key[KEY_ESC]) {
        if (start) remove_int(auto_play);
        return false;
    }    

    if (key[KEY_UP] || key[KEY_DOWN] || key[KEY_RIGHT] || key[KEY_LEFT]) {
       if (key[KEY_UP]) 
          res = offense[p1].move(MOVEUP,formation);
       else if (key[KEY_DOWN]) 
          res = offense[p1].move(MOVEDOWN,formation);
       else if (key[KEY_RIGHT]) 
          res = offense[p1].move(MOVERIGHT,formation);
       else if (key[KEY_LEFT]) {
          if (offense[p1].x > (offense[p2].x+BMP_SIZE+OUTBOUND_OFFSET))      
             res = offense[p1].move(MOVELEFT,formation);
       }
       if (res == DOWN) goto new_down;
       if (formation == PASS && pass_ret == FREE) {
          res = offense[p2].move(formation);
          if (res == DOWN) goto new_down;
       }
       if (num_players < VERSUS) {
          res = defense[p1].move(formation);
          if (res == DOWN) goto new_down;
       }   
       draw_field();
       if (!start) {
           install_int(auto_play,200);
           start = true;
       }
    }

    if (num_players == VERSUS) {
        if (key[KEY_A] || key[KEY_S] || key[KEY_D] || key[KEY_W]) {
           if (key[KEY_W]) 
              res = defense[p1].move(MOVEUP,formation);
           else if (key[KEY_S]) 
              res = defense[p1].move(MOVEDOWN,formation);
           else if (key[KEY_D]) 
              res = defense[p1].move(MOVERIGHT,formation);
           else if (key[KEY_A]) {
              if (formation == PASS) {
                 if (defense[p1].x > yards(scrimmage))
                    res = defense[p1].move(MOVELEFT,formation);
              } else
                    res = defense[p1].move(MOVELEFT,formation);      
           }   
           if (res == DOWN) goto new_down;
           draw_field();
        }
    }
    
    if (key[KEY_K])
      if (!start) kick_fieldgoal();
      
    if (key[KEY_P])
      if (start && formation == PASS) pass();      

    if (key[KEY_F1]) {
       if (!start) {              
           if (formation == RUN) 
              formation = PASS;
           else
              formation = RUN;
           reset_formation();
       }
    }
    
    if (key[KEY_F2]) {
       music_flag = !music_flag;                  
       play_music(music_flag);         
    }
     
    return true;
    
new_down:

    new_down();
    return true;

}

void deinit() {

     for (int i=p1;i<OPLAYERS;i++)
         for (int j=0;j<FRAME;j++)
             destroy_bitmap(offense[i].image[j]);
     
     for (int i=p1;i<DPLAYERS;i++)
         for (int j=0;j<FRAME;j++)
             destroy_bitmap(defense[i].image[j]);

     for (int i=0;i<FRAME;i++)
         destroy_bitmap(kicker.image[i]);
     
    unload_datafile(dat);
    destroy_bitmap(buffer);
	clear_keybuf();
    exit(0);
    
}

void init_game() {

     buffer = create_bitmap(2400,250);

     clear_to_color(screen,LIGHT_BLUE);
    
     rect(screen,14,74,622,334,YELLOW);
     rect(screen,15,75,623,335,YELLOW);
    
     load_fonts();
     load_bitmaps();
      
     for (int i=p1;i<OPLAYERS;i++) {
         offense[i].role = OFFENSE;
         offense[i].number = i;
         if (i > p1) offense[i].speed = BLOCKER;
     }   
     
     for (int i=p1;i<DPLAYERS;i++) {
         defense[i].role = DEFENSE;
         defense[i].number = i;
         if (i > p1 && i < p5) defense[i].speed = BLOCKER;
     }     
       
     offense[p1].speed = RUNNER;

     down_yard = yards(YRD20);
     run_formation(down_yard);
     
     draw_header();
     draw_instruction();
          
     clear_keybuf();
     select_mode(1);
     readkey();
     while (!key[KEY_1] && !key[KEY_2] && !key[KEY_3] && !key[KEY_ESC]);
     if (key[KEY_ESC]) deinit();
     if (key[KEY_2]) num_players = DOUBLE;
     if (key[KEY_3]) num_players = VERSUS;
     select_mode(0);
     clear_keybuf();
     
     int safety_speed = BLOCKER;
     
     if (num_players < VERSUS) {
         send_msg("[B]eginner or [P]ro?");
         readkey();
         while (!key[KEY_B] && !key[KEY_P] && !key[KEY_ESC]);
         if (key[KEY_ESC]) deinit();
         if (key[KEY_P]) safety_speed = SAFETY;
         send_msg();
         clear_keybuf();     
     }

     defense[p1].speed = safety_speed;
     defense[p5].speed = safety_speed;
     defense[p6].speed = safety_speed;
     
     draw_field();
     draw_labels();
     draw_scores();
               
     play_music(music_flag);     

}

void auto_play() {
   
    int p=p2;
    if (formation == PASS) p=p3;
    for (int i=p;i<OPLAYERS;i++)
      offense[i].move(formation);
    for (int j=p2;j<DPLAYERS;j++)
        if (defense[j].move(formation) == DOWN) simulate_keypress(KEY_PAUSE);
    draw_field();     

}
END_OF_FUNCTION(auto_play);

void new_down() {
     
     new_down(false);

}

void new_down(bool start_over) {

      down_yard = yards(yard);
      if (offense[p1].ball_carrier) down_yard = offense[p1].x;
      if (offense[p2].ball_carrier) down_yard = offense[p2].x;

      remove_int(auto_play);
      start = false;
      
      if (formation == PASS) {
         switch (pass_ret) {
                case INTERCEPT:
                     send_scoreboard("Intercepted");
                     pause_game();
                     start_over = true;
                     break;
                case INCOMPLETE:
                     send_scoreboard("Incomplete");
                     pause();
                     clear_field();
                     refresh();
                     break;
         }
      }
         
      if (!start_over) {
          down++;
          yard = xyards(down_yard);
          if (down_yard+BMP_CENTER >= field_zone.x2) {
                 down = FIRST;
                 togo = END_DOWN;
                 yard = TD_ZONE;
                 send_scoreboard("Touch Down!");
                 pause_game();
                 if (turn == HOME) 
                    hscore += TD;
                 else
                    vscore += TD;
                 start_over = true;
          }
          if (down_yard <= field_zone.x1+BMP_CENTER) {
                send_scoreboard("Safety!");
                pause_game();
                if (turn == HOME)
                    vscore += SF;
                else
                    hscore += SF;
                start_over = true;
          }
          if (down > FOURTH && !start_over) {
             send_scoreboard("Play Over");
             pause_game();
             start_over = true;
          }
          if (!start_over) {
              if (yard < scrimmage)
                  togo += scrimmage-yard;
              else
                  togo -= yard-scrimmage;             
              if (togo <= END_DOWN && down <= FOURTH) {
                 down = FIRST;
                 scrimmage = yard;
                 togo = FIRST_DOWN;
                 send_scoreboard("First Down");
                 pause();
              } else {
                  if (down == FOURTH) {
                      send_scoreboard("Fourth Down");
                      pause();
                  }                     
              }
              if ((down > FIRST && down < FOURTH) && pass_ret != INCOMPLETE) {
                    send_scoreboard("Down");
                    pause();
              }   
          }
      }
      
      if (start_over) {
             if (num_players > SINGLE) {
                switch (turn) {
                       case HOME:
                            turn = VISITOR;
                            break;
                       case VISITOR:
                            turn = HOME;
                            break;
                }
                load_bitmaps(turn);
             }                            
             down = FIRST;
             yard = YRD20;
             scrimmage = YRD20;
             togo = FIRST_DOWN;
             down_yard = yards(YRD20);
             formation = RUN;
      }
      
      pass_ret = FREE;
      reset_formation();
      
}

void reset_formation() {

      switch (formation) {
             case RUN:
                  run_formation(down_yard);
                  break;
             case PASS:
                  pass_formation(down_yard);
                  break;       
      }
      load_bitmaps2(turn,formation);
      draw_field();
      draw_scores();
      refresh();
      clear_keybuf();
      
}     

void draw_field() {
     
     draw_field(RUN);
     
}
     
void draw_field(int formation) {

    clear_to_color(buffer,GRASS);
    draw_lines();
    draw_players();
    if (formation == PASS) draw_ball(0);
    move_frame();
    refresh();
         
}

void draw_lines() {
    
    int y = 0;
    acquire_screen();
    line(buffer,field_zone.x1,field_zone.y1,field_zone.x1,field_zone.y2,YELLOW);
    int cnt = 0;     
  	for (int x=1;x<20;x++) {
	    line(buffer,field_zone.x1+(x*YARD_MARKER),field_zone.y1,field_zone.x1+(x*YARD_MARKER),field_zone.y2,WHITE);
	    cnt++;
	    if (cnt == 2) {
    	    if (x <= 10) {
       	       textprintf_ex(buffer, font_yard, field_zone.x1+(x*YARD_MARKER)- 10, field_zone.y1+5, WHITE,-1, "%i",x*5);
       	       textprintf_ex(buffer, font_yard, field_zone.x1+(x*YARD_MARKER)- 10, field_zone.y2-25, WHITE,-1, "%i",x*5);
       	    } else {
               y += 20;
               textprintf_ex(buffer, font_yard, field_zone.x1+(x*YARD_MARKER)- 10, field_zone.y1+5, WHITE,-1, "%i",(x*5)-y);
               textprintf_ex(buffer, font_yard, field_zone.x1+(x*YARD_MARKER)- 10, field_zone.y2-25, WHITE,-1, "%i",(x*5)-y);
            }
            cnt = 0;
        }    
    }    
    line(buffer,field_zone.x1+1200,field_zone.y1,field_zone.x1+1200,field_zone.y2,YELLOW);
    release_screen();
    
}

void run_formation(int x) {
   
    start = false;
    move_frame(0);

    offense[p1].ball_carrier = true;
    offense[p2].ball_carrier = false;
    kicker.ball_carrier = false;
            
    offense[p2].x = x;
    offense[p2].y = CENTER;
    offense[p3].x = offense[p2].x;
    offense[p3].y = CENTER - YARD_MARKER;
    offense[p4].x = offense[p2].x;
    offense[p4].y = CENTER + YARD_MARKER;
    
    offense[p1].x = offense[p2].x - YARD_MARKER;
    if (offense[p1].x <= field_zone.x1+BMP_CENTER)
       offense[p1].x = offense[p2].x - BMP_SIZE;
       
    offense[p1].y = CENTER;

    for (int i=p1;i<OPLAYERS;i++) offense[i].reverse = false;
    for (int j=p1;j<DPLAYERS;j++) defense[j].reverse = false;
    

    defense[p2].y = CENTER;
    int pos = 0;
    if (yard <= RED_ZONE2) pos = rndgen(3);
    switch (pos) {
           case 0:
                if (yard >= RED_ZONE3) {                
                    defense[p2].x = offense[p2].x + BMP_SIZE;
                    defense[p3].x = offense[p3].x + BMP_SIZE;
                    defense[p4].x = offense[p4].x + BMP_SIZE;
                } else {
                    defense[p2].x = offense[p2].x + YARD_MARKER;
                    defense[p3].x = offense[p3].x + YARD_MARKER;
                    defense[p4].x = offense[p4].x + YARD_MARKER;
                }
                defense[p3].y = offense[p3].y;
                defense[p4].y = offense[p4].y;
                break;
           case 1:
                defense[p2].x = offense[p2].x + YARD_MARKER*2;
                defense[p3].x = offense[p3].x + YARD_MARKER;
                defense[p3].y = offense[p3].y - YARD_MARKER+BMP_SIZE;
                defense[p4].x = offense[p4].x + YARD_MARKER;
                defense[p4].y = offense[p4].y + YARD_MARKER-BMP_SIZE;
                break;
           case 2:
                defense[p2].x = offense[p2].x + YARD_MARKER;
                defense[p3].x = offense[p3].x + YARD_MARKER*2;
                defense[p3].y = offense[p3].y + BMP_SIZE;
                defense[p4].x = offense[p4].x + YARD_MARKER*2;
                defense[p4].y = offense[p4].y - BMP_SIZE;
                break;                
    }

    pos = rndgen(3) + 3;
    int p1x = offense[p2].x + (YARD_MARKER*6);
    if (p1x >= yards(TD_ZONE)) p1x = yards(TD_ZONE);
    int p5x = offense[p2].x + (YARD_MARKER*pos);
    if (p5x >= yards(TD_ZONE)) {
        switch (pos) {
               case 3:
                    p5x = yards(TD_ZONE);
                    break;
               case 4:
                    p5x = offense[p2].x + (YARD_MARKER*3);     
                    if (p5x >= yards(TD_ZONE)) p5x = yards(TD_ZONE);
                    break;
               case 5:
                    p5x = offense[p2].x + (YARD_MARKER*4);     
                    if (p5x >= yards(TD_ZONE)) offense[p2].x + (YARD_MARKER*3);
                    if (p5x >= yards(TD_ZONE)) p5x = yards(TD_ZONE);
                    break;
        }
    }                        

    defense[p1].x = p1x;
    defense[p5].x = p5x;
    defense[p6].x = defense[p5].x;
    defense[p1].y = CENTER;
    defense[p5].y = offense[p3].y;
    defense[p6].y = offense[p4].y;
        
}

void pass_formation(int x) {
   
    start = false;
    move_frame(0);

    offense[p1].ball_carrier = false;
    offense[p2].ball_carrier = true;
    kicker.ball_carrier = false;
        
    offense[p2].x = x - YARD_MARKER;
    if (offense[p2].x <= field_zone.x1+BMP_CENTER)
       offense[p2].x = x - BMP_SIZE;

    offense[p2].y = CENTER;
    offense[p3].x = x;
    offense[p3].y = CENTER - BMP_SIZE;
    offense[p4].x = x;
    offense[p4].y = CENTER + BMP_SIZE;
    
    offense[p1].x = x;
    
    int pos = rndgen(2);
    if (pos == 0) offense[p1].y = CENTER - (YARD_MARKER + BMP_SIZE);
    if (pos == 1) offense[p1].y = CENTER + YARD_MARKER + BMP_SIZE;

    for (int i=p1;i<OPLAYERS;i++) offense[i].reverse = false;
    for (int j=p1;j<DPLAYERS;j++) defense[j].reverse = false;
    
    defense[p2].y = CENTER;
    if (yard >= RED_ZONE3) {                
        defense[p2].x = offense[p3].x + BMP_SIZE;
        defense[p3].x = offense[p3].x + BMP_SIZE;
        defense[p4].x = offense[p4].x + BMP_SIZE;
    } else {
        defense[p2].x = offense[p3].x + YARD_MARKER;
        defense[p3].x = offense[p3].x + YARD_MARKER;
        defense[p4].x = offense[p4].x + YARD_MARKER;
    }
    defense[p3].y = CENTER - BMP_SIZE;
    defense[p4].y = CENTER + BMP_SIZE;

    pos = rndgen(3) + 4;
    int p1x = offense[p2].x + (YARD_MARKER*6);
    if (p1x >= yards(TD_ZONE)) p1x = yards(TD_ZONE);
    int p5x = offense[p2].x + (YARD_MARKER*pos);
    if (p5x >= yards(TD_ZONE)) {
        switch (pos) {
               case 4:
                    p5x = yards(TD_ZONE);
                    break;
               case 5:
                    p5x = offense[p2].x + (YARD_MARKER*4);     
                    if (p5x >= yards(TD_ZONE)) p5x = yards(TD_ZONE);
                    break;
               case 6:
                    p5x = offense[p2].x + (YARD_MARKER*5);     
                    if (p5x >= yards(TD_ZONE)) offense[p2].x + (YARD_MARKER*4);
                    if (p5x >= yards(TD_ZONE)) p5x = yards(TD_ZONE);
                    break;
        }
    }                        

    defense[p1].x = p1x;
    defense[p5].x = p5x;
    defense[p6].x = defense[p5].x;
    defense[p1].y = CENTER;
    defense[p5].y = CENTER - YARD_MARKER - BMP_SIZE;
    defense[p6].y = CENTER + YARD_MARKER + BMP_SIZE;
        
}

void kick_formation(int x) {


    offense[p1].ball_carrier = false;
    offense[p2].ball_carrier = false;
    kicker.ball_carrier = true;

    offense[p3].x = x;
    offense[p3].y = CENTER - YARD_MARKER+BMP_SIZE;
    offense[p4].x = x;
    offense[p4].y = CENTER + YARD_MARKER-BMP_SIZE;
    kicker.x = x - YARD_MARKER;
    kicker.y = CENTER;
    defense[p3].x = offense[p3].x + BMP_SIZE;
    defense[p3].y = offense[p3].y;
    defense[p4].x = offense[p4].x + BMP_SIZE;
    defense[p4].y = offense[p4].y;

    clear_to_color(buffer,GRASS);
    draw_lines();
    for (int i=p3;i<p5;i++) {
        draw_player(offense[i]);
        draw_player(defense[i]);
    }
    draw_player(kicker);

    move_frame();
    refresh();
           
}

void pass() {

    pass_ret = THROWN;
    switch_bitmaps(turn,p2);
    fball.x = offense[p2].x+BMP_SIZE;
    fball.y = offense[p2].y;
    offense[p2].ball_carrier = false; 
    while (pass_ret == THROWN) {
        pass_ret = fball.pass();
        draw_field(formation);
        rest(50);
    }
    if (pass_ret == INTERCEPT || pass_ret == INCOMPLETE)
       simulate_keypress(KEY_PAUSE);
    else {
       offense[p1].ball_carrier = true;
       switch_bitmaps(turn,p1);
       draw_field();
    }
         
}

void kick_fieldgoal() {
     
    if (yard < FG_RANGE) {
        pause_game("Out of FG Range",false);
        return;   
    }
    
    kick_formation(yards(yard));
    pause_game();
    
    fball.x = kicker.x+BMP_SIZE;
        
    int fg = fball.kick();
    
    move_frame(0);
    for (int i=0;i<FRAME;i++) {
        clear_field();
        for (int j=p3;j<p5;j++) {
            draw_player(offense[j]);
            draw_player(defense[j]);
        }
        draw_player(kicker);
        move_frame();
        refresh();
        rest(50);
    }
    move_frame(FRAME-1);
    int done = 0;
    int angle = 0;
    while (done == 0) {
        clear_field();
        for (int j=p3;j<p5;j++) {
            draw_player(offense[j]);
            draw_player(defense[j]);
        }
        draw_player(kicker);
        if (angle > 256)
           angle = 0;
        else
           angle += 16;
        draw_ball(angle);
        done = fball.move();
        refresh();
        rest(50);
    }    
    
    move_frame(0);
    if (fg == GOAL) {
      send_scoreboard("FG Is Good!");
      if (turn == HOME) 
          hscore += FG;
      else
          vscore += FG;
    } else
      send_scoreboard("Missed FG");
      
    pause_game();
    new_down(true);   
    
}

void draw_player(CPlayer player) {

     draw_player(player,false);
     
}

void draw_player(CPlayer player, bool reverse) {
           
     if (reverse)
          draw_sprite_h_flip(buffer,player.image[frame_idx],player.x-BMP_CENTER,player.y-BMP_CENTER);
     else
          draw_sprite(buffer,player.image[frame_idx],player.x-BMP_CENTER,player.y-BMP_CENTER);     
     
}

void draw_players() {
     
     for (int i=p1;i<OPLAYERS;i++)
         draw_player(offense[i],offense[i].reverse);
     for (int i=p1;i<DPLAYERS;i++)
         draw_player(defense[i],defense[i].reverse);

}

void draw_ball(int angle) {
     
     rotate_sprite(buffer, fball.image, fball.x, fball.y, itofix(angle));

}

void play_music(int toggle) {

    music = (MIDI *)dat[DATA_MUSIC].dat;
    if (music == NULL) {
            allegro_message("Errors loading music track from data file.");
            exit(-1);
    }
                      
    if (toggle == 0)
       play_midi(NULL,toggle);
    else {
       set_volume(VOLUME,VOLUME);
       play_midi(music,toggle);
    }
             
}

void load_fonts() {

     font_yard = (FONT *)dat[DATA_DIGITS].dat;
     font_label = (FONT *)dat[DATA_ECLIPSE].dat;
     font_score = (FONT *)dat[DATA_D3SQUARE].dat;
     font_msg = (FONT *)dat[DATA_TAHOMA].dat;
     font_big = (FONT *)dat[DATA_BIGLETTERS].dat;
     font_label2 = (FONT *)dat[DATA_LABEL2].dat;

     if (font_yard == NULL || 
         font_label == NULL || 
         font_score == NULL ||
         font_msg == NULL || 
         font_big == NULL || 
         font_label2 == NULL) {
            allegro_message("Errors loading fonts from data file.");
            exit(-1);
     }
          
}

void get_bmp(int turn, int frame) {
  
      for (int i=0;i<FRAME;i++) {
             if (kicker.image[i] != NULL) 
                destroy_bitmap(kicker.image[i]);
             kicker.image[i] = create_bitmap(BMP_SIZE,BMP_SIZE);
             if (turn == HOME)
                blit(hsrc,kicker.image[i],i*BMP_SIZE,frame*BMP_SIZE,0,0,BMP_SIZE,BMP_SIZE);                     
             if (turn == VISITOR)
                blit(vsrc,kicker.image[i],i*BMP_SIZE,frame*BMP_SIZE,0,0,BMP_SIZE,BMP_SIZE);                     
             if (kicker.image[i] == NULL) {
                allegro_message("Errors creating player bitmaps (kicker).");
                exit(-1);
             }
      } 
            
}

void get_bmp(int turn, int type, int number, int frame) {
      

      for (int i=0;i<FRAME;i++) {
             switch (type) {
                    case OFFENSE:
                         if (offense[number].image[i] != NULL) 
                            destroy_bitmap(offense[number].image[i]);
                         offense[number].image[i] = create_bitmap(BMP_SIZE,BMP_SIZE);     
                         if (turn == HOME)
                            blit(hsrc,offense[number].image[i],i*BMP_SIZE,frame*BMP_SIZE,0,0,BMP_SIZE,BMP_SIZE);                     
                         if (turn == VISITOR)
                            blit(vsrc,offense[number].image[i],i*BMP_SIZE,frame*BMP_SIZE,0,0,BMP_SIZE,BMP_SIZE);                     
                         if (offense[number].image[i] == NULL) goto error;
                         break;   
                    case DEFENSE:
                         if (defense[number].image[i] != NULL) 
                            destroy_bitmap(defense[number].image[i]);
                         defense[number].image[i] = create_bitmap(BMP_SIZE,BMP_SIZE);     
                         if (turn == HOME)
                            blit(hsrc,defense[number].image[i],i*BMP_SIZE,frame*BMP_SIZE,0,0,BMP_SIZE,BMP_SIZE);                     
                         if (turn == VISITOR)
                            blit(vsrc,defense[number].image[i],i*BMP_SIZE,frame*BMP_SIZE,0,0,BMP_SIZE,BMP_SIZE);                     
                         if (defense[number].image[i] == NULL) goto error;
                         break;
             }
      }          
      
      return;
      
error:
      allegro_message("Errors creating player bitmaps.");
      exit(-1);
                 
}

void load_bitmaps() {
     
     load_bitmaps(HOME);
     load_bitmaps2(HOME,formation);
     
}

void load_bitmaps(int turn) {
     
     int other_turn;
     switch (turn) {
            case HOME:
                 other_turn = VISITOR;
                 break;
            case VISITOR:
                 other_turn = HOME;
                 break;
     }
     
     fball.image = (BITMAP *)dat[DATA_BALL].dat;
     hsrc = (BITMAP *)dat[DATA_HSPRITE].dat;
     vsrc = (BITMAP *)dat[DATA_VSPRITE].dat;    

     if (fball.image == NULL ||
         hsrc == NULL || 
         vsrc == NULL) {
              allegro_message("Errors loading bitmaps from data file.");
              exit(-1);
     }

     get_bmp(turn,KICK_SPRITE);
     
     for (int j=p3;j<OPLAYERS;j++) 
          get_bmp(turn,OFFENSE,j,BLOCK_SPRITE); 
     for (int j=p1;j<DPLAYERS;j++)
          get_bmp(other_turn,DEFENSE,j,DEF_SPRITE); 
     
}
     
void load_bitmaps2(int turn, int formation) {

     switch (formation) {
            case RUN:
                get_bmp(turn,OFFENSE,p1,RUN_SPRITE);
                get_bmp(turn,OFFENSE,p2,BLOCK_SPRITE);
                break;
            case PASS:
                get_bmp(turn,OFFENSE,p1,BLOCK_SPRITE);
                get_bmp(turn,OFFENSE,p2,PASS_SPRITE);
                break;
     }
               
}

void switch_bitmaps(int turn, int player) {

     if (player == p1)
         get_bmp(turn,OFFENSE,p1,RUN_SPRITE);
     else    
         get_bmp(turn,OFFENSE,p2,BLOCK_SPRITE);
     
}    
               
void draw_header() {
     
     acquire_screen();
     textout_ex(screen, font_label, "A", 10, 5, RED, -1);
     textout_ex(screen, font_label, "re", 60, 5, YELLOW, -1);
     textout_ex(screen, font_label, "na", 146, 5, GREEN, -1);
     textout_ex(screen, font_label, "F", 250, 5, RED, -1);
     textout_ex(screen, font_label, "oot", 300, 5, YELLOW, -1);
     textout_ex(screen, font_label, "b", 430, 5, RED, -1);
     textout_ex(screen, font_label, "all", 473, 5, GREEN, -1);
     release_screen();     
}
     
void draw_labels() {

     acquire_screen();
     textout_right_ex(screen, font_label2, "Home:", 200, 360, GREEN, -1);
     
     if (num_players > SINGLE) {
        textout_right_ex(screen, font_label2, "Visitor:", 200, 400, YELLOW, -1);
        masked_blit(vsrc,screen,0,BLOCK_SPRITE*BMP_SIZE,300,400,BMP_SIZE,BMP_SIZE);
     }   
     
     masked_blit(hsrc,screen,0,BLOCK_SPRITE*BMP_SIZE,300,360,BMP_SIZE,BMP_SIZE);
     
     textout_right_ex(screen, font_label2, "Down:", 440, 360, GREEN, -1);
     textout_right_ex(screen, font_label2, "ToGo:", 440, 400, YELLOW, -1);
     textout_right_ex(screen, font_label2, "Yard:", 440, 435, GREEN, -1);
     
     release_screen();     
    
}

void draw_scores() {

     acquire_screen();
     stat = create_bitmap(100,120);     
     
     clear_to_color(stat,LIGHT_BLUE);
     textprintf_right_ex(stat, font_score, 80, 5,  WHITE, BLACK, "%i",hscore);     
     
     if (num_players > SINGLE)
        textprintf_right_ex(stat, font_score, 80, 40, WHITE, BLACK, "%i",vscore);     
     
     blit(stat,screen,0,0,200,350,150,75);
     clear_to_color(stat,LIGHT_BLUE);
     textprintf_right_ex(stat, font_score, 85, 5,  WHITE, BLACK, "%i",down);     
     textprintf_right_ex(stat, font_score, 85, 40, WHITE, BLACK, "%i",togo);
     
     int display_yard = yard;
     char *yard_ind = "o";
     if (yard > MID_FIELD) { 
        display_yard = TD_ZONE - yard;
        yard_ind = "d";
     }
     
     textprintf_right_ex(stat, font_score, 85, 75, WHITE, BLACK, "%s%i",yard_ind,display_yard);     
     blit(stat,screen,0,0,450,350,100,300);
     destroy_bitmap(stat);
     release_screen();
     
}    
void send_msg() {
     
     send_msg("");
         
}

void send_msg(char *message) {

     acquire_screen();
     msg = create_bitmap(300,60);
     clear_to_color(msg,LIGHT_BLUE);
     
     if (strlen(message) > 0)
        textprintf_ex(msg, font_msg, 1, 0, YELLOW, -1, "%s", message);
     
     blit(msg,screen,0,0,15,440,300,60);
     destroy_bitmap(msg);
     release_screen();     

} 

void select_mode(int display) {

     acquire_screen();
     BITMAP *msg2 = create_bitmap(500,60);
     if (msg2 == NULL) goto error;
     if (display == 1) {
         BITMAP *hplayer = create_bitmap(BMP_SIZE,BMP_SIZE);
         BITMAP *vplayer = create_bitmap(BMP_SIZE,BMP_SIZE);
         BITMAP *dplayer = create_bitmap(BMP_SIZE,BMP_SIZE);
         blit(hsrc,hplayer,0,PASS_SPRITE*BMP_SIZE,0,0,BMP_SIZE,BMP_SIZE);
         blit(vsrc,vplayer,0,PASS_SPRITE*BMP_SIZE,0,0,BMP_SIZE,BMP_SIZE);
         blit(vsrc,dplayer,0,DEF_SPRITE*BMP_SIZE,0,0,BMP_SIZE,BMP_SIZE);
         if (hplayer == NULL || vplayer == NULL || dplayer == NULL)
            goto error;
         clear_to_color(msg2,LIGHT_BLUE);
         textout_ex(msg2, font_msg, "Play Mode: [1]", 1, 0, YELLOW, -1);
         draw_sprite(msg2,hplayer,190,5);
         textout_ex(msg2, font_msg, "[2]", 250, 0, YELLOW, -1);
         draw_sprite(msg2,hplayer,290,5);
         draw_sprite(msg2,vplayer,315,5);
         textout_ex(msg2,font_msg, "[3]", 360, 0, YELLOW, -1);
         draw_sprite(msg2,hplayer,400,5);
         draw_sprite(msg2,dplayer,425,5);
         blit(msg2,screen,0,0,15,440,500,60);
         destroy_bitmap(hplayer);
         destroy_bitmap(vplayer);
         destroy_bitmap(dplayer);
    } else {    
         clear_to_color(msg2,LIGHT_BLUE);
         blit(msg2,screen,0,0,15,440,500,60);
    }
    destroy_bitmap(msg2);
    release_screen();     

    return;
    
error:
    allegro_message("Errors creating Play Mode Selection bitmap.");
    exit(-1);

}
 
void send_scoreboard(char *message) {
     
     acquire_screen();
     textprintf_centre_ex(screen, font_big, 340, 160, WHITE, -1, "%s", message);
     release_screen();

}

void move_frame() {
     
     move_frame(-1);
     
}

void move_frame(int frame) {
     
     if (frame < 0) {
          frame_idx++;      
          if (frame_idx >= FRAME) frame_idx = 0;         
     } else
           frame_idx = frame;
          
}

void pause() {
     
      rest(1250);
              
}

void pause_game() {
     
     pause_game("Press [Enter]",true);
     
}

void pause_game(char *msg,bool wait) {

      clear_keybuf();
      send_msg(msg);
      readkey();
      if (wait) while(!key[KEY_ENTER]);
      send_msg();
      clear_keybuf();
      
}          

void clear_field() {
     
      clear_to_color(buffer,GRASS);
      draw_lines();

}

void draw_instruction() {
     
     textout_ex(screen, font, "One/Two Player Mode: use arrow keys to move", 20,100, WHITE, -1);                   
     textout_ex(screen, font, "Versus Mode Offense: use arrow keys to move", 20,110, WHITE, -1);                   
     textout_ex(screen, font, "Versus Mode Defense: use [A][S][D][W] to move", 20,120, WHITE, -1);                   

     textout_ex(screen, font, "[P] to pass", 20, 140, WHITE, -1);                    
     textout_ex(screen, font, "[K] to kick", 20, 150, WHITE, -1);                          

     textout_ex(screen, font, "[F1] Toggle run/pass", 20, 170, WHITE, -1);                    
     textout_ex(screen, font, "[F2] Music on/off", 20, 180, WHITE, -1);                    
     
     textout_ex(screen, font, "[Esc] Quit", 20, 220, WHITE, -1);                    
          
}     
     
void refresh() {
     
     acquire_screen();
     if (yard <= RED_ZONE)
         blit(buffer,screen,0,0,SCREEN_X,SCREEN_Y,display_zone.x2,display_zone.y2);   
     else {
          if (offense[p2].ball_carrier || (formation == PASS && pass_ret == THROWN)) {
             blit(buffer,screen,offense[p2].x - YARD_MARKER,0,SCREEN_X,SCREEN_Y,display_zone.x2,display_zone.y2);              
          } else if (kicker.ball_carrier) {
             blit(buffer,screen,kicker.x - YARD_MARKER,0,SCREEN_X,SCREEN_Y,display_zone.x2,display_zone.y2);             
          } else if (offense[p1].ball_carrier) {
             blit(buffer,screen,offense[p1].x - YARD_MARKER,0,SCREEN_X,SCREEN_Y,display_zone.x2,display_zone.y2);
          } else
             blit(buffer,screen,yards(yard),0,SCREEN_X,SCREEN_Y,display_zone.x2,display_zone.y2);
     }           
     release_screen();
     
}     
