/* weapon.m,
 *
 * This file contains some prebuilt weapon classes so we don't need to
 * rewrite the same code over and over.
 */

#include <math.h>
#include "common.h"
#include "difficulty.h"
#include "projectile.h"
#include "weapon.h"

/*------------------------------------------------------------*/
/* PulseWeapon Protocol helpers. */
#define derive_setWaveDelay(wave)					\
	(id) setWaveDelay:(int)t					\
	{ if (not difficulty_modifier)					\
	    wave = t;							\
	  else								\
	    wave = rint(t * wave_delay_multiplier[difficulty]);		\
	  return self; }

#define derive_setShotsPerWave(num)	(id) setShotsPerWave:(int)n { current_shot_num = 0; num = n; return self; }
#define derive_setShotDelayWaveDelay	(id) setShotDelay:(int)t1 WaveDelay:(int)t2 { return [[self setShotDelay:t1] setWaveDelay:t2]; }
#define derive_currentShot(num)		(int) currentShot { return num; }


/*------------------------------------------------------------*/

/* Weapon handles single-shot weapons that spawns projectiles at the
 * centre of a unit (and at an angle).
 */
@implementation Weapon
+ newWithProjectile:(Class)c
{
    return [[self new] setProjectile:c];
}

- init
{
    [super init];
    fire_tics = 15;
    difficulty_modifier = YES;
    return self;
}

- setDifficultyModifier:(BOOL)yesno
{
    difficulty_modifier = yesno;
    return self;
}

- setProjectile:(Class)c
{
    projectile = c;
    return self;
}

- setShotDelay:(int)t
{
    if (not difficulty_modifier)
	shot_delay = t;
    else
	shot_delay = rint(t * shot_delay_multiplier[difficulty]);

    return self; 
}

- setFireTics:(int)t
{
    fire_tics = t;
    return self;
}

- (int) fireTics
{
    return fire_tics;
}

- (BOOL) fireFromX:(double)x Y:(double)y
{
    return [self fireFromX:x Y:y Angle:-M_PI_2];
}

- (BOOL) fireFromX:(double)x Y:(double)y Angle:(double)theta
{
    if (fire_tics > 0) {
	fire_tics--;
	return NO;
    }
    else {
	[spawn_projectile(projectile, x, y, NO) setAngle:theta];

	fire_tics = shot_delay;
	return YES;
    }
}
@end


/* PulseWeapon: fires bullets in waves from the centre of the unit
 * (and at an angle).
 */
@implementation PulseWeapon
- init
{
    [super init];
    current_shot_num = 0;
    return self;
}

- setShotDelay:(int)t
{
    shot_delay = rint(t * shot_delay_multiplier[difficulty]);
    shot_delay = MAX(1, shot_delay);
    return self; 
}

- derive_setWaveDelay(wave_delay);
- derive_setShotDelayWaveDelay;
- derive_setShotsPerWave(shots_per_wave);
- derive_currentShot(current_shot_num);

- (BOOL) fireFromX:(double)x Y:(double)y Angle:(double)theta
{
    if ([super fireFromX:x Y:y Angle:theta]) {
	current_shot_num++;
	if (current_shot_num >= shots_per_wave) {
	    current_shot_num = 0;
	    fire_tics = wave_delay;
	}
	return YES;
    }
    else
	return NO;
}
@end

/*--------------------------------------------------------------*/

/* OffcentreWeapons: handles weapons for rotatable units that fire
 * from an off-centre position.
 */
@implementation OffcentreWeapon
- setXDisplacement:(double)x YDisplacement:(double)y
{
    xd = x;
    yd = y;
    return self;
}

- (BOOL) fireFromX:(double)x Y:(double)y Angle:(double)theta
{
    if (fire_tics > 0) {
	fire_tics--;
	return NO;
    }
    else {
	x += yd*sin(theta) + xd*sin(theta+M_PI_2);
	y += yd*cos(theta) + xd*cos(theta+M_PI_2);

	[spawn_projectile(projectile, x, y, NO) setAngle:theta];

	fire_tics = shot_delay;
	return YES;
    }
}
@end


@implementation OffcentrePulseWeapon
- init
{
    [super init];
    current_shot_num = 0;
    return self;
}

- derive_setWaveDelay(wave_delay);
- derive_setShotDelayWaveDelay;
- derive_setShotsPerWave(shots_per_wave);
- derive_currentShot(current_shot_num);

- (BOOL) fireFromX:(double)x Y:(double)y Angle:(double)theta
{
    if ([super fireFromX:x Y:y Angle:theta]) {

	current_shot_num++;
	if (current_shot_num >= shots_per_wave) {
	    current_shot_num = 0;
	    fire_tics = wave_delay;
	}
	return YES;
    }
    else
	return NO;
}
@end


/* OffcentrePausePulseWeapons: OffcentrePulseWeapons with a longer
 * pause between half way through the rounds of fire.
 */
@implementation OffcentrePausePulseWeapon
- (BOOL) fireFromX:(double)x Y:(double)y Angle:(double)theta
{
    if ([super fireFromX:x Y:y Angle:theta]) {
	/* Longer pause at the half way mark. */
	if (current_shot_num == shots_per_wave/2) {
	    fire_tics *= 5;
	}
	return YES;
    }
    else
	return NO;
}
@end

/*--------------------------------------------------------------*/

/* Dual weapons: for two identical offcentre weapons located on
 * opposite sides of the ship, except for their y location.
 *
 * ShotDelay > 0: Other barrel will fire in ShotDelay tics.
 * ShotDelay < 0: Barrels fire simultaneously, w/ ShotDelay tics
 *		  between the pulses.  Hard to describe.
 */
@implementation DualWeapon
- init
{
    [super init];
    current_shot_num = 0;
    return self;
}

- derive_setWaveDelay(wave_delay);
- derive_setShotDelayWaveDelay;
- derive_setShotsPerWave(shots_per_wave);
- derive_currentShot(current_shot_num);

- (BOOL) fireFromX:(double)x Y:(double)y Angle:(double)theta
{
    if ([super fireFromX:x Y:y Angle:theta]) {
	yd = -yd;

	/* Fire both barrels simulaneously. */
	if (shot_delay <= 0)
	    [super fireFromX:x Y:y Angle:theta];

	current_shot_num++;
	if (current_shot_num >= shots_per_wave) {
	    current_shot_num = 0;
	    fire_tics = wave_delay;
	}
	else
	    fire_tics = ABS(shot_delay);

	return YES;
    }
    return NO;
}
@end

/*--------------------------------------------------------------*/

/* Side weapons: fires two projectiles out the sides of the ship.
 * XXX: Only works for non-rotated ships.
 */
@implementation SideWeapon
- (BOOL) fireFromX:(double)x Y:(double)y Angle:(double)theta
{
    (void)theta;
    if (fire_tics > 0) {
	fire_tics--;
	return NO;
    }

    fire_tics = shot_delay;
    [spawn_projectile(projectile, x-xd, y+yd, NO) setAngle:M_PI];
    [spawn_projectile(projectile, x+xd, y+yd, NO) setAngle:0];
    return YES;
}
@end

/*--------------------------------------------------------------*/

/* Ring weapons: fires in all directions at once, forming a ring of
 * bullets.  This is not for sweeping type guns.
 */
@implementation RingWeapon
- setStepSize:(double)ss { step_size = ss; return self; }
- setNumSteps:(int)steps { num_steps = steps; return self; }

- (BOOL) fireFromX:(double)x Y:(double)y Angle:(double)theta
{
    int steps = num_steps;

    if (fire_tics > 0) {
	fire_tics--;
	return NO;
    }

    if (steps) {
	while (steps) {
	    [spawn_projectile(projectile, x, y, NO) setAngle:theta];
	    theta += step_size;
	    steps--;
	}
    }
    else {
	/* theta = angle to begin firing from. */
	while (theta < 2*M_PI) {
	    [spawn_projectile(projectile, x, y, NO) setAngle:theta];
	    theta += step_size;
	}
    }

    fire_tics = shot_delay;
    return YES;
}
@end

/*--------------------------------------------------------------*/

/* Spray weapons: spray many shots at once, with a small angle between
 * each pair of bullets.
 */
@implementation SprayWeapon
- setShotsPerWave:(int)n { shots_per_wave = n; return self; }
- setSpray:(double)theta { spray = theta; return self; }

- (BOOL) fireFromX:(double)x Y:(double)y Angle:(double)theta
{
    double phi;
    int i;

    if (fire_tics > 0) {
	fire_tics--;
	return NO;
    }

    phi = theta - spray * (double)(shots_per_wave-1)/2.0;
    for (i = 0; i < shots_per_wave; i++, phi += spray)
	[spawn_projectile(projectile, x, y, NO) setAngle:phi];

    fire_tics = shot_delay;
    return YES;
}
@end
