/* video.m,
 *
 * Stuff to handle the three different buffering methods nicely.
 *
 * 'active page' refers to the bitmap to draw onto.  In the case of
 * page flipping and tripple buffering, this is a video bitmap, which
 * will be shown the next time you call 'video flip to screen'.  For
 * double buffering, active page is the screen.
 */

#ifndef NO_VIDEO_BITMAPS

#include <allegro.h>
#include <assert.h>
#include "common.h"
#include "video.h"


BOOL raid_use_page_flipping = YES;
BOOL raid_use_triple_buffering = YES;
static BITMAP *page[3], *active_page;
static int active_page_n;

/*--------------------------------------------------------------*/

static int next_active_page(const int curr)
{
    const int pf_next[3] = { 1, 0, 0 };
    const int tb_next[3] = { 1, 2, 0 };
    assert(curr < 3);

    if (raid_use_triple_buffering)
	return tb_next[curr];
    elif (raid_use_page_flipping)
	return pf_next[curr];
    else
	assert("Next active page for double buffering!" && NO);
}

/*--------------------------------------------------------------*/

void video_blit_to_all_pages(BITMAP* const src,
			     const int x, const int y,
			     const int w, const int h)
{
    assert(src);
    blit(src, page[0], 0, 0, x, y, w, h);
    blit(src, page[1], 0, 0, x, y, w, h);

    if (raid_use_triple_buffering)
	blit(src, page[2], 0, 0, x, y, w, h);
}


void video_flip_to_screen(void)
{
    if (raid_use_triple_buffering) {
	while (poll_scroll());
	request_video_bitmap(active_page);
    }
    elif (raid_use_page_flipping) {
	show_video_bitmap(active_page);
    }
    else
	return;

    active_page_n = next_active_page(active_page_n);
    active_page = page[active_page_n];
}


void video_flip_to_default(void)
{
    if (raid_use_triple_buffering) {
	active_page_n = 0;
	active_page = page[0];
	request_video_bitmap(page[0]);
    }
    elif (raid_use_page_flipping) {
	active_page_n = 0;
	active_page = page[0];
	show_video_bitmap(page[0]);
    }
}


BITMAP *video_active_page(void)
{
    return active_page;
}

/*--------------------------------------------------------------*/

void video_bitmap_init(void)
{
    /* Do we need to create any video bitmaps? */
    if (not (gfx_capabilities & GFX_CAN_SCROLL))
	raid_use_page_flipping = NO;
    if (not (gfx_capabilities & GFX_CAN_TRIPLE_BUFFER))
	raid_use_triple_buffering = NO;

    if (raid_use_page_flipping || raid_use_triple_buffering) {
	page[0] = create_video_bitmap(SCREEN_W, SCREEN_H);
	page[1] = create_video_bitmap(SCREEN_W, SCREEN_H);

	if (not page[0] || not page[1]) {
	    FREE_BITMAP(page[0]);
	    FREE_BITMAP(page[1]);
	    raid_use_page_flipping = NO;
	    raid_use_triple_buffering = NO;
	    active_page = screen;
	    goto error;
	}

	active_page_n = 0;
	active_page = page[0];

	if (not raid_use_triple_buffering)
	    return;

	page[2] = create_video_bitmap(SCREEN_W, SCREEN_H);
	if (not page[2]) {
	    /* We created 2, but not a 3rd.  We can still use page
	       flipping.  */
	    raid_use_triple_buffering = NO;
	    goto error;
	}
    }
    else {
	/* Double buffering. */
	active_page = screen;
    }
    return;

 error:
    fprintf(stderr, "Error creating video memory page(s).\n");
}


void video_bitmap_shutdown(void)
{
    FREE_BITMAP(page[0]);
    FREE_BITMAP(page[1]);
    FREE_BITMAP(page[2]);
}

#endif
