/* sea-urchin.m,
 *
 * A submarine.  Lurks under water until it reaches the activation
 * line, where rises.  While it is lurking, the sprite is drawn by
 * SeaUrchinLurker so it will be drawn under sea units.  A mess
 * really.
 */

#include <assert.h>
#include <math.h>
#include "candy/boat-sinking.h"
#include "candy/ripple.h"
#include "common.h"
#include "map.h"
#include "projectiles/shockwave.h"
#include "seborrhea/seborrhea.h"
#include "units/sea-urchin.h"


static SebFile *boat_data;

/*--------------------------------------------------------------*/

@interface SeaUrchinLurker: Candy
- (void) die;
@end

@interface SeaUrchinSinking: BoatSinking
@end

/*--------------------------------------------------------------*/

@implementation SeaUrchin
+ (BOOL) loadPrerequisites
{
    return LOAD_PROJECTILE_DATA_FOR(Shockwave);
}

+ derive_loadData(boat_data, "data/sea-urchin");
+ derive_shutdown(boat_data);

- init
{
    [super init];

    health = 120;
    speed = 0.5;

    sprite = [boat_data getSebumByName:"body"];
    w = [sprite width];
    h = [sprite height];
    rotated_sprite = YES;

    gun = [[[[RingWeapon newWithProjectile:[Shockwave class]]
		setShotDelay:64]
	       setStepSize:deg2rad(360.0/8.0)]
	      setNumSteps:8];

    lurk = nil;
    rise_tics = 25;

    chunk_colours = CHUNK_BLUE;
    flags |= FLAG_INVINCIBLE;
    return self;
}

- (void) die
{
    [super die];
    [spawn_candy([SeaUrchinSinking class], x, y, FLOOR_LAYER)
		setAngle:angle];
}

- free
{
    gun = [gun free];

    if (lurk) {
	[(SeaUrchinLurker *)lurk die];
	lurk = nil;
    }

    return [super free];
}

- (void) activate
{
    lurk = [spawn_candy([SeaUrchinLurker class], x, y, FLOOR_LAYER)
		       setAngle:angle];
}

- (void) enableMovement
{
    [super enableMovement];

    if (flags & FLAG_MOVING_ENABLED) {
	[(SeaUrchinLurker *)lurk die];
	lurk = nil;
	flags &=~ FLAG_INVINCIBLE;
    }
}

- (void) move
{
    [super move];

    if (SQ(x - last_ripple_x) + SQ(y - last_ripple_y) >= SQ(6.0)) {
	double delta_x, delta_y;

	last_ripple_x = x;
	last_ripple_y = y;

	delta_x = 9*sin(angle) - 24*sin(angle+M_PI_2);
	delta_y = 9*cos(angle) - 24*cos(angle+M_PI_2);
	spawn_candy([Ripple class], x+delta_x, y+delta_y, FLOOR_LAYER);

	delta_x = -9*sin(angle) - 24*sin(angle+M_PI_2);
	delta_y = -9*cos(angle) - 24*cos(angle+M_PI_2);
	spawn_candy([Ripple class], x+delta_x, y+delta_y, FLOOR_LAYER);
    }
}

- (void) fire
{
    [gun fireFromX:x Y:y];
}

- (enum THING_UPDATE_STATE) update
{
    /* Move (slower) before we activate. */
    if (not (flags & FLAG_MOVING_ENABLED)) {
	x += speed * cos(angle) / 1.5;
	y -= speed * sin(angle) / 1.5;
    }
    else if (rise_tics > 0) {
	rise_tics--;
	if (rise_tics <= 0) {
	    flags |= FLAG_FIRING_ENABLED;
	}
    }

    return [super update];
}

- (void) draw:(BITMAP *)dest
{
    if (lurk) {
	return;
    }

    if (rise_tics > 0) {
	int alpha = 0xe0 * rise_tics/25;
	[sprite drawTo:dest X:x-offsetX Y:y-offsetY Tint:0x00:0x20:0x40:alpha Angle:angle];
    }
    else {
	[super draw:dest];
    }
}

- (int) collisionLists
{
    if (not (flags & FLAG_MOVING_ENABLED))
	return 0;
    else
	return COLLIDES_WITH_PROJECTILES_AND_NUKES;
}
@end

/*--------------------------------------------------------------*/

@implementation SeaUrchinLurker
- init
{
    [super init];
    health = 1;
    sprite = [boat_data getSebumByName:"body"];
    speed = 0.5 / 1.5;
    return self;
}

- (void) draw:(BITMAP *)dest
{
    [sprite drawTo:dest X:x-offsetX Y:y-offsetY Tint:0x00:0x20:0x40:0xe0 Angle:angle];
}

- (void) die
{
    health = 0;
}

- (BOOL) mayStillInfluenceGameplay
{
    return YES;
}
@end

/*--------------------------------------------------------------*/

@implementation SeaUrchinSinking
- init
{
    [super init];
    sprite = [boat_data getSebumByName:"deb"];
    return self;
}
@end
