/* scurvy.m,
 *
 * "We're sure to avoid scurvy if we'd all eat an orange."
 */

#include <math.h>
#include "candy/boat-sinking.h"
#include "candy/ripple.h"
#include "common.h"
#include "map.h"
#include "projectiles/fireball.h"
#include "rotate.h"
#include "seborrhea/seborrhea.h"
#include "sound.h"
#include "unit-seeking.h"
#include "units/all-units.h"
#include "units/scurvy.h"
#include "units/cluster-bomb.h"


static SebFile *boat_data;

/*--------------------------------------------------------------*/

@interface ScurvySinking: BoatSinking
@end

/*--------------------------------------------------------------*/

@implementation Scurvy
+ (BOOL) loadPrerequisites
{
    return LOAD_UNIT_DATA_FOR(ClusterBomb, YES);
}

+ derive_loadData(boat_data, "data/scurvy");
+ derive_shutdown(boat_data);

- init
{
    [super init];

    health = 400;
    critical_health = 200;
    speed = 0.75;

    sprite = [[SebAnimator new] setAnimation:(SebAnimation *)
				[boat_data getSebumByName:"body"]];
    w = [sprite width];
    h = [sprite height];
    rotated_sprite = YES;

    turret = [[SebAnimator new] setAnimation:(SebAnimation *)
				[boat_data getSebumByName:"gun-l"]];
    turret_rot_speed = deg2rad(1.0);
    gun = [[[[DualWeapon newWithProjectile:[Fireball class]]
		setShotDelay:10 WaveDelay:20]
	       setShotsPerWave:2]
	      setXDisplacement:20 YDisplacement:5];

    upper_turret = [[SebAnimator new] setAnimation:(SebAnimation *)
				      [boat_data getSebumByName:"gun-u"]];

    base_debris = nil;
    return self;
}

- free
{
    upper_turret = [upper_turret free];
    return [super free];
}


- (void) die
{
    [super die];
    [spawn_candy([ScurvySinking class], x, y, FLOOR_LAYER) setAngle:angle];
}


- (void) drawTurret:(BITMAP *)dest
{
    /* Front turret. */
    if (upper_turret) {
	double x_ = x + (8.0 * cos(angle)) - offsetX;
	double y_ = y - (8.0 * sin(angle)) - offsetY;
	[upper_turret drawTo:dest X:x_ Y:y_ Angle:upper_turret_angle];
    }

    /* Rear turret. */
    if (turret) {
	double x_ = x + (-24.0 * cos(angle)) - offsetX;
	double y_ = y - (-24.0 * sin(angle)) - offsetY;
	[turret drawTo:dest X:x_ Y:y_ Angle:turret_angle];
    }
}

- (void) fire
{
    /* Rear gun. */
    {
	double x_ = x + (-24.0 * cos(angle));
	double y_ = y - (-24.0 * sin(angle));
	if ([gun fireFromX:x_ Y:y_ Angle:turret_angle]) {
	    [(SebAnimator *)turret resetAnimation];
	}
    }

    /* Front gun. */
    if (--upper_gun_tics == 4*4) {
	[(SebAnimator *)upper_turret resetAnimation];
    }
    else if (upper_gun_tics <= 0) {
	double x_ = x + (8.0*cos(angle)) + (16.0*cos(upper_turret_angle));
	double y_ = y - (8.0*sin(angle)) - (16.0*sin(upper_turret_angle));
	upper_gun_tics = 40;

	[spawn_unit([ClusterBomb class], x_, y_, ACTIVE_AIR_LIST, YES)
		   setAngle:upper_turret_angle];

	play_panned_sample((Sebum<SebSample> *)[boat_data getSebumByName:"snd"], x);
    }
}

- (enum THING_UPDATE_STATE) update
{
    int ret = [super update];

    /* Rotate turret. */
    if (not target || not unit_exists(target, ALLY_LIST))
	target = FIND_CLOSEST_ALLY(x, y);

    if (target && upper_gun_tics > 10) {
	upper_desired_angle = angle_towards_unit(x, y-10, target);
	limited_rotate(&upper_turret_angle, upper_desired_angle, deg2rad(4.0));
	simplify_angle(&upper_turret_angle);
    }

    return ret;
}

- (void) activate
{
    upper_turret_angle = upper_desired_angle = turret_angle;
}

- (void) move
{
    if (SQ(x - last_ripple_x) + SQ(y - last_ripple_y) >= SQ(6.0)) {
	double delta_x, delta_y;
	double yy = rnd(-15, 15);

	last_ripple_x = x;
	last_ripple_y = y;

 	delta_x =  yy*sin(angle) - 47*sin(angle+M_PI_2);
	delta_y =  yy*cos(angle) - 47*cos(angle+M_PI_2);
	spawn_candy([Ripple class], x+delta_x, y+delta_y, FLOOR_LAYER);
    }

    [super move];
}
@end

/*--------------------------------------------------------------*/

@implementation ScurvySinking
- init
{
    [super init];
    sprite = [boat_data getSebumByName:"dead"];
    return self;
}
@end
