/* scourge.m,
 *
 * A weak ship that sort of resembles the 'scourge' unit from
 * Starcraft.  Also looks a little like the Protoss shuttle.
 */

#include "common.h"
#include "map-editor/unit-editor.h"
#include "map-editor/unit-extras.h"
#include "projectiles/fireball.h"
#include "seborrhea/seborrhea.h"
#include "units/all-units.h"
#include "units/scourge.h"


static SebFile *unit_data;
static SebAnimation *move_anim, *left_anim, *right_anim;

/*--------------------------------------------------------------*/

@interface BasicScourgeFollower (Private)
- (void) setBankingSprite;
@end

/*--------------------------------------------------------------*/

@implementation Scourge
+ (BOOL) loadPrerequisites
{
    return LOAD_PROJECTILE_DATA_FOR(Fireball);
}

+ (BOOL) loadData
{
    if (not [self loadData:&unit_data :"data/scourge"])
	return NO;

    move_anim  = (SebAnimation *)[unit_data getSebumByName:"move-anim"];
    left_anim  = (SebAnimation *)[unit_data getSebumByName:"left-anim"];
    right_anim = (SebAnimation *)[unit_data getSebumByName:"right-anim"];
    return YES;
}

+ (void) shutdown
{
    FREE_SEBFILE(unit_data);
    move_anim = nil;
    left_anim = nil;
    right_anim = nil;
}

- init
{
    [super init];

    health = 30;
    speed = 1.3;

    sprite = [[SebAnimator new] setAnimation:move_anim];
    shadow = [unit_data getSebumByName:"shadow"];
    w = [sprite width];
    h = [sprite height] - 8;

    gun = [[[[DualWeapon newWithProjectile:[Fireball class]]
		setXDisplacement:12 YDisplacement:6]
	       setShotDelay:-50 WaveDelay:50]
	      setShotsPerWave:1];

    return self;
}
@end

/*--------------------------------------------------------------*/
/* Abstract Path Follower Scourge class.			*/
/*--------------------------------------------------------------*/

@implementation BasicScourgeFollower
+ (BOOL) loadPrerequisites
{
    return LOAD_UNIT_DATA_FOR(Scourge, YES);
}

- init
{
    [super init];

    health = 30;
    speed = 3.0;

    sprite = [[SebAnimator new] setAnimation:move_anim];
    shadow = [unit_data getSebumByName:"shadow"];
    w = [sprite width];
    h = [sprite height] - 8;

    gun = [[[[DualWeapon newWithProjectile:[Fireball class]]
		setXDisplacement:12 YDisplacement:6]
	       setShotDelay:-50 WaveDelay:50]
	      setShotsPerWave:1];

    flags |= FLAG_FIRING_ENABLED;
    return self;
}

- free
{
    gun = [gun free];
    return [super free];
}

- (void) die
{
    /* Only spawn powerups if entire group dies. */
    if ([group dropGroupRef] == NO)
	flags &=~FLAG_DEATH_SPAWN_POWERUP;

    [super die];
}

- (void) fire
{
    [gun fireFromX:x Y:y];
}

- (void) move
{
    [super move];

    if (path_progress == 35) {
	[self setBankingSprite];
    }
}

	/* Load/Save.  format: group_num;x;y;a[;flags] */
- (void) importUnitProperties:(char *)str
{
    char str2[1024];

    sscanf(str, "%d;%s", &group_num, str2);
    [super importUnitProperties:str2];

    /* Find the appropriate group to add it to. */
    group = getGroupRef(group_num);
}

- (char *) exportUnitProperties:(char *)str
{
    char str2[1024] = "\0";

    [super exportUnitProperties:str2];
    snprintf(str, sizeof(str2), "%d;%s", group_num, str2);
    return str;
}

	/* Protocols. */
- derive_setGroup(group_num);
@end


@implementation BasicScourgeFollower (Private)
- (void) setBankingSprite
{
    if (mirror_path) {
	[(SebAnimator *)sprite setAnimation:left_anim];
	shadow = [unit_data getSebumByName:"shadow-l"];
    }
    else {
	[(SebAnimator *)sprite setAnimation:right_anim];
	shadow = [unit_data getSebumByName:"shadow-r"];
    }
}
@end


@implementation BasicScourgeFollower (MapEditor)
- (void) drawMapEditorExtras:(BITMAP *)dest
{
    [super drawMapEditorExtras:dest];
    draw_group_num(dest, x, y, group_num);
}
@end

/*--------------------------------------------------------------*/
/* Path Follower Scourges.					*/
/*--------------------------------------------------------------*/

@implementation ScourgePath
- init
{
    [super init];
    the_way = (SebSpline *)[unit_data getSebumByName:"path"];
    return self;
}
@end


@implementation ScourgeMirror
- init
{
    [super init];
    the_way = (SebSpline *)[unit_data getSebumByName:"path2"];
    return self;
}
@end


@implementation ScourgeMirror (Private)
- (void) setBankingSprite
{
    if (mirror_path) {
	[(SebAnimator *)sprite setAnimation:right_anim];
	shadow = [unit_data getSebumByName:"shadow-r"];
    }
    else {
	[(SebAnimator *)sprite setAnimation:left_anim];
	shadow = [unit_data getSebumByName:"shadow-l"];
    }
}
@end
