/* robot.m,
 *
 * Little ally ships that follow the player around and fire lots.
 * They are considered as a tertiary weapon.
 */

#include <assert.h>
#include <math.h>
#include "candy/spark.h"
#include "collision.h"
#include "common.h"
#include "nuke.h"
#include "projectiles/laser.h"
#include "projectiles/toothpaste.h"
#include "projectiles/vulcan.h"
#include "seborrhea/seborrhea.h"
#include "units/robot.h"


#define MAX_KAMIKAZE_DISTANCE	60

/*--------------------------------------------------------------*/

@interface Robot (Private)
- (void) moveKamikaze;
@end

/*--------------------------------------------------------------*/

@implementation Robot
- init
{
    [super init];

    health = 15;

    sprite = [[SebAnimator new] setAnimation:(SebAnimation *)
				[base_sebum getSebumByName:"robot/body"]];
    shadow = [base_sebum getSebumByName:"robot/shadow"];
    w = 20;
    h = 20;

    /* Start invincible when coming from bottom of screen. */
    flags |= FLAG_INVINCIBLE;
    return self;
}

- (id) setParent:(Player *)p
{
    parent = p;
    pid = [parent playerID];
    return self;
}

- (void) exciteMeIgniteMe
{
    /* Robots don't stack up with satellites. */
    [self die];
}

- (id) setDisplacementX:(int)x_ Y:(int)y_
{
    x_displacement = x_;
    y_displacement = y_;
    return self;
}

- (void) spawnDyingExplosions
{
    if (rnd(0, 2*health) == 0)
	spawn_candy([Spark class], x, y, HIGH_LAYER);
}

- (void) die
{
    if (flags & FLAG_DYING)
	return;

    /* Go on a kamikaze after death. */
    health = 1.0;
    kamikaze_distance = MAX_KAMIKAZE_DISTANCE;
    yv = -1.0;
    flags |= FLAG_DYING;

    if (parent) {
	[parent disownChild:self];
	parent = nil;
    }
}

- (enum THING_UPDATE_STATE) update
{
    if (health <= 2)
        [self spawnDyingExplosions];

    if (flags & FLAG_DYING) {
        [self moveKamikaze];

        if (health <= 0) {
	    [spawn_nuke(x, y, pid) setNukeRadius:75];
            return THING_DEAD;
	}
    }
    else {
        [self move];

        if (--fire_tics <= 0)
            [self fire];
    }

    return THING_NORMAL;
}

- (void) move
{
    if (parent) {
	double dest_x, dest_y;
	[parent getX:&dest_x Y:&dest_y];

	if (flags & FLAG_FIRING_ENABLED) {
	    x += (dest_x + x_displacement - x) / 10;
	    y += (dest_y + y_displacement - y) / 10;
	}
	else {
	    x += (dest_x + x_displacement/2 - x) / 5;
	    y += (dest_y + y_displacement/2 - y) / 5;
	}

	if (ABS(dest_y - y) < ABS(y_displacement*2))
	    flags &=~FLAG_INVINCIBLE;
    }
}

- (void) fire
{
#define SPAWN_ALLY_PROJ(c,x,y)	[(AllyProjectile *)spawn_projectile([c class],x,y,YES) setParentID:pid]

    if (not parent)
	return;

    if ([parent isFiring]) {
	w = 20;
	flags |= FLAG_FIRING_ENABLED;
    }
    else {
	w = 14;
	flags &=~FLAG_FIRING_ENABLED;
	return;
    }

    switch ([parent weaponType]) {
      case PRIMARY_WEAPON_LASER:
	  /* Note: LaserMedium is weaker than Laser. */
	  SPAWN_ALLY_PROJ(LaserMedium, x, y-5);
	  fire_tics = 22;
	  break;
      case PRIMARY_WEAPON_TOOTHPASTE:
	  SPAWN_ALLY_PROJ(Toothpaste, x, y-5);
	  fire_tics = 12;
	  break;
      case PRIMARY_WEAPON_VULCAN:
	  SPAWN_ALLY_PROJ(Vulcan, x-3, y-5);
	  SPAWN_ALLY_PROJ(Vulcan, x+3, y-5);
	  fire_tics = 15;
	  break;
      default:
	  assert(NO);
    }

#undef SPAWN_ALLY_PROJ
}

- (int) collisionLists
{
    return ACTIVE_AIR_LIST|COLLIDES_WITH_PROJECTILES;
}

- (BOOL) checkCollisionWith:(Thing<DetectsCollision> *)object
{
    /* Allow collision even if we are dying. */
    double x1, y1, x2, y2;
    int w1, h1, w2, h2;

    [self   getX:&x1 Y:&y1 W:&w1 H:&h1];
    [object getX:&x2 Y:&y2 W:&w2 H:&h2];

    return bounding_box_collision(x1-w1/2.0, y1-h1/2.0, x1+w1/2.0, y1+h1/2.0,
				  x2-w2/2.0, y2-h2/2.0, x2+w2/2.0, y2+h2/2.0);
}
@end


@implementation Robot (Private)
- (void) moveKamikaze
{
    /* Kamikaze!  (Bonzai!) */
    yv -= 0.1;
    x += 0.5 * ((x_displacement < 0) ? -1 : 1);
    y += yv;

    if (--kamikaze_distance <= 0)
	health = 0.0;
}
@end
