/* miniboss0.m,
 *
 * The miniboss for the desert level, a triple-turreted tank.  100%
 * bonus damage for laser and vulcan, 20% reduction for toothpaste.
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "candy/explosion.h"
#include "candy/large-chunk.h"
#include "candy/tank-debris.h"
#include "common.h"
#include "map-editor/unit-editor.h"
#include "map.h"
#include "nuke.h"
#include "projectiles/fireball.h"
#include "projectiles/pulse-laser.h"
#include "projectiles/railgun.h"
#include "rotate.h"
#include "seborrhea/seborrhea-allegro.h" /* XXX */
#include "seborrhea/seborrhea.h"
#include "sound.h"
#include "unit-seeking.h"
#include "units/miniboss0.h"


#define foreach_child(c)	for (c = 0; c < MINIBOSS0_NUM_CHILDREN; c++)
#define GUN_TURRET_HEALTH	4000
#define PULSE_LASER_HEALTH	2000
#define RAILGUN_HEALTH		4000
#define TOTAL_HEALTH		(GUN_TURRET_HEALTH+PULSE_LASER_HEALTH+RAILGUN_HEALTH)
#define TRACK_HEALTH		300
#define TANK_SPEED		0.65


typedef struct {
    int x, y;
} displacement_t;

static const displacement_t weapon_displacements[MINIBOSS0_NUM_CHILDREN] = {
    { 28, -30}, { -28, -14}, { 0, 23}
};

static const displacement_t overlay_displacements[MINIBOSS0_NUM_CHILDREN] = {
    { 23, -32}, { -28, -20}, { 0, 25}
};

static Sebum<SebImage> *overlay[MINIBOSS0_NUM_CHILDREN];


static SebFile *boss_sebum;
static SAMPLE *engine_sample;	/* XXX */
static BITMAP *meter_gray, *meter_purple; /* XXX */

/*------------------------------------------------------------*/

@interface MiniBoss0BodyDebris: Candy
@end

@interface GunTurret: MiniBoss0AbstractChild
{
    DualWeapon *gun[2];
}
@end

@interface LaserTurret: MiniBoss0AbstractChild
{
    OffcentreWeapon *gun;
}
@end

@interface RailgunTurret: MiniBoss0AbstractChild
{
    int shot_delay, fire_tics;
}
@end

@interface MiniBoss0Tracks: Candy
@end

@interface RailgunTurretDebris: TurretDebris
@end

/*------------------------------------------------------------*/

@implementation MiniBoss0
	/* Initialization. */
+ (BOOL) loadPrerequisites
{
    return (LOAD_PROJECTILE_DATA_FOR(Fireball) &&
	    LOAD_PROJECTILE_DATA_FOR(PulseLaser) &&
	    LOAD_PROJECTILE_DATA_FOR(Railgun));
}

+ (BOOL) loadData
{
    boss_sebum = [SebFile new];
    if (not [boss_sebum loadSebumDirectory:"data/miniboss0"])
	return NO;

    meter_gray   = [(SebImageAllegro *)[base_sebum getSebumByName:"meter/bg-gray"] bitmap];
    meter_purple = [(SebImageAllegro *)[base_sebum getSebumByName:"meter/fg-purple"] bitmap];

    overlay[MINIBOSS0_LASER_TURRET]   = [boss_sebum getSebumByName:"laser-d"];
    overlay[MINIBOSS0_GUN_TURRET]     = [boss_sebum getSebumByName:"gun-d"];
    overlay[MINIBOSS0_RAILGUN_TURRET] = [boss_sebum getSebumByName:"railgun-d"];
    return YES;
}

+ (void) shutdown
{
    int c;

    FREE_SEBFILE(boss_sebum);
    meter_gray   = NULL;
    meter_purple = NULL;

    foreach_child (c)
	overlay[c] = nil;
}

- init
{
    [super init];

    sprite = [[SebAnimatorManualWithDelay new] setAnimation:(SebAnimation *)
					       [boss_sebum getSebumByName:"body"]];
    w = [sprite width];
    h = [sprite height];

    flags |= FLAG_INVINCIBLE;
    activate_y = -200;

    travel_range = 610;
    speed = TANK_SPEED;
    last_track_y = -1;

    ring_gun = [[[RingWeapon newWithProjectile:[Fireball class]]
		    setShotDelay:100]
		   setStepSize:deg2rad(20.0)];

    return self;
}

- free
{
    if (engine_sample) {
	stop_sample(engine_sample);	/* XXX */
	engine_sample = NULL;
    }
    return [super free];
}

- (void) drawHealthMeter:(BITMAP *)dest
{
    int c, ww, life = 0;

    foreach_child (c)
	if (child[c])
	    life += [child[c] health];

    ww = 75 * life/TOTAL_HEALTH;
    blit(meter_gray,   dest,     0, 0, dest->w/2-75, dest->h-10,  150, 8);
    blit(meter_purple, dest, 75-ww, 1, dest->w/2-ww, dest->h-9,  2*ww, 6);
}

- (void) activate
{
#define SPAWN_CHILD(c)		[(<MiniBoss0Child>)spawn_unit([c class], x, y, ACTIVE_GROUND_LIST, NO) setParent:self];

    boss = self;

    child[MINIBOSS0_GUN_TURRET]     = SPAWN_CHILD(GunTurret);
    child[MINIBOSS0_LASER_TURRET]   = SPAWN_CHILD(LaserTurret);
    child[MINIBOSS0_RAILGUN_TURRET] = SPAWN_CHILD(RailgunTurret);

    engine_sample = [(SebSampleAllegro *)[boss_sebum getSebumByName:"engine"] sample];
    if (sound_vol > 0)
	play_sample(engine_sample, 192*sound_vol/255, 128, 1000, 1);

#undef SPAWN_CHILD
}

- (void) draw:(BITMAP *)dest
{
    int c;

    [super draw:dest];

    foreach_child (c) {
	if (child[c])
	    continue;

	[overlay[c] drawTo:dest
		X:x+overlay_displacements[c].x-[overlay[c]  width]/2-offsetX
		Y:y+overlay_displacements[c].y-[overlay[c] height]/2-offsetY];
    }
}

- (void) move
{
    int c;

    if (travel_range <= 0.0)
	return;

    y -= speed;
    travel_range -= speed;

    foreach_child (c) {
	if (debris[c])
	    [debris[c] setX:x Y:y];
    }

    [(SebAnimatorManualWithDelay *)sprite nextFrame];

    if (ABS(last_track_y - y) > 28) {
	int y_ = y + [sprite height]/2 - 55;
	/* -55 is for track's height. */
	spawn_candy([MiniBoss0Tracks class], x, y_, FLOOR_LAYER);
	last_track_y = y;
    }

    if (travel_range <= 130 && not (flags & FLAG_DYING))
	[current_map setScrollRate:0.0 :YES];
}

- (void) fire
{
    /* Same coordinates as railgun. */
    if (!child[MINIBOSS0_RAILGUN_TURRET])
	[ring_gun fireFromX:x+weapon_displacements[MINIBOSS0_RAILGUN_TURRET].x
		  Y:y+weapon_displacements[MINIBOSS0_RAILGUN_TURRET].y 
		  Angle:0.0];
}

- (void) die
{
    int c;

    foreach_child (c) {
	if (debris[c]) {
	    [debris[c] die];
	    debris[c] = nil;
	}
    }

    /* Sprite was a SebAnimator. */
    [sprite free];
    sprite = [boss_sebum getSebumByName:"deb"];
    [super die];
}

- (void) spawnDyingExplosions
{
    if (rnd(health, 0) < -40) {
	spawn_candy([LargeChunkGreen class], x, y, HIGH_LAYER);

	/* Scatter some debris on the ground around us. */
	{
	    int dist = rnd(h, h+50);
	    double theta = deg2rad(rnd(-180, 180));

	    spawn_candy([TankDebris class],
			x + dist*cos(theta), y + dist*sin(theta),
			FLOOR_LAYER);
	}
    }

    if (health > -100) {
	w = [sprite width]/2;
	h = [sprite height]/2;
	spawn_candy([BigExplosion class], x + rnd(-w, w), y + rnd(-h, h), MEDIUM_LAYER);
	play_explosion_sample(x);

	if (health < -50)	
	    game_flags |= FLAG_SCREEN_SHAKE;
    }
    else {
	spawn_nuke(x, y, -1);
        [self delete];
    }

    health--;
    flash_tics += 2;
}

- (void) delete
{
    [current_map setScrollRate:DEFAULT_SCROLL_RATE :YES];
    spawn_candy([MiniBoss0BodyDebris class], x, y, LOW_LAYER);

    if (boss == self)
	boss = nil;

    [super delete];
}

- (double) velocity
{
    /* Railgun turret wants this. */
    if ((flags & FLAG_MOVING_ENABLED) && (travel_range > 0.0))
	return speed;
    else
	return 0.0;
}

	/* Protocol stuff. */
- (void) disownChild:(Unit *)unit
{
    int c;

    assert(unit);
    foreach_child (c) {
	if (child[c] == unit) {
	    child[c] = nil;
	    break;
	}
    }
    assert(c < MINIBOSS0_NUM_CHILDREN && "Attempted to disown non-child-unit from MiniBoss0");

    if (c == MINIBOSS0_RAILGUN_TURRET)
	flags |= FLAG_FIRING_ENABLED;

    if (not child[MINIBOSS0_GUN_TURRET] &&
	not child[MINIBOSS0_LASER_TURRET] &&
	not child[MINIBOSS0_RAILGUN_TURRET])
	[self die];
}

- (int) collisionLists { return 0; }

- derive_fixedAngle;
- derive_alwaysInfluenceGameplay;
- derive_setXYAlwaysCentred;
@end


@implementation MiniBoss0 (MapEditor)
- (void) drawMapEditorExtras:(BITMAP *)dest
{
    const char *weapon_sprite_names[MINIBOSS0_NUM_CHILDREN] = {
	"gun-l", "laser", "railgun"
    };

    unsigned int c;

    foreach_child (c) {
	[(<SebImage>)[boss_sebum getSebumByName:weapon_sprite_names[c]]
		     drawTo:dest
		     X:x+weapon_displacements[c].x - offsetX
		     Y:y+weapon_displacements[c].y - offsetY
		     Angle:M_PI_2];
    }

    [super drawMapEditorExtras:dest];
}
@end


@implementation MiniBoss0BodyDebris
- init
{
    [super init];
    sprite = [boss_sebum getSebumByName:"deb"];
    return self;
}
@end

/*------------------------------------------------------------*/
/* Abstract MiniBsos Child stuff.			      */
/*------------------------------------------------------------*/

@implementation MiniBoss0AbstractChild
- init
{
    [super init];
    rotated_sprite = YES;
    angle = M_PI_2;
    flags |= FLAG_FIRING_ENABLED;
    return self;
}

- (void) delete
{
    if (parent) {
	[parent disownChild:self];
	parent = nil;
    }

    [super delete];
}

- (void) move
{
    /* Parent not set?  This should never happen. */
    assert(parent && "Parent (MiniBoss0) went missing!");

    if (unit_exists(parent, ACTIVE_GROUND_LIST)) {
	[parent getX:&x Y:&y];
	x += weapon_displacements[child_num].x;
	y += weapon_displacements[child_num].y;
    }
    else {
	/* Parent removed before we were.  This should never happen
	   since the parent is invincible. */
	assert("Parent (MiniBoss0) deleted before child!\n" && NO);
    }
}

- (int) health { return health; }
- (id) setParent:(Unit<OwnsChildren> *)unit { parent = (MiniBoss0 *)unit; return self; }
- (int) collisionLists { return COLLIDES_WITH_PROJECTILES; }

- (int) receiveDamage:(int)damage type:(enum DAMAGE_TYPE)type
{
    if (type == DAMAGE_TYPE_VULCAN)
	damage *= 2.0;
    else if (type == DAMAGE_TYPE_LASER)
	damage *= 2.0;
    else if (type == DAMAGE_TYPE_TOOTHPASTE)
	damage *= 0.80;

    return [super receiveDamage:damage type:type];
}

- (void) findDesiredAngle
{
    /* Relocate the target and rotate towards it. */
    Unit *target = find_closest_ally_according_to_rotation(x, y, angle);
    if (target) {
	double x_, y_;
	[target getX:&x_ Y:&y_];
	desired_angle = atan2(y-y_, x_-x);
    }
    else
	desired_angle = deg2rad(90.0);

    limited_rotate(&angle, desired_angle, max_rotation_angle);
    simplify_angle(&angle);
}

- (enum THING_UPDATE_STATE) update
{
    [self findDesiredAngle];
    return [super update];
}

- derive_alwaysInfluenceGameplay;
@end

/*------------------------------------------------------------*/
/* Gun Turret.						      */
/*------------------------------------------------------------*/

@interface GunTurretDebris: TurretDebris
@end

@implementation GunTurret
- init
{
    [super init];

    child_num = MINIBOSS0_GUN_TURRET;

    health = GUN_TURRET_HEALTH;
    critical_health = GUN_TURRET_HEALTH/2;

    sprite = [[SebAnimator new] setAnimation:(SebAnimation *)[boss_sebum getSebumByName:"gun-r"]];
    w = [sprite width];
    h = [sprite height];

    max_rotation_angle = deg2rad(2.1);

    gun[0] = [[[DualWeapon newWithProjectile:[Fireball class]]
		  setShotDelay:20 WaveDelay:20]
		 setXDisplacement:15 YDisplacement:-4];
    gun[1] = [[[DualWeapon newWithProjectile:[Fireball class]]
		  setShotDelay:20 WaveDelay:20]
		 setXDisplacement:15 YDisplacement:-4];

    return self;
}

- free
{
    gun[0] = [gun[0] free];
    gun[1] = [gun[1] free];
    return [super free];
}

- (void) die
{
    [super die];
    spawn_candy([GunTurretDebris class],
		x+weapon_displacements[child_num].x,
		y+weapon_displacements[child_num].y,
		HIGH_LAYER);
}

- (void) fire
{
    if ([gun[0] fireFromX:x Y:y Angle:angle - deg2rad(2.5)]) {
	SebAnimation *gun_turret_left  = [boss_sebum getSebumByName:"gun-l"];
	SebAnimation *gun_turret_right = [boss_sebum getSebumByName:"gun-r"];

	if ([(SebAnimator *)sprite getAnimation] == gun_turret_left)
	    [(SebAnimator *)sprite setAnimation:gun_turret_right];
	else
	    [(SebAnimator *)sprite setAnimation:gun_turret_left];
    }
    [gun[1] fireFromX:x Y:y Angle:angle + deg2rad(2.5)];
}
@end


@implementation GunTurretDebris
- init
{
    [super init];
    health = max_health = 90;
    sprite = [(SebAnimation *)[boss_sebum getSebumByName:"gun-l"] getFrame:-1];
    return self;
}
@end

/*------------------------------------------------------------*/
/* Laser Turret.					      */
/*------------------------------------------------------------*/

@interface LaserTurretDebris: TurretDebris
@end

@implementation LaserTurret
- init
{
    [super init];

    child_num = MINIBOSS0_LASER_TURRET;

    health = PULSE_LASER_HEALTH;
    critical_health = PULSE_LASER_HEALTH/2;

    sprite = [boss_sebum getSebumByName:"laser"];
    w = [sprite width];
    h = [sprite height];

    gun = [[[OffcentreWeapon newWithProjectile:[PulseLaser class]]
	       setShotDelay:28]
	      setXDisplacement:28 YDisplacement:0];

    max_rotation_angle = deg2rad(3.0);

    return self;
}

- free
{
    gun = [gun free];
    return [super free];
}

- (void) fire
{
    [gun fireFromX:x Y:y Angle:angle];
}

- (void) die
{
    [super die];
    spawn_candy([LaserTurretDebris class], 
		x+weapon_displacements[child_num].x,
		y+weapon_displacements[child_num].y,
		HIGH_LAYER);
}
@end


@implementation LaserTurretDebris
- init
{
    [super init];
    health = max_health = 80;
    sprite = [boss_sebum getSebumByName:"laser"];
    return self;
}
@end

/*------------------------------------------------------------*/
/* Railgun.						      */
/*------------------------------------------------------------*/

@implementation RailgunTurret
- init
{
    [super init];

    child_num = MINIBOSS0_RAILGUN_TURRET;

    health = RAILGUN_HEALTH;
    critical_health = RAILGUN_HEALTH/2;

    sprite = [boss_sebum getSebumByName:"railgun"];
    w = [sprite width];
    h = [sprite height];

    shot_delay = 40;
    return self;
}

- (void) die
{
    [super die];
    spawn_candy([RailgunTurretDebris class], 
		x+weapon_displacements[child_num].x,
		y+weapon_displacements[child_num].y,
		HIGH_LAYER);
}

- (void) fire
{
    if (--fire_tics <= 0) {
	Railgun *proj;
	fire_tics = shot_delay;

	proj = (Railgun *)spawn_projectile([Railgun class],
					   x + 25*sin(angle+M_PI_2),
					   y + 25*cos(angle+M_PI_2), NO);
	[[proj setAngle:angle] setRailgunTrailVelocity:[parent velocity]];
    }
}

- (void) findDesiredAngle
{
    if (fire_tics >= shot_delay - 10)
	return;
    else {
	if (fire_tics < 10)
	    max_rotation_angle = deg2rad(1.0);
	else
	    max_rotation_angle = deg2rad(4.0);

	[super findDesiredAngle];
    }
}
@end


@implementation RailgunTurretDebris
- init
{
    [super init];
    health = max_health = 100;
    sprite = [boss_sebum getSebumByName:"railgun"];
    return self;
}
@end

/*------------------------------------------------------------*/

@implementation MiniBoss0Tracks
- init
{
    [super init];

    sprite = [boss_sebum getSebumByName:"tracks"];
    w = [sprite width];
    h = [sprite height];
    health = TRACK_HEALTH;

    return self;
}

- (void) draw:(BITMAP *)dest
{
    [sprite drawTo:dest
	    X:x - offsetX - [sprite width]/2
	    Y:y - offsetY - [sprite height]/2
	    Alpha:0xff * health / TRACK_HEALTH];

    health--;
}
@end
