/* khan.m,
 *
 * A big tank.  Water/snow level boss.
 */

#include <allegro.h>
#include <math.h>
#include "candy.h"
#include "candy/explosion.h"
#include "candy/large-chunk.h"
#include "candy/zeus.h"
#include "collision.h"
#include "common.h"
#include "end-level.h"
#include "map.h"
#include "nuke.h"
#include "player.h"
#include "projectiles/evil-vulcan.h"
#include "projectiles/fireball.h"
#include "projectiles/lightning-gun.h"
#include "rotate.h"
#include "seborrhea/seborrhea-allegro.h" /* XXX */
#include "seborrhea/seborrhea.h"
#include "sound.h"
#include "unit-seeking.h"
#include "units/all-units.h"
#include "units/basic-turret.h"
#include "units/khan.h"
#include "units/spirtle.h"
#include "weapon.h"


#define TURRET_HEALTH		7500
#define CANNON_HEALTH		5000
#define CANNON_FIRE_DELAY	150
#define CANNON_CHARGE_TICS	(34*2)
#define CANNON_HOLD_TICS	50

static SebFile *boss_sebum;
static SAMPLE *shot_sample;	/* XXX */

/*--------------------------------------------------------------*/

@protocol BossChild
- (int) health;
@end

@interface Khan (Private)
- (void) updatePhase1;
- (void) updatePhase2;
- (void) initPhase3;
- (void) updatePhase3;
- (void) updatePhase4;
- (void) relocateChildren;
@end

@interface NazcaCannon: Unit <BossChild>
{
    int fire_tics;
    double relax_y;		/* Pixels to move after kickback. */
    BOOL charging;
    BOOL virgin_shot;
}
- (void) readyToFire;
@end

@interface NazcaCharge: AnimatedCandyNoLoop
@end

@interface NazcaShot: Projectile
@end

@interface NazcaDebriLeft: Candy
{
    double theta, omega;
}
@end

@interface NazcaDebriRight: NazcaDebriLeft
@end

@interface KhanTurret: Unit <BossChild>
{
    int hold_tics;		/* Tics to wait after firing. */
    int firing_sequence;
    int next_sweep_dir;
    DualWeapon *gun;
}
- (void) readyToFire;
@end

@interface KhanTurret (Private)
- (BOOL) checkPlayerInSweepZone;
- (void) initSweep;
- (void) updateSweepFromRight;
- (void) updateSweepFromLeft;
- (void) initTrack;
- (void) updateTrack;
@end

@interface KhanVulcanAbstract: BasicTurret
{
    Sebum<SebImage> *hide;
    int open_tics;
}
- (void) readyToFire;
@end

@interface KhanVulcanLeft: KhanVulcanAbstract
@end

@interface KhanVulcanRight: KhanVulcanAbstract
@end

/*--------------------------------------------------------------*/

@interface KhanLegs: Candy
- (void) breakup;
@end


@implementation Khan
+ (BOOL) loadPrerequisites
{
    return (LOAD_PROJECTILE_DATA_FOR(EvilVulcan) &&
	    LOAD_PROJECTILE_DATA_FOR(Fireball) &&
	    LOAD_PROJECTILE_DATA_FOR(LightningGun) &&
	    LOAD_UNIT_DATA_FOR(Spirtle, YES));
}

+ derive_loadData(boss_sebum, "data/khan");
+ derive_shutdown(boss_sebum);

- init
{
    [super init];

    sprite = [boss_sebum getSebumByName:"body"];
    w = [sprite width];
    h = [sprite height];

    chunk_colours = CHUNK_BLUE;
    flags |= FLAG_INVINCIBLE;
    return self;
}

- (void) activate
{
#define SPAWN_CHILD(c,x,y)	spawn_unit([c class], x, y, ACTIVE_GROUND_LIST, NO)

    boss = self;

    legs = spawn_candy([KhanLegs class], x, y-10, FLOOR_LAYER);
    child[KHAN_VULCAN_L] = SPAWN_CHILD(KhanVulcanLeft,  x, y);
    child[KHAN_VULCAN_R] = SPAWN_CHILD(KhanVulcanRight, x, y);
    child[KHAN_CANNON] = SPAWN_CHILD(NazcaCannon, x, y+50+10);
    child[KHAN_TURRET] = SPAWN_CHILD(KhanTurret,  x, y-50);

#undef SPAWN_CHILD
}

- (void) spawnDyingExplosions
{
    if (die_time == 8*SECONDS) {
	spawn_candy([LargeChunk class], x, y-50, LOW_LAYER);
	end_level(5*SECONDS);
	die_time++;
    }
    else if (die_time < 8*SECONDS) {
	int r = 25 + die_time * 75/(8*SECONDS);
	int xx = r*cos(die_time*8*M_PI/(8*SECONDS));
	int yy = r*sin(die_time*8*M_PI/(8*SECONDS)) - 50;

	spawn_candy([Explosion class],
		    x+rnd(-50,50),
		    y+rnd(-50,50) - 50,
		    LOW_LAYER);

	spawn_candy([BigExplosion class], x+xx, y+yy, LOW_LAYER);
	spawn_candy([LargeChunk class],   x+xx, y+yy, LOW_LAYER);

	die_time++;

	if ((die_time & 0x0f) == 0x0f)
	    play_explosion_sample(x);
    }
}

- (void) drawHealthMeter:(BITMAP *)dest
{
    BITMAP *bg, *fg;
    int ww;

    if (boss_phase < KHAN_RELEASE) {
	bg = [(SebImageAllegro *)[base_sebum getSebumByName:"meter/bg-purple"] bitmap];
	fg = [(SebImageAllegro *)[base_sebum getSebumByName:"meter/fg-red"] bitmap];
	ww = 75 * [(<BossChild>)child[KHAN_CANNON] health] / CANNON_HEALTH;
    }
    else {
	bg = [(SebImageAllegro *)[base_sebum getSebumByName:"meter/bg-gray"] bitmap];
	fg = [(SebImageAllegro *)[base_sebum getSebumByName:"meter/fg-purple"] bitmap];
	ww = 75 * [(<BossChild>)child[KHAN_TURRET] health] / TURRET_HEALTH;
    }

    blit(bg, dest, 0, 0, dest->w/2-75, dest->h-10, 150, 8);
    blit(fg, dest, 75-ww, 1, dest->w/2-ww, dest->h-9, 2*ww, 6);
}

- (enum THING_UPDATE_STATE) update
{
    switch (boss_phase) {
      case KHAN_STOP_AT_CANNON:
	  [self updatePhase1];
	  break;
      case KHAN_FIRE_CANNON:
	  [self updatePhase2];
	  break;
      case KHAN_RELEASE:
	  [self updatePhase3];
	  break;
      case KHAN_RETREAT:
	  [self updatePhase4];
	  break;
      case KHAN_FIRE_TURRET:
	  /* Do nothing. */
	  break;
    }

    return [super update];
}

- (void) disownChild:(Unit *)ch
{
    unsigned int c;

    if (ch == child[KHAN_TURRET]) {
	flags |= FLAG_DYING;
	sprite = [boss_sebum getSebumByName:"body3"];
	boss = nil;

	for (c = 0; c < KHAN_NUM_CHILDREN; c++) {
	    if (child[c]) {
		[child[c] delete];
		child[c] = nil;
	    }
	}

	return;
    }

    for (c = 0; c < KHAN_NUM_CHILDREN; c++) {
	if (child[c] == ch) {
	    child[c] = nil;
	    break;
	}
    }

    if (c == KHAN_CANNON) {
	if (vulcan_activation_tics < 250) {
	    if (child[KHAN_VULCAN_L])
		[(KhanVulcanAbstract *)child[KHAN_VULCAN_L] readyToFire];

	    if (child[KHAN_VULCAN_R])
		[(KhanVulcanAbstract *)child[KHAN_VULCAN_R] readyToFire];
	}

	[self initPhase3];
    }
}

- (int) collisionLists
{
    /* Don't collide with anything. Die when all of the children are
       dead. */
    return 0;
}

- derive_setXYAlwaysCentred;
@end


@implementation Khan (Private)
- (void) updatePhase1
{
    if (offsetY < y-10) {
	[(NazcaCannon *)child[KHAN_CANNON] readyToFire];
	[current_map setScrollRate:0.0 :YES];
	boss_phase++;
    }
}

- (void) updatePhase2
{
    if (vulcan_activation_tics < 250) {
	vulcan_activation_tics++;
    }
    else {
	return;
    }

    if (vulcan_activation_tics == 250) {
	if (child[KHAN_VULCAN_L]) {
	    [(KhanVulcanAbstract *)child[KHAN_VULCAN_L] readyToFire];
	}

	if (child[KHAN_VULCAN_R]) {
	    [(KhanVulcanAbstract *)child[KHAN_VULCAN_R] readyToFire];
	}
    }
}

- (void) initPhase3
{
    yv = -8.0;

    sprite = [boss_sebum getSebumByName:"body2"];

    [(KhanLegs *)legs breakup];
    legs = nil;

    [current_map setScrollRate:-0.80 :YES];
    boss_phase++;
    retreat_tics = 200;
}

- (void) updatePhase3
{
    retreat_tics--;

    y += yv;
    yv *= 0.95;
    [self relocateChildren];

    if (retreat_tics > 200);
    else if (retreat_tics > 100) {
	if ((retreat_tics % 6) == 0) {
	    int z = 30 * (retreat_tics-100) / 100;

	    /* Left leg. */
	    spawn_candy([Explosion class], x-150+z, y-7-z, LOW_LAYER);

	    /* Right leg. */
	    spawn_candy([Explosion class], x+150-z, y-7-z, LOW_LAYER);
	}
    }
    else if (retreat_tics > 0);
    else {
	[(KhanTurret *)child[KHAN_TURRET] readyToFire];
	[current_map setScrollRate:-0.30 :YES];
	boss_phase++;
    }
}

- (void) updatePhase4
{
    retreat_tics--;

    if (retreat_tics > -2300) {
	y -= 0.20;
	[self relocateChildren];

	if (retreat_tics == -1800) {
	    [current_map setScrollRate:0.0 :YES];
	}
    }
    else {
	boss_phase++;
    }
}

- (void) relocateChildren
{
    [child[KHAN_TURRET] setX:x Y:y-50];
    if (child[KHAN_VULCAN_L]) [child[KHAN_VULCAN_L] setX:x Y:y];
    if (child[KHAN_VULCAN_R]) [child[KHAN_VULCAN_R] setX:x Y:y];
}
@end


@implementation KhanLegs
- init
{
    [super init];
    sprite = [boss_sebum getSebumByName:"legs"];
    h = [sprite height] / 2;
    return self;
}

- (void) breakup
{
    int i;

    for (i = 0; i < 5; i++) {
	spawn_candy([LargeChunk class], x-125, y+25, LOW_LAYER);
	spawn_candy([LargeChunk class], x+125, y+25, LOW_LAYER);
    }

    sprite = [boss_sebum getSebumByName:"legs-d"];
}

- (BOOL) mayStillInfluenceGameplay
{
    return (y-offsetY <= screen_h + h/2);
}
@end


/*--------------------------------------------------------------*/
/* Nazca Cannon.						*/
/*--------------------------------------------------------------*/

@implementation NazcaCannon
- init
{
    [super init];

    health = CANNON_HEALTH;
    critical_health = 2000;

    sprite = [boss_sebum getSebumByName:"cannon"];
    w = 68;
    h = 118;

    fire_tics = CANNON_FIRE_DELAY;
    virgin_shot = YES;

    flags |= FLAG_INVINCIBLE;
    return self;
}

- (int) health
{
    return health;
}

- (void) readyToFire
{
    flags &=~FLAG_INVINCIBLE;
    flags |= FLAG_FIRING_ENABLED;
}

- (void) spawnDyingExplosions
{
    if (rnd(0, 75*3) <= -health) {
	spawn_candy([BigExplosion class],
		    rnd(x-20, x+20), rnd(y-40, y+40), LOW_LAYER);
    }

    /* Spawn chunks every 16 updates. */
    if ((health & 0x0f) == 0x0f) {
	[self burstIntoChunks];
	play_explosion_sample(x);
    }

    health--;
}

- (void) delete
{
    game_flags |= FLAG_SCREEN_SHAKE;

    spawn_nuke(x, y+40, -1);

    spawn_candy([BigExplosion class], x-10, y+40, LOW_LAYER);
    spawn_candy([BigExplosion class], x,    y+40, LOW_LAYER);
    spawn_candy([BigExplosion class], x+10, y+40, LOW_LAYER);

    spawn_candy([NazcaDebriLeft  class], x-57, y, LOW_LAYER);
    spawn_candy([NazcaDebriRight class], x+57, y, LOW_LAYER);

    if (shot_sample) {
	stop_sample(shot_sample);	/* XXX */
	shot_sample = NULL;
    }

    [self burstIntoChunks];
    [super delete];
}

- (void) draw:(BITMAP *)dest
{
    Sebum<SebImage> *stabiliser = [boss_sebum getSebumByName:"stabiliser"];
    double delta_x = 0;

    if (fire_tics <= 0) {
	double t = (double)-fire_tics/CANNON_HOLD_TICS;

	/* Two slightly different frequency cosine functions for
	   jitter, and a multiplier for diminishing amplitude. */
	delta_x = (cos(2*M_PI*5 * t) + cos(2*M_PI*6 * t)) *
	    (CANNON_HOLD_TICS+fire_tics)/CANNON_HOLD_TICS;
    }

    [stabiliser drawTo:dest
		X:x + delta_x*0.40 - [stabiliser  width]/2 - offsetX
		Y:y + relax_y*0.75 - [stabiliser height]/2 - offsetY - 50];

    /* Dodgey. */
    x += delta_x;
    [super draw:dest];
    x -= delta_x;
}

- (enum THING_UPDATE_STATE) update
{
    if ((fire_tics > 0) &&
	(relax_y > 0.0)) {
	y += 0.5;
	relax_y -= 0.5;
    }

    /* If started charging, but players died, fire anyway. */
    if (not (game_flags & FLAG_PLAYERS_ALIVE) &&
	fire_tics < CANNON_CHARGE_TICS)
	[self fire];

    /* Fire one last shot when we're dying, which results in a
       disaster. */
    if (flags & FLAG_DYING) {
	[self spawnDyingExplosions];
	[self fire];

	if (fire_tics == 0) {
	    [(Khan *)boss disownChild:self];
	    [self delete];
	}
    }

    return [super update];
}

- (void) fire
{
    fire_tics--;

    if (fire_tics <= -CANNON_HOLD_TICS) {
	if (shot_sample) {
	    stop_sample(shot_sample);	/* XXX */
	    shot_sample = NULL;
	}

	fire_tics = CANNON_FIRE_DELAY;
    }
    else if (fire_tics <= 0) {
	if (charging) {
	    shot_sample = [(SebSampleAllegro *)[boss_sebum getSebumByName:"snd2"] sample];
	    if (sound_vol > 0)
		play_sample(shot_sample, 192*sound_vol/255, 128, 1000, 1);

	    spawn_projectile([NazcaShot class], x, y+223, NO);

	    relax_y = 8;
	    y -= relax_y;

	    charging = NO;
	    virgin_shot = NO;
	}
    }
    else if ((fire_tics <= CANNON_CHARGE_TICS) &&
	     (not charging)) {
	play_voice_on_channels((Sebum<SebSample> *)[boss_sebum getSebumByName:"snd"],
			       x, 255, PRIMARY_WEAPON_CHANNELS);
	spawn_candy([NazcaCharge class], x, y+80, MEDIUM_LAYER);
	charging = YES;
    }

    /* Spawn Spirtles. */
    if (!virgin_shot &&
	(fire_tics == 46 || fire_tics == 49 ||
	 fire_tics == 52 || fire_tics == 55)) {
#define SPAWN_UNIT	(SuperSpirtle *)spawn_unit
	int xx;
	SuperSpirtle *ss;

	xx = ((fire_tics == 55) ? 15 :
	      (fire_tics == 55) ? 15 :
	      (fire_tics == 52) ? 35 :
	      (fire_tics == 49) ? 55 : 75);


	ss = SPAWN_UNIT([SuperSpirtle class],
			-150, offsetY+40, ACTIVE_AIR_LIST, YES);
	[ss setDestX:xx :NO];

	ss = SPAWN_UNIT([SuperSpirtle class],
			screen_w+150, offsetY+40, ACTIVE_AIR_LIST, YES);
	[ss setDestX:screen_w-xx :YES];

#undef SPAWN_UNIT
    }
}

- (int) collisionLists
{
    if (charging) {
	/* This is for the charging ball. */
	return COLLIDES_WITH_PROJECTILES|ALLY_LIST;
    }
    else {
	return COLLIDES_WITH_PROJECTILES;
    }
}

- (int) collisionRoutinePriority
{
    /* Higher priority than toothpaste. */
    return 85;
}

- (BOOL) checkCollisionWith:(Thing<DetectsCollision> *)object
{
    /* Do a very small bounding box collision detection, around the
       tip of the barrel. */
    double x2, y2;
    int w2, h2;

    [object getX:&x2 Y:&y2 W:&w2 H:&h2];

    return bounding_box_collision(x-16.0, y+51.0, x+16.0, y+59.0,
				  x2-w2/2.0, y2-h2/2.0, x2+w2/2.0, y2+h2/2.0);
}

- (int) receiveDamage:(int)damage type:(enum DAMAGE_TYPE)type
{
    if (type == DAMAGE_TYPE_UNIT) {
	return 0;
    }

    return [super receiveDamage:damage type:type];
}

- (void) getX:(double *)x_ Y:(double *)y_
{
    /* Hack for toothpaste. */
    *x_ = x;
    *y_ = y + (51.0+59.0)/2.0;
}
@end


@implementation NazcaCharge
- init
{
    [super init];
    sprite = [[SebAnimator new] setAnimation:(SebAnimation *)
				[boss_sebum getSebumByName:"charge"]];
    return self;
}
@end


@implementation NazcaShot
- init
{
    [super init];

    health = CANNON_HOLD_TICS;
    speed = 0.0;

    sprite = [boss_sebum getSebumByName:"fire"];
    w = [sprite width];
    h = [sprite height];

    return self;
}

- (void) die
{
    /* Don't die when we kill something.  Die when we run out of
       time. */
}

+ (int) collisionLists
{
    /* Don't collide with nukes. */
    return ALLY_LIST;
}

- (int) damage
{
    /* Insta-kill robots, etc. */
    return 666;
}

- (enum THING_UPDATE_STATE) update
{
    health--;
    return [super update];
}

- (void) draw:(BITMAP *)dest
{
    /* Flicker. */
    if ((health > 14) || (health & 0x02)) {
	[super draw:dest];
    }
}

- (int) collisionRoutinePriority
{
    /* Higher priority than normal units, lower than nukes (although
       it shouldn't matter). */
    return 95;
}

- (BOOL) checkCollisionWith:(Thing<DetectsCollision> *)object
{
    if ([(Unit *)object flags] & (FLAG_DYING | FLAG_INVINCIBLE)) {
	return NO;
    }
    else {
	int y_top;
	double ux, uy;
	int uw, uh;

	[object getX:&ux Y:&uy W:&uw H:&uh];

	y_top = y+5 - h/2;
	
	if (uy+uh/2 < y_top) {
	    return NO;
	}
	else if (uy >= y_top + 50) {
	    /* The bottom of the beam is a rectangle. */
	    return ((x-w/2 < ux+uw/2) && (ux-uw/2 < x+w/2));
	}
	else {
	    /* Approximate the top of the beam as a quadratic. */
	    int yy;

	    /* Lower right of player. */
	    yy = y_top + SQ(ux+uw/2-x)/75;
	    if (uy+uh/2 > yy)
		return YES;

	    /* Lower left of player. */
	    yy = y_top + SQ(ux-uw/2-x)/75;
	    if (uy+uh/2 > yy)
		return YES;

	    return NO;
	}
    }
}

- (BOOL) collidesWith:(Thing<DetectsCollision> *)object
{
    /* Using 'collidesWith' instead of 'checkCollisionWith' is correct
       because we want a screenshake and some candy (ir-)regardless of
       who did the collision detection. */
    if ([super collidesWith:object]) {
	double xx, yy;
	[object getX:&xx Y:&yy];
	[SPAWN_CANDY(ZeusEffect, x, y-h/2+15, HIGH_LAYER)
		    setDestX:xx Y:yy];

	game_flags |= FLAG_SCREEN_SHAKE;
	return YES;
    }
    else {
	return NO;
    }
}

#ifdef DEBUG_BOUNDING_BOX
- (void) drawBoundingBox:(BITMAP *)dest
{
    int i, yy;
    for (i = -(signed)w/2; i < (signed)w/2; i++) {
	yy = y-h/2 + 5 + i*i/75;
	putpixel(dest, x+i-offsetX, yy-offsetY, BOUNDING_BOX_COLOUR);
    }
}
#endif
@end


@implementation NazcaDebriLeft
- init
{
    [super init];

    speed = 3.0;
    angle = deg2rad(190.0);
    theta = 0.0;
    omega = deg2rad(-6.0);

    sprite = [boss_sebum getSebumByName:"cannon-dl"];

    return self;
}

- (enum THING_UPDATE_STATE) update
{
    theta += omega;
    return [super update];
}

- (void) draw:(BITMAP *)dest
{
    [sprite drawTo:dest X:x-offsetX Y:y-offsetY Angle:theta];
}
@end


@implementation NazcaDebriRight
- init
{
    [super init];

    angle = deg2rad(30.0);
    omega = deg2rad(6.0);
    
    sprite = [boss_sebum getSebumByName:"cannon-dr"];

    return self;
}
@end


/*--------------------------------------------------------------*/
/* Khan Turret.							*/
/*--------------------------------------------------------------*/

@implementation KhanTurret
- init
{
    [super init];

    health = TURRET_HEALTH;
    critical_health = 2250;
    speed = 0.0;

    sprite = [boss_sebum getSebumByName:"turret"];
    w = 56; /* [sprite width]; */
    h = 78; /* [sprite height]; */
    rotated_sprite = YES;

    gun = [[DualWeapon newWithProjectile:[LightningGun class]]
	      setXDisplacement:33 YDisplacement:11];

    flags |= FLAG_INVINCIBLE;
    return self;
}

- (int) health
{
    return health;
}

- (void) readyToFire
{
    hold_tics = 250;
    flags |= FLAG_FIRING_ENABLED;
}

- (void) die
{
    [super die];
    [(Khan *)boss disownChild:self];

    [self burstIntoChunks];
    flags |= FLAG_DEAD;
}

- (enum THING_UPDATE_STATE) update
{
    if (flags & FLAG_DEAD)
	return THING_DEAD;

    if (hold_tics > 0) {
	hold_tics--;

	if (hold_tics == 0)
	    flags &=~FLAG_INVINCIBLE;
	return THING_NORMAL;
    }

    if (flags & FLAG_INVINCIBLE) {
	return THING_NORMAL;
    }

    if (firing_sequence == 0) {
	[self initSweep];
    }
    else if (firing_sequence < 20) {
	if ([self checkPlayerInSweepZone]) {
	    [self updateSweepFromRight];
	}
	else {
	    [self initTrack];
	}
    }
    else if (firing_sequence < 30) {
	if ([self checkPlayerInSweepZone]) {
	    [self updateSweepFromLeft];
	}
	else {
	    [self initTrack];
	}
    }
    else {
	[self updateTrack];
    }

    simplify_angle(&angle);

    return [super update];
}

- (void) fire
{
    if ([gun fireFromX:x Y:y Angle:angle]) {
	play_panned_sample((Sebum<SebSample> *)[boss_sebum getSebumByName:"zap"], x);
	hold_tics = 8;

	/* Special: In this mode we only fire when *VERY* close to the
	   target. */
	if (firing_sequence >= 30) {
	    flags &=~FLAG_FIRING_ENABLED;

	    firing_sequence++;
	    if (firing_sequence > 36) {
		firing_sequence = 0;
	    }
	}
    }
}
@end


@implementation KhanTurret (Private)
- (BOOL) checkPlayerInSweepZone
{
    unsigned int p;
    double px, py;

    for (p = 0; p < num_players; p++) {
	if (player[p]) {
	    [player[p] getX:&px Y:&py];

	    if ((py > y) &&
		(SQ(px-x) < SQ(py-y+10))) {
		return YES;
	    }
	}
    }

    return NO;
}

- (void) initSweep
{
    if (not [self checkPlayerInSweepZone]) {
	[self initTrack];
	firing_sequence = 99;	/* Single shot. */
	return;
    }

    [[gun setShotDelay:10 WaveDelay:5] setShotsPerWave:2];

    if (next_sweep_dir == 0) {
	firing_sequence = 10;
	[gun setXDisplacement:33 YDisplacement:11];
    }
    else {
	firing_sequence = 20;
	[gun setXDisplacement:33 YDisplacement:-11];
    }

    flags &=~FLAG_FIRING_ENABLED;
    next_sweep_dir = !next_sweep_dir;
}

#define SWEEP_CW(omega)	limited_rotate(&angle, deg2rad(-135.0), omega)
#define SWEEP_CCW(omega)	limited_rotate(&angle, deg2rad(-45.0), omega)

- (void) updateSweepFromRight
{
    if (firing_sequence == 10) {
	if (SWEEP_CCW(deg2rad(2.5))) {
	    firing_sequence++;
	    [gun setFireTics:1];
	    flags |= FLAG_FIRING_ENABLED;
	}
    }
    elif (firing_sequence == 11) {
	if (SWEEP_CW(deg2rad(1.0))) {
	    [gun setShotDelay:5 WaveDelay:10];
	    firing_sequence++;
	}
    }
    elif (firing_sequence == 12) {
	if (SWEEP_CCW(deg2rad(1.0))) {
	    [self initTrack];
	}
    }
}

- (void) updateSweepFromLeft
{
    if (firing_sequence == 20) {
	if (SWEEP_CW(deg2rad(2.5))) {
	    firing_sequence++;
	    [gun setFireTics:1];
	    flags |= FLAG_FIRING_ENABLED;
	}
    }
    elif (firing_sequence == 21) {
	if (SWEEP_CCW(deg2rad(1.0))) {
	    [gun setShotDelay:5 WaveDelay:10];
	    firing_sequence++;
	}
    }
    elif (firing_sequence == 22) {
	if (SWEEP_CW(deg2rad(1.0))) {
	    [self initTrack];
	}
    }
}

#undef SWEEP_CCW
#undef SWEEP_CW

- (void) initTrack
{
    [[gun setShotDelay:-3 WaveDelay:15] setShotsPerWave:1];
    firing_sequence = 30;
}

- (void) updateTrack
{
    Unit *target;
    double theta_desired;

    /* If we're close to firing, hold still. */
    if ((flags & FLAG_FIRING_ENABLED) &&
	([gun fireTics] < 3)) {
	return;
    }

    target = find_closest_ally_according_to_rotation(x, y, angle);
    theta_desired = angle_towards_unit(x, y, target);
    limited_rotate(&angle, theta_desired, deg2rad(2.0));
    simplify_angle(&angle);

    if (ABS(angle - theta_desired) < deg2rad(0.5)) {
	flags |= FLAG_FIRING_ENABLED;
    }
}
@end


/*--------------------------------------------------------------*/
/* Vulcan Turrets.						*/
/*--------------------------------------------------------------*/

@implementation KhanVulcanAbstract
- init
{
    [super init];

    health = 3000;
    critical_health = 1500;
    speed = 0.0;

    sprite = [[SebAnimator new] setAnimation:(SebAnimation *)
				[boss_sebum getSebumByName:"vulcan"]];
    hide = [[SebAnimatorManualWithDelay new] setAnimation:(SebAnimation *)
					     [boss_sebum getSebumByName:"open"]];
    w = [sprite width];
    h = [sprite height];
    rotated_sprite = YES;

    gun = [[[[[OffcentrePausePulseWeapon newWithProjectile:[EvilVulcan class]]
		 setDifficultyModifier:NO]
		setXDisplacement:25 YDisplacement:0]
	       setShotDelay:2 WaveDelay:118*2+24*2+8*3]
	      setShotsPerWave:24];

    /* We stopped scrolling, so don't aim above the player. */
    aim_delta_y = 0;

    /* Starts off disabled. */
    flags |= FLAG_INVINCIBLE;
    flags &=~FLAG_FIRING_ENABLED;
    return self;
}

- free
{
    hide = [hide free];
    return [super free];
}

- (void) die
{
    [super die];
    [(Khan *)boss disownChild:self];	/* XXX */
}

- (void) fire
{
    if ([gun fireFromX:x Y:y Angle:angle]) {
	if ([(<PulseWeapon>)gun currentShot] & 0x02) {
	    [(SebAnimator *)sprite resetAnimation];
	}
    }
}

- (enum THING_UPDATE_STATE) update
{
    if (open_tics > 0) {
	open_tics--;
	if (open_tics == 0) {
	    angle = -M_PI_2;
	    hide = [hide free];
	    flags &=~FLAG_INVINCIBLE;
	    flags |= FLAG_FIRING_ENABLED;
	}
	else if (open_tics <= 15*2) {
	    [(SebAnimatorManualWithDelay *)hide nextFrame];
	}
    }

    return [super update];
}

- (void) draw:(BITMAP *)dest
{
    if (hide) {
	[hide drawTo:dest
	      X:x-[hide  width]/2-offsetX
	      Y:y-[hide height]/2-offsetY+7];
    }
    else {
	[super draw:dest];
    }
}

- (void) readyToFire
{
    open_tics = 200+15*2;
}
@end


@implementation KhanVulcanLeft
- init
{
    [super init];
    [gun setFireTics:56+150];
    return self;
}

- setX:(double)x_ Y:(double)y_
{
    x = x_ - 105;
    y = y_ + 59;
    return self;
}
@end


@implementation KhanVulcanRight
- init
{
    [super init];
    [gun setFireTics:26+300];
    return self;
}

- setX:(double)x_ Y:(double)y_
{
    x = x_ + 105;
    y = y_ + 59;
    return self;
}

- (void) readyToFire
{
    [super readyToFire];
    open_tics += 50;
}
@end
