/* hatch.m,
 *
 * Bunkers for tanks and stuff.  Good for keeping units safe until
 * they activate.
 */

#include <allegro.h>
#include <assert.h>
#include <math.h>
#include "common.h"
#include "map-editor/unit-editor.h"
#include "map-editor/unit-extras.h"
#include "map.h"
#include "seborrhea/seborrhea.h"
#include "sound.h"
#include "units/all-units.h"
#include "units/hatch.h"


#define INITIAL_SPAWN_TICS	(6*8)

static SebFile *hatch_sebum;
static SebAnimation *open_anim;


/*--------------------------------------------------------------*/

@interface Hatch (Private)
- (void) spawnUnit;
@end

/*--------------------------------------------------------------*/

@implementation Hatch
        /* Initialization. */
+ (BOOL) loadData
{
    if (not [self loadData:&hatch_sebum :"data/hatch"])
	return NO;

    open_anim = (SebAnimation *)[hatch_sebum getSebumByName:"open-anim"];
    return YES;
}

+ (void) shutdown
{
    FREE_SEBFILE(hatch_sebum);
    open_anim = nil;
}

- init
{
    [super init];

    sprite = [[SebAnimatorManualWithDelay new] setAnimation:open_anim];
    w = [sprite width];
    h = [sprite height];

    spawn_tics = INITIAL_SPAWN_TICS;
    spawnee = nil;

    flags |= FLAG_INVINCIBLE;	/* Don't get found by find_closest_unit. */
    return self;
}

	/* Update. */
- (enum THING_UPDATE_STATE) update
{
    /* If all players are dead, don't open the hatch unless we have
       already begun. */
    if (!(game_flags & FLAG_PLAYERS_ALIVE) &&
	[(SebAnimatorManualWithDelay *)sprite currentFrameNumber] == 0)
	return THING_NORMAL;

    if (spawnee && (y + activate_y >= offsetY)) {
	if (spawn_tics == INITIAL_SPAWN_TICS) {
	    play_panned_sample((Sebum<SebSample> *)[hatch_sebum getSebumByName:"snd"], x);
	}

	spawn_tics--;
	if (spawn_tics <= 0)
	    [self spawnUnit];

	[(SebAnimatorManualWithDelay *)sprite nextFrame];

	//alpha = 255 * (anim.frame * anim.delay + 6-anim.tics) / ([open_anim numFrames] * anim.delay);
	alpha = 255 - 255 * spawn_tics/INITIAL_SPAWN_TICS;
    }

    return [super update];
}

	/* Collisions. */
- (int) collisionLists { return 0; }
@end


@implementation Hatch (Private)
- (void) spawnUnit
{
    assert(spawnee != nil);
    [spawn_unit(spawnee, x, y, ACTIVE_GROUND_LIST, NO) setAngle:angle];
    spawnee = nil;
}
@end

/*------------------------------------------------------------*/

@implementation HatchWithVariableTravelRange
	/* Load/Save.  Format: travel_range;activate_y;x;y;a[;flags] */
- (void) importUnitProperties:(char *)str
{
    char str2[1024];
    sscanf(str, "%d;%s", &travel_range, str2);
    [super importUnitProperties:str2];
}

- (char *) exportUnitProperties:(char *)str
{
    char str2[1024] = "\0";

    [super exportUnitProperties:str2];
    snprintf(str, sizeof(str2), "%d;%s", travel_range, str2);
    return str;
}

	/* Protocol stuff. */
- derive_setTravelRange(travel_range);
@end


@implementation HatchWithVariableTravelRange (Private)
- (void) spawnUnit
{
    assert(spawnee != nil);
    [[spawn_unit(spawnee, x, y, ACTIVE_GROUND_LIST, NO)
		setAngle:angle]
	setTravelRange:travel_range];
    spawnee = nil;
}
@end


@implementation HatchWithVariableTravelRange (MapEditor)
- (void) drawMapEditorExtras:(BITMAP *)dest
{
    [super drawMapEditorExtras:dest];
    draw_travel_range(dest, x, y, angle, travel_range);
}
@end
