/* elephant.m,
 *
 * A very heavily armoured tank with a fixed turret.  Note that for
 * ElephantReverse, the gun is antiparallel to its angle.
 */

#include <math.h>
#include "candy/tank-debris.h"
#include "candy/tank-track.h"
#include "common.h"
#include "map.h"
#include "projectiles/plasma.h"
#include "seborrhea/seborrhea.h"
#include "units/all-units.h"
#include "units/elephant.h"


static SebFile *tank_data;
static SebAnimation *move_anim, *fire_anim, *move_animr, *fire_animr;

/*------------------------------------------------------------*/

@interface ElephantTrack: TankTrack
@end

@interface ElephantTrackCCW: ElephantTrack
@end

/*------------------------------------------------------------*/

@implementation Elephant
+ (BOOL) loadPrerequisites
{
    return LOAD_PROJECTILE_DATA_FOR(Plasma);
}

+ (BOOL) loadData
{
    if (not [self loadData:&tank_data :"data/elephant"])
	return NO;

    move_anim  = (SebAnimation *)[tank_data getSebumByName:"move-anim"];
    fire_anim  = (SebAnimation *)[tank_data getSebumByName:"fire-anim"];
    move_animr = (SebAnimation *)[tank_data getSebumByName:"move-animr"];
    fire_animr = (SebAnimation *)[tank_data getSebumByName:"fire-animr"];
    return YES;
}

+ (void) shutdown
{
    FREE_SEBFILE(tank_data);
    move_anim  = nil;
    fire_anim  = nil;
    move_animr = nil;
    fire_animr = nil;
}

- init
{
    [super init];

    health = 300;
    critical_health = 150;
    speed = 0.60;
    the_way = (SebSpline *)[tank_data getSebumByName:"path"];
    activate_y = 0;

    sprite = [[SebAnimatorManualWithDelay new] setAnimation:move_anim];
    w = [sprite width];
    h = [sprite height];
    rotated_sprite = YES;

    gun = [[[OffcentreWeapon newWithProjectile:[Plasma class]]
	       setShotDelay:55]
	      setXDisplacement:25 YDisplacement:0];

    flags |= FLAG_FIRING_ENABLED;
    chunk_colours = CHUNK_COFFEE;
    return self;
}

- free
{
    gun = [gun free];
    return [super free];
}

- (void) die
{
    [super die];

    /* This is a big tank.  Spawn lots of debri. */
    spawn_candy([TankDebris class], x,    y-10, FLOOR_LAYER);
    spawn_candy([TankDebris class], x+10, y,    FLOOR_LAYER);
    spawn_candy([TankDebris class], x-10, y+10, FLOOR_LAYER);
}

- (void) move
{
    /* Spawn a track? */
    if (SQ(last_track_x - x) + SQ(last_track_y - y) >= SQ(11.0)) {
	BOOL cw = NO;

	if ((angle < 0  && not mirror_path) ||
	    (angle < M_PI_2 && mirror_path))
	    cw = YES;

	if (cw)
	    [spawn_candy([ElephantTrack class], x, y, FLOOR_LAYER) setAngle:angle];
	else
	    [spawn_candy([ElephantTrackCCW class], x, y, FLOOR_LAYER) setAngle:angle];

	last_track_x = x;
	last_track_y = y;
    }

    [super move];
}

- (void) fire
{
    if ([gun fireFromX:x Y:y Angle:angle]) {
	[(SebAnimatorManual *)sprite setAnimation:fire_anim];
    }
    else if ([(SebAnimatorManual *)sprite animationEnded])
	[(SebAnimatorManual *)sprite setAnimation:move_anim];
}

- (enum THING_UPDATE_STATE) update
{
    [(SebAnimatorManual *)sprite nextFrame];
    return [super update];
}

- (id) setX:(double)x_ Y:(double)y_
{
    /* This is really just to look good in the map editor. */
    if (x_ < screen_w/2)
	angle = 0;
    else
	angle = M_PI;

    return [super setX:x_ Y:y_];
}
@end

/*------------------------------------------------------------*/

@implementation ElephantReverse
+ (BOOL) loadPrerequisites
{
    return LOAD_UNIT_DATA_FOR(Elephant, NO);
}

- init
{
    [super init];

    angle = M_PI_2;
    the_way = (SebSpline *)[tank_data getSebumByName:"path2"];
    activate_y = -80;

    /* Note: Elephant has allocated sprite already. */
    [(SebAnimatorManual *)sprite setAnimation:move_animr];

    return self;
}

- (void) move
{
    if (the_way)
	[super move];
    else {
	[(SebAnimatorManual *)sprite setLoopMethod:LOOP_NONE];

	if (y > offsetY + screen_h + 50)
	    [self delete];
    }
}

- (void) fire
{
    if ([gun fireFromX:x Y:y Angle:angle + M_PI]) {
	[(SebAnimatorManual *)sprite setAnimation:fire_animr];
    }
    else if (the_way &&
	     [(SebAnimatorManual *)sprite animationEnded])
	[(SebAnimatorManual *)sprite setAnimation:move_animr];
}

- (id) setX:(double)x_ Y:(double)y_
{
    /* This is really just to look good in the map editor. */
    [super setX:x_ Y:y_];
    angle = M_PI_2;
    return self;
}
@end

/*------------------------------------------------------------*/

@implementation ElephantTrack
- init
{
    [super init];

    /* This is a HEAVY tank.  Make near-permanant marks! */
    health = max_track_life = 500;
    sprite = [tank_data getSebumByName:"tracks"];

    return self;
}
@end

@implementation ElephantTrackCCW
- init
{
    [super init];
    sprite = [tank_data getSebumByName:"tracks2"];
    return self;
}
@end
