/* dragonfly.m,
 *
 * Units that circle around the player.  The path is generated during
 * the game depending on where the player is.
 */

#include <assert.h>
#include <math.h>
#include "candy/speed-shadow.h"
#include "common.h"
#include "map.h"
#include "player.h"
#include "projectiles/fireball.h"
#include "seborrhea/seborrhea.h"
#include "unit-seeking.h"
#include "units/all-units.h"
#include "units/dragonfly.h"


static SebFile *unit_data;

/*--------------------------------------------------------------*/

@interface DragonFlyTrail: RotatedSpeedShadow
@end

/*------------------------------------------------------------*/

@implementation DragonFly
+ (BOOL) loadPrerequisites
{
    return LOAD_PROJECTILE_DATA_FOR(Fireball);
}

+ derive_loadData(unit_data, "data/dragonfly");
+ derive_shutdown(unit_data);

- init
{
    [super init];

    health = 35;

    sprite = [unit_data getSebumByName:"body"];
    shadow = [unit_data getSebumByName:"shadow"];
    w = 16;
    h = 16;
    rotated_sprite = YES;

    [gun free];				/* Mozzie creates a gun. */
    gun = [[[DualWeapon newWithProjectile:[Fireball class]]
	       setShotDelay:9 WaveDelay:18]
	      setXDisplacement:15 YDisplacement:4];
    
    chunk_colours = CHUNK_GREEN;
    return self;
}

- (BOOL) generatePath
{
#define CP1_SET_XY	[cp1 setX:x Y:y steps:DRAGONFLY_STEPS]
#define CP2_SET_XY	[cp2 setX:x_ Y:y_]

    double x_, y_;
    List *list;
    ControlPoint *cp1, *cp2;

    if (path_segment == 0) {
	target = find_closest_unit(x, y, ALLY_LIST);
	if (not target)
	    return NO;
    }
    else if (not unit_exists(target, ALLY_LIST)) {
	/* Target did exist, but died.  This is rare, so just die. */
	if (the_path) {
	    [self freePathSpline];
	    path_segment = 0;
	}
	[self die];
	return NO;
    }

    [target getX:&x_ Y:&y_];
    list = [LIST_NEW];
    cp1 = [ControlPoint new];
    cp2 = [ControlPoint new];

    switch (path_segment) {
      case 0:			/* Part 1: Above to left of player. */
	  x_ -= 75;
	  y_ += 25;
	  [CP1_SET_XY setControlPoint:1 X:x_-x Y:0];
	  [CP2_SET_XY setControlPoint:0 X:0 Y:y-y_];
	  break;
      case 1:			/* Part 2: Left to below player. */
	  x_ += 75;
	  y_ += 25;
	  [CP1_SET_XY setControlPoint:1 X:0 Y:y_-y];
	  [CP2_SET_XY setControlPoint:0 X:x_-x Y:0];
	  break;
      case 2:		       /* Part 3: Below to right of player. */
	  x_ += 75;
	  y_ -= 200;
	  [CP1_SET_XY setControlPoint:1 X:x-x_ Y:0];
	  [CP2_SET_XY setControlPoint:0 X:0 Y:y-y_];
	  break;
      case 3:			/* Part 4: Right to above player. */
	  x_ -= 75;
	  y_ -= 200;
	  [CP1_SET_XY setControlPoint:1 X:0 Y:y_-y];
	  [CP2_SET_XY setControlPoint:0 X:x-x_ Y:0];
	  break;
      case 4:			/* Part 5: Above to bottom of screen. */
	  y_ += screen_h;
	  [CP1_SET_XY setControlPoint:1 X:-50 Y:0];
	  [CP2_SET_XY setControlPoint:0 X:-50 Y:0];
	  break;
      default:
	  if (the_path) {
	      [self freePathSpline];
	      path_segment = 0;
	  }
	  return NO;
    }

    /* Articulating spline. */
    [list insertItem:cp2];
    [list insertItem:cp1];
    if (the_path)
	the_path = reticulate_spline(the_path, list);
    else
	the_path = articulate_spline(list);
    [list free];

    return YES;

#undef CP1_SET_XY
#undef CP2_SET_XY
}

- (void) move
{
    if (the_path) {
	[spawn_candy([DragonFlyTrail class], x, y, MEDIUM_LAYER) setAngle:angle];
	x = the_path[path_progress].x;
	y = the_path[path_progress].y;

	path_progress++;
	if (path_progress >= DRAGONFLY_STEPS-10) {
	    path_progress = 0;
	    path_segment++;
	    [self generatePath];
	    flags |= FLAG_FIRING_ENABLED;
	}

	if (unit_exists(target, ALLY_LIST)) {
	    /* Face the target. */
	    double x_, y_;
	    [target getX:&x_ Y:&y_];
	    angle = atan2(y-(y_-10), x_-x);
	}
	else
	    target = nil;
    }
    else {
	[self delete];
    }
}

- derive_alwaysInfluenceGameplay;
@end

/*--------------------------------------------------------------*/

@implementation DragonFlyAPC
+ (BOOL) loadPrerequisites
{
    return (LOAD_UNIT_DATA_FOR(APC, NO) &&
	    LOAD_UNIT_DATA_FOR(DragonFly, YES));
}

- init
{
    [super init];
    spawnee = [DragonFly class];
    spawnees_remaining = 3;
    return self;
}

- (void) spawnUnit
{
    assert(spawnee != nil);

    [[spawn_unit(spawnee, x, y, ACTIVE_AIR_LIST, YES) setAngle:angle]
	generatePath];

    spawnees_remaining--;
    if (spawnees_remaining > 0) {
	open_tics = 0.75 * APC_INITIAL_OPEN_TICS;
    }
    else {
	spawnee = nil;
    }
}

- (void) draw:(BITMAP *)dest
{
    [super draw:dest];

    /* Fade a DragonFly into existance. */
    if (spawnee) {
	[(<SebImage>)[unit_data getSebumByName:"body"]
		     drawTo:dest X:x-offsetX Y:y-offsetY Alpha:alpha Angle:angle];
    }
}
@end

/*--------------------------------------------------------------*/

@implementation DragonFlyTricky
+ (BOOL) loadPrerequisites
{
    return (LOAD_UNIT_DATA_FOR(APC, NO) &&
	    LOAD_UNIT_DATA_FOR(DragonFly, YES));
}

- init
{
    [super init];
    spawnee = [DragonFly class];
    spawnees_remaining = 3;
    return self;
}

- (void) spawnUnit
{
    assert(spawnee != nil);

    if (not FIND_CLOSEST_ALLY(x,y)) {
	/* Don't spawn if no one to shoot at. */
	open_tics = 1;
	return;
    }

    [[spawn_unit(spawnee, x, y, ACTIVE_AIR_LIST, YES) setAngle:angle]
	generatePath];

    spawnees_remaining--;
    if (spawnees_remaining > 0) {
	open_tics = APC_INITIAL_OPEN_TICS;
    }
    else {
	spawnee = nil;
    }
}

- (void) draw:(BITMAP *)dest
{
    [super draw:dest];

    /* Fade a DragonFly into existance. */
    if (spawnee) {
	[(<SebImage>)[unit_data getSebumByName:"body"]
		     drawTo:dest X:x-offsetX Y:y-offsetY Alpha:alpha Angle:angle];
    }
}
@end

/*------------------------------------------------------------*/

@implementation DragonFlyTrail
- init
{
    [super init];
    health = max_health = 10;
    sprite = [unit_data getSebumByName:"speedy"];
    return self;
}
@end
