/* chopper.m,
 *
 * A helicopter that fires a evil vulcans in bursts.
 */

#include <allegro.h>
#include <math.h>
#include "candy/large-chunk.h"
#include "candy/explosion.h"
#include "common.h"
#include "map.h"
#include "projectiles/evil-vulcan.h"
#include "seborrhea/seborrhea.h"
#include "sound.h"
#include "units/all-units.h"
#include "units/chopper.h"


static SebFile *unit_data;
static Sebum<SebImage> *blade;
static Sebum<SebSample> *fire_snd;


/*--------------------------------------------------------------*/
/* Chopper Abstract Class.					*/
/*--------------------------------------------------------------*/

@implementation BasicChopper
+ (BOOL) loadPrerequisites
{
    return LOAD_PROJECTILE_DATA_FOR(EvilVulcan);
}

+ (BOOL) loadData
{
    if (not [self loadData:&unit_data :"data/chopper"])
	return NO;

    if (bitmap_color_depth(screen) == 32)
	blade = [unit_data getSebumByName:"blade"];
    else
	blade = [unit_data getSebumByName:"bladeb"];

    fire_snd = [unit_data getSebumByName:"snd"];
    return YES;
}

+ (void) shutdown
{
    FREE_SEBFILE(unit_data);
    blade = nil;
    fire_snd = nil;
}

- init
{
    [super init];

    health = 59;
    omega = 0.0;
    omega_blade = deg2rad(13.0);
    
    sprite = [unit_data getSebumByName:"body"];
    shadow = [unit_data getSebumByName:"shadow"];
    w = [sprite width];
    h = [sprite height];
    rotated_sprite = NO;

    gun = [[[PulseWeapon newWithProjectile:[EvilVulcan class]]
	       setShotDelay:3 WaveDelay:16]
	      setShotsPerWave:9];

    /* Pre-activate because our blades are quite big. */
    activate_y = 10;

    flags &=~FLAG_SPAWN_CHUNKS_ON_DEATH;
    chunk_colours = CHUNK_GRAY|CHUNK_RED;
    return self;
}

- (void) draw:(BITMAP *)dest
{
    if (flags & FLAG_DYING) {
	[sprite drawTo:dest X:x-offsetX Y:y-offsetY Angle:angle Scale:scale];
	[blade  drawTo:dest X:x-offsetX Y:y-offsetY Angle:blade_angle Scale:scale];
    }
    else {
	[super draw:dest];
	[blade drawTo:dest X:x-offsetX Y:y-offsetY+4 Angle:blade_angle];
    }

    blade_angle += omega_blade;
    simplify_angle(&blade_angle);
}

- (void) die
{
    xv =  speed*cos(angle)/2.0;
    yv = -speed*sin(angle)/2.0;

    scale = 1.0;
    angle = 0.0;
    rotated_sprite = YES;
    [super die];
}

- (void) spawnDyingExplosions
{
    if (health < -75) {
	[self burstIntoChunks];
        [self delete];
    }
    else if (health < -70) {
	spawn_candy([BigExplosion class], x, y, HIGH_LAYER);
    }
    else if (health % 10 == 0) {
	spawn_candy([Explosion class], x, y, HIGH_LAYER);
    }

    health--;
    flash_tics += 2;
}

- (void) move
{
    if (flags & FLAG_DYING) {
	x += xv;
	y += yv;
	xv *= 0.98;
	yv *= 0.98;
	omega_blade *= 0.989;
	omega = 1.02*omega - deg2rad(0.12);
	angle += omega;
	scale *= 0.995;
    }
    else {
	[super move];
    }
}

- (void) fire
{
    if ([gun fireFromX:x Y:y Angle:-M_PI_2]) {
	if ([(<PulseWeapon>)gun currentShot] == 1)
	    play_panned_sample(fire_snd, x);
    }
}

- derive_fixedAngle;
@end

/*--------------------------------------------------------------*/

@implementation Chopper
+ (BOOL) loadPrerequisites
{
    return LOAD_UNIT_DATA_FOR(BasicChopper, YES);
}

- init
{
    [super init];
    speed = 4.5;
    the_way = (SebSpline *)[unit_data getSebumByName:"path1"];
    return self;
}

- (enum THING_UPDATE_STATE) update
{
    if (path_progress >= 195)
	flags &=~FLAG_FIRING_ENABLED;

    else if (path_progress >= 40)
	flags |= FLAG_FIRING_ENABLED;

    return [super update];
}
@end


@implementation ChopChop
+ (BOOL) loadPrerequisites
{
    return LOAD_UNIT_DATA_FOR(BasicChopper, YES);
}

- init
{
    [super init];
    speed = 3.0;
    the_way = (SebSpline *)[unit_data getSebumByName:"path2"];
    return self;
}

- (enum THING_UPDATE_STATE) update
{
    if (path_progress >= 5)
	flags &=~FLAG_FIRING_ENABLED;

    else if (path_progress >= 2)
	flags |= FLAG_FIRING_ENABLED;

    return [super update];
}
@end


@implementation ChopChopFake
+ (BOOL) loadPrerequisites
{
    return LOAD_UNIT_DATA_FOR(BasicChopper, YES);
}

- init
{
    [super init];
    speed = 3.0;
    the_way = (SebSpline *)[unit_data getSebumByName:"path3"];
    return self;
}

- (enum THING_UPDATE_STATE) update
{
    if (path_progress >= 5)
	flags &=~FLAG_FIRING_ENABLED;

    else if (path_progress >= 2)
	flags |= FLAG_FIRING_ENABLED;

    return [super update];
}
@end
