/* basic-turret.m,
 *
 * A turret that aims at the closest player.
 */

#include <allegro.h>
#include <math.h>
#include "candy.h"
#include "common.h"
#include "map.h"
#include "projectile.h"
#include "rotate.h"
#include "unit-seeking.h"
#include "units/basic-turret.h"


@implementation BasicTurret
	/* Initialization. */
- init
{
    [super init];
    aim_delta_y = 25;
    flags |= FLAG_FIRING_ENABLED;
    return self;
}

- free
{
    if ([turret conformsTo:@protocol(UniqueSebum)])
	turret = [turret free];

    gun = [gun free];
    return [super free];
}

- (void) die
{
    [super die];

    if (base_debris) {
	spawn_candy(base_debris, x, y, FLOOR_LAYER);
	base_debris = nil;
    }

    [self spawnTurretDebris];
}

	/* Drawing */
- (void) draw:(BITMAP *)dest
{
    [super draw:dest];
    [self drawTurret:dest];
}

	/* Update. */
- (void) activate
{
    angle = ANGLE_TO_CLOSEST_ALLY(x, y);
}

- (enum THING_UPDATE_STATE) update
{
    [self aimTurret];
    return [super update];
}

- (void) fire
{
    if (gun)
	[gun fireFromX:x Y:y Angle:angle];
    else
	flags &=~FLAG_FIRING_ENABLED;
}

- (int) collisionLists
{
    return COLLIDES_WITH_PROJECTILES_AND_NUKES;
}

	/* For use in map editor only! */
- (void) drawTo:(BITMAP *)dest at:(int)x_ :(int)y_
{
    x_ += 32;
    y_ += 32;
    [sprite drawTo:dest X:x_-[sprite width]/2 Y:y_-[sprite height]/2];
    [turret drawTo:dest X:x_-[turret width]/2 Y:y_-[turret height]/2];
}
@end


@implementation BasicTurret (Turret)
- (void) drawTurret:(BITMAP *)dest
{
    [self drawTurret:dest Displacement:0 :0];
}

- (void) drawTurret:(BITMAP *)dest Displacement:(int)x_ :(int)y_
{
    [turret drawTo:dest X:x+x_-offsetX Y:y+y_-offsetY Angle:angle];
}

- (void) aimTurret
{
    if (not (game_flags & FLAG_PLAYERS_ALIVE)) {
	flags &=~FLAG_FIRING_ENABLED;
	desired_angle = deg2rad(-90.0);
    }
    else if (not target || not unit_exists(target, ALLY_LIST)) {
	target = FIND_CLOSEST_ALLY(x, y);
	desired_angle = angle_towards_unit(x, y, target);
    }
    else {
	double x_, y_, delta_x, delta_y;

	/* Relocate the target and rotate towards it. */
	[target getX:&x_ Y:&y_];
	delta_x = x_-x;
	delta_y = y-y_+aim_delta_y;
	desired_angle = atan2(delta_y, delta_x);
    }

    /* Rotate turret. */
    limited_rotate(&angle, desired_angle, deg2rad(2.0));
    simplify_angle(&angle);
}

- (void) spawnTurretDebris
{
    if ((candy_amount < 2) ||
	(not turret_debris))
	return;

    /* 20% chance of spawning debris. */
    if (rnd(0, 4) == 0) {
	[spawn_candy(turret_debris, x, y, MEDIUM_LAYER) setAngle:angle];
	turret_debris = nil;
    }
}
@end
