/* basic-tank.m,
 *
 * 'Basic-tank' features common tank things like turrets, firing,
 * animations, etc.  It is an abstract class.
 *
 * Note that 'sprite' is assumed to be a SebAnimator of some sort,
 * most likely a SebAnimatorManualWithDelay.
 */

#include <allegro.h>
#include <math.h>
#include "candy/tank-debris.h"
#include "common.h"
#include "map-editor/unit-editor.h"
#include "map-editor/unit-extras.h"
#include "map.h"
#include "rotate.h"
#include "unit-seeking.h"
#include "units/basic-tank.h"


@implementation BasicTank
	/* Initialization. */
- init
{
    [super init];

    rotated_sprite = YES;
    gun = nil;

    base_debris = [TankDebris class];

    turret_rot_speed = deg2rad(2.0);

    flags |= FLAG_FIRING_ENABLED;
    return self;
}

- free
{
    if ([turret conformsTo:@protocol(UniqueSebum)])
	turret = [turret free];
    gun = [gun free];
    return [super free];
}

- (void) die
{
    [super die];

    if (base_debris) {
	spawn_candy(base_debris, x, y, FLOOR_LAYER);
	base_debris = nil;
    }

    if ((candy_amount >= 2) &&
	turret_debris && rnd(0,9) == 0) {
	[spawn_candy(turret_debris, x, y, MEDIUM_LAYER) setAngle:turret_angle];
	turret_debris = nil;
    }
}

- (id) setAngle:(double)theta
{
    turret_angle = theta;
    return [super setAngle:theta];
}

	/* Drawing */
- (void) draw:(BITMAP *)dest
{
    [super draw:dest];
    [self drawTurret:dest];
}

	/* Update. */
- (enum INACTIVE_UNIT_UPDATE_STATE) readyToActivate
{
    enum INACTIVE_UNIT_UPDATE_STATE r = [super readyToActivate];

    if (r == INACTIVE_UNIT_ACTIVATE)
	turret_angle = ANGLE_TO_CLOSEST_ALLY(x, y);

    return r;
}

- (enum THING_UPDATE_STATE) update
{
    [self aimTurret];
    return [super update];
}

- (void) move
{
    if (travel_range <= 0)
        return;

    /* We want to update the animation in here so that not yet moving units
       will not animate. */  
    [(SebAnimatorManual *)sprite nextFrame];
    travel_range -= speed;

    /* Spawn a track? */
    [self layTracks];

    return [super move];
}

- (void) fire
{
    if (gun)
	[gun fireFromX:x Y:y Angle:turret_angle];
}

- (int) collisionLists
{
    return COLLIDES_WITH_PROJECTILES_AND_NUKES;
}

	/* Load/Save.  Format: travel_range;activate_y;x;y;a[;flags] */
- (void) importUnitProperties:(char *)str
{
    char str2[1024];
    double a;
    int b;

    sscanf(str, "%lf;%d;%s", &a, &b, str2);
    [super importUnitProperties:str2];
    travel_range = a;
    activate_y = b;
}

- (char *) exportUnitProperties:(char *)str
{
    char str2[1024] = "\0";

    [super exportUnitProperties:str2];
    snprintf(str, sizeof(str2), "%g;%d;%s", travel_range, activate_y, str2);
    return str;
}

	/* Protocol stuff. */
- derive_setActivationLine;
- derive_setTravelRange(travel_range);
@end


@implementation BasicTank (Turret)
- (void) drawTurret:(BITMAP *)dest
{
    [self drawTurret:dest Displacement:0 :0];
}

- (void) drawTurret:(BITMAP *)dest Displacement:(int)x_ :(int)y_
{
    [turret drawTo:dest X:x+x_-offsetX Y:y+y_-offsetY Angle:turret_angle];
}

- (void) aimTurret
{
    if (not (game_flags & FLAG_PLAYERS_ALIVE)) {
	desired_turret_angle = angle;
	flags &=~FLAG_FIRING_ENABLED;
    }
    else if (not target || not unit_exists(target, ALLY_LIST)) {
	target = FIND_CLOSEST_ALLY(x, y);
	desired_turret_angle = angle_towards_unit(x, y, target);
    }
    else {
	double x_, y_, delta_x, delta_y;

	/* Relocate the target and rotate towards it. */
	[target getX:&x_ Y:&y_];
	delta_x = x_-x;
	delta_y = y-y_+25;	 /* Aim higher to compensate for scrolling. */
	desired_turret_angle = atan2(delta_y, delta_x);
    }

    /* Rotate turret. */
    limited_rotate(&turret_angle, desired_turret_angle, turret_rot_speed);
    simplify_angle(&turret_angle);
}
@end


@implementation BasicTank (Tracks)
- (void) layTracks
{
    if ((tracks) &&
	SQ(last_track_x-x) + SQ(last_track_y-y) >= track_separation_sq) {
	last_track_x = x;
	last_track_y = y;

	[spawn_candy(tracks, x, y, FLOOR_LAYER) setAngle:angle];
    }
}
@end


@implementation BasicTank (MapEditor)
- (void) drawTo:(BITMAP *)dest at:(int)x_ :(int)y_
{
    x_ += 32;
    y_ += 32;
    [sprite drawTo:dest X:x_-[sprite width]/2 Y:y_-[sprite height]/2];
    [turret drawTo:dest X:x_-[turret width]/2 Y:y_-[turret height]/2];
}

- (void) drawMapEditorExtras:(BITMAP *)dest
{
    [super drawMapEditorExtras:dest];
    draw_activation_line(dest, x, y, activate_y);
    draw_travel_range(dest, x, y, angle, travel_range);
}
@end
