/* apc.m,
 *
 * An apc is similar to a hatch, but they move.  Units are spawned
 * after they have reached their destination.
 */

#include <assert.h>
#include "candy/explosion.h"
#include "candy/tank-debris.h"
#include "candy/tank-track.h"
#include "common.h"
#include "map-editor/unit-editor.h"
#include "map-editor/unit-extras.h"
#include "seborrhea/seborrhea.h"
#include "units/apc.h"
#include "units/all-units.h"


static SebFile *apc_sebum;
static SebAnimation *move_anim, *open_anim;


/*--------------------------------------------------------------*/

@interface BasicAPC (Private)
- (void) maybeSpawnUnit;
@end

@interface APCTrack: TankTrack
@end

/*--------------------------------------------------------------*/

@implementation BasicAPC
+ (BOOL) loadData
{
    if (not [self loadData:&apc_sebum :"data/apc"])
	return NO;

    move_anim = (SebAnimation *)[apc_sebum getSebumByName:"move-anim"];
    open_anim = (SebAnimation *)[apc_sebum getSebumByName:"open-anim"];
    return YES;
}

+ (void) shutdown
{
    FREE_SEBFILE(apc_sebum);
    move_anim = nil;
    open_anim = nil;
}

- init
{
    [super init];

    speed = 0.5;

    w = [sprite width];
    h = [sprite height];
    rotated_sprite = YES;

    open_tics = APC_INITIAL_OPEN_TICS;
    spawnee = nil;

    sprite = [[SebAnimator new] setAnimation:move_anim];
    last_track_x = last_track_y = 0;
    chunk_colours = CHUNK_BLUE;

    return self;
}

- (void) move
{
    double delta_x, delta_y;

    if (travel_range >= 0.0) {
	travel_range -= speed;
	[super move];

	/* Spawn tracks? */
	delta_x = x - last_track_x;
	delta_y = y - last_track_y;
	if (delta_x*delta_x + delta_y*delta_y >= 17.0*17.0) {
	    last_track_x = x;
	    last_track_y = y;

	    [spawn_candy([APCTrack class], x, y, FLOOR_LAYER) setAngle:angle];
	}
    }
}

- (void) spawnUnit
{
    assert(spawnee != nil);
    [spawn_unit(spawnee, x, y, ACTIVE_AIR_LIST, NO) setAngle:angle];
    spawnee = nil;
}

- (int) collisionLists
{
    return COLLIDES_WITH_PROJECTILES_AND_NUKES;
}


	/* Load/Save.  Format: travel_range;activate_y;x;y;a[;flags] */
- (void) importUnitProperties:(char *)str
{
    char str2[1024];
    double a;
    int b;

    sscanf(str, "%lf;%d;%s", &a, &b, str2);
    [super importUnitProperties:str2];
    travel_range = a;
    activate_y = b;
}

- (char *) exportUnitProperties:(char *)str
{
    char str2[1024] = "\0";

    [super exportUnitProperties:str2];
    snprintf(str, sizeof(str2), "%g;%d;%s", travel_range, activate_y, str2);
    return str;
}

	/* Protocol stuff. */
- derive_setActivationLine;
- derive_setTravelRange(travel_range);
@end


@implementation BasicAPC (Private)
- (void) maybeSpawnUnit
{
    if (not spawnee)
	return;

    open_tics--;
    alpha = 255 - 192 * open_tics/APC_INITIAL_OPEN_TICS;

    if (open_tics <= 0)
	[self spawnUnit];
}
@end


@implementation BasicAPC (MapEditor)
- (void) drawMapEditorExtras:(BITMAP *)dest
{
    [super drawMapEditorExtras:dest];
    draw_activation_line(dest, x, y, activate_y);
    draw_travel_range(dest, x, y, angle, travel_range);
}
@end

/*--------------------------------------------------------------*/

@implementation APC
+ (BOOL) loadPrerequisites
{
    return LOAD_UNIT_DATA_FOR(BasicAPC, NO);
}

- init
{
    [super init];
    health = 200;
    critical_health = 100;
    return self;
}

- (void) die
{
    [super die];
    spawn_candy([TankDebris class], x, y, FLOOR_LAYER);
}

- (void) move
{
    if (travel_range <= 0.0) {
	[self maybeSpawnUnit];
    }
    else {
	[super move];

	if ((travel_range <= 0.0) && (spawnee)) {
	    if (game_flags & FLAG_PLAYERS_ALIVE) {
		[(SebAnimator *)sprite setAnimation:open_anim];
	    }
	    else {
		/* Don't open/spawn units if no players. */
		spawnee = nil;
	    }
	}
    }
}
@end

/*--------------------------------------------------------------*/

@implementation APCTricky
+ (BOOL) loadPrerequisites
{
    return LOAD_UNIT_DATA_FOR(BasicAPC, NO);
}

- init
{
    [super init];
    health = 150;		/* Lower health than APC. */
    critical_health = 50;
    return self;
}

- (void) die
{
    [super die];
    critical_health = 0;

    flags &=~FLAG_DYING;
    travel_range = 0.0;

    spawn_candy([BigExplosion class], x, y, LOW_LAYER);

    [sprite free];		/* SebAnimator. */
    sprite = [apc_sebum getSebumByName:"tricky"];
}

- (enum THING_UPDATE_STATE) update
{
    /* Died. */
    if (flags & FLAG_INVINCIBLE) {
	[self maybeSpawnUnit];
	return ([self mayStillInfluenceGameplay] ? THING_NORMAL : THING_DEAD);
    }

    return [super update];
}

- (int) collisionLists
{
    if (flags & FLAG_INVINCIBLE)
	return 0;
    else
	return COLLIDES_WITH_PROJECTILES_AND_NUKES;
}
@end

/*--------------------------------------------------------------*/

@implementation APCTrack
- init
{
    [super init];
    health = max_track_life = 150;
    sprite = [apc_sebum getSebumByName:"tracks"];
    return self;
}
@end
