/* all-units.m,
 *
 * This contains a list of the units for use in the map editor and
 * when loading/saving maps.
 */

#include <allegro.h>
#include <assert.h>
#include "common.h"
#include "debug.inc"
#include "units/all-units.h"
#include "units/all-units.inc"


unit_table_t air_units_table, ground_units_table;

/*--------------------------------------------------------------*/
/* Datafiles.							*/
/*--------------------------------------------------------------*/

BOOL load_unit_data_for(Class class, const BOOL air)
{
    unsigned int i;
    unit_table_t *table;
    unit_table_helper_t *class_data = NULL;

    if (air)
	table = &air_units_table;
    else
	table = &ground_units_table;

    for (i = 0; i < table->num_classes; i++) {
	if (class == table->class_data[i].class) {
	    class_data = &table->class_data[i];
	    break;
	}
    }

    assert(class_data != NULL);
    class_data->required = YES;

    if (not [class loadPrerequisites])
	return NO;

    if (not class_data->loads_data)
	return YES;

    if (not class_data->loaded) {
	class_data->loaded = [class loadData];

	if (not class_data->loaded)
	    fprintf(stderr, "Error loading %s datafiles.\n", [class name]);
    }

    return class_data->loaded;
}


static void unload_unnecessary_unit_data_in_table(unit_table_t* const table)
{
    unsigned int i;

    for (i = 0; i < table->num_classes; i++) {
	unit_table_helper_t* const class_data = &table->class_data[i];

	if (class_data->loaded && not class_data->required) {
#ifdef DEBUG_LOG_SHUTDOWN_UNITS
	    printf("Shutting down %s.\n", [class_data->class name]);
#endif

	    [class_data->class shutdown];
	    class_data->loaded = NO;
	}
    }
}


void unload_unnecessary_unit_data(void)
{
    unload_unnecessary_unit_data_in_table(&air_units_table);
    unload_unnecessary_unit_data_in_table(&ground_units_table);
}


static void
mark_all_unit_data_in_table_required(unit_table_t* const table, BOOL req)
{
    unsigned int i;

    for (i = 0; i < table->num_classes; i++)
	table->class_data[i].required = req;
}


void mark_all_unit_data_necessary(void)
{
    mark_all_unit_data_in_table_required(&air_units_table, YES);
    mark_all_unit_data_in_table_required(&ground_units_table, YES);
}


void mark_all_unit_data_unnecessary(void)
{
    mark_all_unit_data_in_table_required(&air_units_table, NO);
    mark_all_unit_data_in_table_required(&ground_units_table, NO);
}

/*--------------------------------------------------------------*/

static Class
find_class_by_name_in_table(const char *name, const unit_table_t* const table)
{
    unsigned int i;

    for (i = 0; i < table->num_classes; i++) {
	Class class = table->class_data[i].class;

	if (streq(name, [class name]))
	    return class;
    }

    return nil;
}


Class find_class_by_name(const char *name, const BOOL air)
{
    if (air)
	return find_class_by_name_in_table(name, &air_units_table);
    else
	return find_class_by_name_in_table(name, &ground_units_table);
}

/*--------------------------------------------------------------*/

static void add_air_units(void);
static void add_boss_units(void);
static void add_ground_units(void);


static void add_unit(const Class class, unit_table_t* const table,
		     const BOOL show_in_map_editor, const BOOL loads_data)
{
    unit_table_helper_t *class_data;
    assert(class != NULL);
    assert(table != NULL);

    /* Reallocate if table is full. */
    if (table->max_classes <= table->num_classes) {
	unsigned int new_max;
	void *p;

	new_max = table->max_classes + 1;
	p = realloc(table->class_data, sizeof(*table->class_data) * new_max);
	assert(p != NULL);

	table->max_classes = new_max;
	table->class_data = p;
    }

    assert(table->num_classes < table->max_classes);
    class_data = &table->class_data[table->num_classes];
    class_data->class = class;
    class_data->loads_data = loads_data;
    class_data->loaded = NO;
    class_data->required = NO;
    class_data->show_in_map_editor = show_in_map_editor;
    table->num_classes++;
}


void make_all_units_tables(void)
{
    air_units_table.max_classes = 0;
    air_units_table.num_classes = 0;
    air_units_table.class_data = NULL;
    ground_units_table.max_classes = 0;
    ground_units_table.num_classes = 0;
    ground_units_table.class_data = NULL;

    /* We don't want units to register themselves so we can keep the
       ordering for REdit. */
    add_air_units();
    add_ground_units();
    add_boss_units();
}


void shutdown_special_units(void)
{
    /* XXX?  Special stuff. */
    [Player shutdown];
    [Robot shutdown];
    [Shield shutdown];
}

/*--------------------------------------------------------------*/

#define AIR_UNIT(m,d,u)	add_unit([u class], &air_units_table, m, d)
#define GND_UNIT(m,d,u)	add_unit([u class], &ground_units_table, m, d)

static void add_air_units(void)
{
    /* Big Burgers. */
    AIR_UNIT(YES, YES, BabyBoy);
    AIR_UNIT(YES, YES, Challenger);

    /* Brutus's Legion. */
    AIR_UNIT(YES, YES, Deceiver);
    AIR_UNIT(YES, YES, Schemer);

    /* Cargo. */
    AIR_UNIT(YES, YES, Cargo);
    AIR_UNIT(YES, NO,  CargoHealth);

    /* Chopper. */
    AIR_UNIT(NO,  YES, BasicChopper);
    AIR_UNIT(YES, NO,  Chopper);
    AIR_UNIT(YES, NO,  ChopChop);
    AIR_UNIT(YES, NO,  ChopChopFake);

    /* Fire Brand. */
    AIR_UNIT(YES, YES, FireFlea);
    AIR_UNIT(YES, YES, FireFly);
    AIR_UNIT(YES, YES, FireFox);

    /* Insecticons. */
    AIR_UNIT(YES, YES, Mozzie);
    AIR_UNIT(YES, YES, DragonFly);
    GND_UNIT(YES, NO,  MozzieHatch);
    GND_UNIT(YES, NO,  DragonFlyAPC);
    GND_UNIT(YES, NO,  DragonFlyTricky);

    /* IX. */
    AIR_UNIT(YES, YES, LittleDeath);
    AIR_UNIT(YES, YES, Fear);

    /* Laserjets. */
    AIR_UNIT(YES, YES, Battler);
    AIR_UNIT(YES, YES, Conquerer);
    AIR_UNIT(YES, YES, BattleCruiser);

    /* Little Spacies. */
    AIR_UNIT(NO,  YES, BasicLint);
    AIR_UNIT(YES, NO,  Lint);
    AIR_UNIT(YES, NO,  LintMirror);
    AIR_UNIT(YES, NO,  LintReverse);
    AIR_UNIT(NO,  YES, BasicMote);
    AIR_UNIT(YES, NO,  MoteTop);
    AIR_UNIT(YES, NO,  MoteUp);
    AIR_UNIT(YES, NO,  MoteDown);
    AIR_UNIT(YES, NO,  MoteStrange);
    AIR_UNIT(YES, NO,  MoteCharm);
    AIR_UNIT(YES, YES, Grunt);
    AIR_UNIT(YES, YES, Moan);

    /* Nostalgic Collection. */
    AIR_UNIT(NO,  YES, BasicKamikaze);
    AIR_UNIT(YES, NO,  Kamikaze);
    AIR_UNIT(YES, NO,  KamikazeCenter);
    AIR_UNIT(YES, NO,  KamikazeCharger);
    AIR_UNIT(YES, NO,  KamikazeHook);
    AIR_UNIT(YES, YES, Bug);
    AIR_UNIT(YES, YES, Cruiser);

    /* Starcraft(s). */
    AIR_UNIT(YES, YES, Scourge);
    AIR_UNIT(YES, NO,  ScourgePath);
    AIR_UNIT(YES, NO,  ScourgeMirror);
    AIR_UNIT(YES, YES, Interceptor);
    AIR_UNIT(YES, YES, Wraith);

    /* Stumpies. */
    AIR_UNIT(YES, YES, Stumpy);
    AIR_UNIT(YES, YES, Stubby);
    AIR_UNIT(YES, YES, MsHoover);

    /* Trippy. */
    AIR_UNIT(YES, YES, Centipede);
    AIR_UNIT(YES, YES, Strumble);
    AIR_UNIT(YES, YES, Zippy);
    AIR_UNIT(YES, YES, Spirtle);
    AIR_UNIT(YES, NO,  SpirtleMirror);

    AIR_UNIT(YES, YES, MineLayer);
    AIR_UNIT(YES, YES, Plump);

    /* Destroyable Projectiles. */
    AIR_UNIT(NO,  YES, HunterSeeker);
    AIR_UNIT(NO,  YES, ClusterBomb);
    AIR_UNIT(NO,  YES, Mine);

    AIR_UNIT(YES, YES, Trigger);
}


static void add_boss_units(void)
{
    GND_UNIT(YES, YES, MiniBoss0);
    AIR_UNIT(YES, YES, MiniBoss1);
    GND_UNIT(NO,  YES, AbstractMini2Turret);
    GND_UNIT(YES, NO,  Basturk);
    GND_UNIT(YES, NO,  Mini2Turret);
    GND_UNIT(YES, NO,  Mini2Cannon);

    AIR_UNIT(YES, YES, Boss0);
    AIR_UNIT(YES, YES, Boss1);
    GND_UNIT(YES, YES, Khan);
}

static void add_ground_units(void)
{
    /* Baleog's Turrets. */
    GND_UNIT(YES, YES, Stone);
    GND_UNIT(YES, YES, Viking);
    GND_UNIT(YES, YES, Anvil);

    /* Captain LeChimp. */
    GND_UNIT(YES, YES, SeaCucumber);
    GND_UNIT(NO,  YES, BasicSeaDog);
    GND_UNIT(YES, NO,  SeaDogSweeper);
    GND_UNIT(YES, YES, SeaUrchin);

    /* Lizards. */
    GND_UNIT(YES, YES, Newt);
    GND_UNIT(YES, YES, Gecko);
    GND_UNIT(YES, YES, Lizard);

    /* L'odeur Animale. */
    GND_UNIT(YES, YES, Armadillo);
    GND_UNIT(YES, NO,  ArmadilloHatch);
    GND_UNIT(YES, YES, Hippo);
    GND_UNIT(YES, NO,  HippoHatch);
    GND_UNIT(YES, YES, Elephant);
    GND_UNIT(YES, NO,  ElephantReverse);

    /* Space Turrets. */
    GND_UNIT(YES, YES, Tripe);
    GND_UNIT(YES, YES, Hunter);

    /* Shiny Magic. */
    GND_UNIT(YES, YES, Freeze);
    GND_UNIT(YES, NO,  FreezeSW);
    GND_UNIT(YES, YES, Bolt);

    /* Snow-mobiles. */
    GND_UNIT(NO,  YES, BasicAPC);
    GND_UNIT(YES, NO,  APC);
    GND_UNIT(YES, YES, Turtle);
    GND_UNIT(YES, NO,  TurtlePath);
    GND_UNIT(YES, YES, RedBackSpider);
    GND_UNIT(YES, YES, PolarBear);

    /* Boats. */
    GND_UNIT(YES, YES, Scurvy);

    /* Scenery. */
    GND_UNIT(YES, YES, Hatch);
}

#undef GND_UNIT
#undef AIR_UNIT
